// CoWatch Background Service Worker
// Cross-browser compatible

// Production URL
const COWATCH_URL = 'https://cowatch.app';

// Import polyfill for module context (Chrome MV3)
try {
  importScripts('browser-polyfill.js');
} catch (e) {
  // Module import not available in all contexts
}

// Use unified API if available
const browserAPI = typeof CoWatchBrowser !== 'undefined' ? CoWatchBrowser : null;
const chromeAPI = typeof chrome !== 'undefined' ? chrome : null;

// Helper to get the right API
function getAPI() {
  return browserAPI || { 
    action: chromeAPI?.action,
    tabs: chromeAPI?.tabs,
    runtime: chromeAPI?.runtime,
    storage: chromeAPI?.storage
  };
}

// Handle extension icon click
if (chromeAPI?.action?.onClicked) {
  chromeAPI.action.onClicked.addListener((tab) => {
    // This is handled by the popup, but we can add additional logic here
  });
}

// Handle installation
if (chromeAPI?.runtime?.onInstalled) {
  chromeAPI.runtime.onInstalled.addListener((details) => {
    if (details.reason === 'install') {
      // Set initial storage values
      chromeAPI.storage.local.set({
        language: 'en',
        installed: true,
        installDate: Date.now()
      });
      console.log('CoWatch extension installed successfully!');
      
      // Set uninstall URL for feedback
      if (chromeAPI?.runtime?.setUninstallURL) {
        chromeAPI.runtime.setUninstallURL('https://cowatch.app/uninstall');
      }
    } else if (details.reason === 'update') {
      // Ensure uninstall URL is set on update too
      if (chromeAPI?.runtime?.setUninstallURL) {
        chromeAPI.runtime.setUninstallURL('https://cowatch.app/uninstall');
      }
    }
  });
}

// Set uninstall URL on service worker startup (in case it wasn't set)
if (chromeAPI?.runtime?.setUninstallURL) {
  chromeAPI.runtime.setUninstallURL('https://cowatch.app/uninstall');
}

// Listen for tab removal to clear room state when video tab is closed
if (chromeAPI?.tabs?.onRemoved) {
  chromeAPI.tabs.onRemoved.addListener((tabId) => {
    chromeAPI.storage.local.get(['activeRoomTabId', 'roomCode'], (result) => {
      if (result.activeRoomTabId === tabId && result.roomCode) {
        console.log('🎬 Background: Active room tab closed, clearing room state');
        chromeAPI.storage.local.remove(['roomCode', 'isHost', 'activeRoomTabId', 'messages']);
      }
    });
  });
}

// Listen for tab updates to inject content script if needed
if (chromeAPI?.tabs?.onUpdated) {
  chromeAPI.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url) {
      const supportedSites = [
        'netflix.com',
        'youtube.com',
        'disneyplus.com',
        'primevideo.com',
        'hbomax.com',
        'max.com',
        'twitch.tv',
        'vimeo.com',
        'dailymotion.com',
        'crunchyroll.com'
      ];
      
      const isSupported = supportedSites.some(site => tab.url.includes(site));
      
      if (isSupported && chromeAPI?.action) {
        chromeAPI.action.setBadgeText({ tabId, text: '✓' });
        chromeAPI.action.setBadgeBackgroundColor({ tabId, color: '#4ade80' });
      } else if (chromeAPI?.action) {
        chromeAPI.action.setBadgeText({ tabId, text: '' });
      }
      
      // Check if this is the auth tab and user just logged in
      if (tab.url.includes('cowatch.app')) {
        chromeAPI.storage.local.get(['pendingAuthTabId'], async (result) => {
          if (result.pendingAuthTabId === tabId) {
            // Check if user is now logged in
            try {
              const response = await fetch(`${COWATCH_URL}/api/auth/session`, {
                credentials: 'include'
              });
              const data = await response.json();
              
              if (data.user) {
                console.log('🎬 Background: User logged in, saving session and closing tab');
                // Save user to storage
                chromeAPI.storage.local.set({ 
                  user: data.user, 
                  guestMode: false,
                  pendingAuthTabId: null 
                });
                
                // Close the login tab if user successfully logged in
                // Close on success page, verified page, set-password, callback, or dashboard
                if (tab.url.includes('/auth/success') ||
                    tab.url.includes('/auth/verified') || 
                    tab.url.includes('/auth/set-password') ||
                    tab.url.includes('/api/auth/callback/') ||
                    tab.url === `${COWATCH_URL}/` ||
                    tab.url.includes('?logged_in=true') ||
                    (tab.url.includes('cowatch.app') && !tab.url.includes('/auth/login') && !tab.url.includes('/auth/forgot'))) {
                  setTimeout(() => {
                    chromeAPI.tabs.remove(tabId).catch(() => {});
                  }, 1500);
                }
              }
            } catch (error) {
              console.error('🎬 Background: Failed to check session:', error);
            }
          }
        });
      }
    }
  });
}

// Handle messages from content script
if (chromeAPI?.runtime?.onMessage) {
  chromeAPI.runtime.onMessage.addListener((message, sender, sendResponse) => {
    switch (message.type) {
      case 'CLOSE_AUTH_TAB':
        // Close the tab that sent this message (auth tab)
        if (sender.tab?.id) {
          console.log('🎬 Background: Closing auth tab', sender.tab.id);
          setTimeout(() => {
            chromeAPI.tabs.remove(sender.tab.id).catch(() => {});
          }, 500);
        }
        sendResponse({ success: true });
        break;
      
      case 'ROOM_ACTIVE':
        // Track which tab has an active room
        chromeAPI.storage.local.set({ activeRoomTabId: sender.tab?.id });
        sendResponse({ success: true });
        break;
        
      case 'OPEN_COWATCH':
        chromeAPI.tabs.create({
          url: `${COWATCH_URL}${message.path || '/'}`
        });
        break;
        
      case 'GET_AUTH_TOKEN':
        // Get auth token from storage
        chromeAPI.storage.local.get(['authToken'], (result) => {
          sendResponse({ token: result.authToken });
        });
        return true; // Keep message channel open for async response
        
      case 'SET_AUTH_TOKEN':
        chromeAPI.storage.local.set({ authToken: message.token });
        sendResponse({ success: true });
        break;
        
      case 'GET_BROWSER_INFO':
        sendResponse({
          browser: browserAPI?.features?.browser?.name || 'Chrome',
          manifest: chromeAPI.runtime.getManifest().manifest_version
        });
        break;
        
      case 'INJECT_CONTENT_SCRIPT':
        // Programmatically inject content script into tab
const tabId = message.tabId;
        console.log('🎬 Background: Injecting content script into tab', tabId);
        
        chromeAPI.scripting.executeScript({
          target: { tabId: tabId },
          files: ['browser-polyfill.js', 'content.js']
        }).then(() => {
          console.log('🎬 Background: Content script injected successfully');
          // Also inject CSS
          return chromeAPI.scripting.insertCSS({
            target: { tabId: tabId },
            files: ['content.css']
          });
        }).then(() => {
          sendResponse({ success: true });
        }).catch((error) => {
          console.log('🎬 Background: Injection error (may already be loaded):', error.message);
          sendResponse({ success: false, error: error.message });
        });
        return true; // Keep message channel open for async response
      
      case 'REGISTER_PUSH_NOTIFICATIONS':
        // Register for push notifications
        console.log('🎬 Background: Registering push notifications');
        registerPushNotifications()
          .then((subscription) => {
            sendResponse({ success: true, subscription });
          })
          .catch((error) => {
            console.error('🎬 Background: Push registration failed:', error);
            sendResponse({ success: false, error: error.message });
          });
        return true;
      
      case 'SHOW_NOTIFICATION':
        // Show a local notification
        showNotification(message.title, message.options);
        sendResponse({ success: true });
        break;
    }
  });
}

// Push notification registration
async function registerPushNotifications() {
  try {
    // Register service worker for Firebase messaging
    const registration = await chromeAPI.runtime.getRegistration?.() || 
                         await navigator.serviceWorker.register('firebase-messaging-sw.js');
    
    console.log('🎬 Background: Service worker registered for push');
    
    // Get push subscription
    const subscription = await registration.pushManager.subscribe({
      userVisibleOnly: true,
      applicationServerKey: urlBase64ToUint8Array(
        'YOUR_VAPID_PUBLIC_KEY' // Replace with actual VAPID key
      )
    });
    
    console.log('🎬 Background: Push subscription created');
    
    // Save subscription to storage
    await chromeAPI.storage.local.set({ pushSubscription: JSON.stringify(subscription) });
    
    // Send to server
    try {
      const token = await chromeAPI.storage.local.get(['authToken']);
      if (token.authToken) {
        await fetch(`${COWATCH_URL}/api/notifications/subscribe`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token.authToken}`
          },
          body: JSON.stringify({
            subscription: subscription,
            userAgent: navigator.userAgent
          })
        });
      }
    } catch (e) {
      console.log('🎬 Background: Failed to send subscription to server:', e);
    }
    
    return subscription;
  } catch (error) {
    console.error('🎬 Background: Push registration error:', error);
    throw error;
  }
}

// Helper function to convert VAPID key
function urlBase64ToUint8Array(base64String) {
  const padding = '='.repeat((4 - base64String.length % 4) % 4);
  const base64 = (base64String + padding)
    .replace(/\-/g, '+')
    .replace(/_/g, '/');
  
  const rawData = atob(base64);
  const outputArray = new Uint8Array(rawData.length);
  
  for (let i = 0; i < rawData.length; ++i) {
    outputArray[i] = rawData.charCodeAt(i);
  }
  return outputArray;
}

// Show local notification
function showNotification(title, options = {}) {
  chromeAPI.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon128.png',
    title: title,
    message: options.body || '',
    priority: 2,
    requireInteraction: options.requireInteraction || false,
    ...options
  });
}

// Handle notification clicks
if (chromeAPI.notifications?.onClicked) {
  chromeAPI.notifications.onClicked.addListener((notificationId) => {
    // Get notification data from storage
    chromeAPI.storage.local.get(['lastNotification'], (result) => {
      const data = result.lastNotification || {};
      
      if (data.roomCode) {
        // Open join page
        chromeAPI.tabs.create({ url: `${COWATCH_URL}/join/${data.roomCode}` });
      } else if (data.url) {
        chromeAPI.tabs.create({ url: data.url });
      }
      
      chromeAPI.notifications.clear(notificationId);
    });
  });
}

