/**
 * CoWatch Browser Compatibility Layer
 * Provides a unified API for Chrome, Firefox, Edge, Safari, and Opera
 */

(function(global) {
  'use strict';

  // Detect browser environment
  const isChrome = typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.id;
  const isFirefox = typeof browser !== 'undefined' && browser.runtime && browser.runtime.id;
  const isEdge = isChrome && navigator.userAgent.includes('Edg');
  const isSafari = typeof safari !== 'undefined' || navigator.userAgent.includes('Safari') && !navigator.userAgent.includes('Chrome');
  const isOpera = navigator.userAgent.includes('OPR');

  // Create unified browser API
  const browserAPI = (function() {
    // Use browser (Firefox) or chrome API
    if (typeof browser !== 'undefined' && browser.runtime) {
      return browser;
    }
    if (typeof chrome !== 'undefined' && chrome.runtime) {
      return chrome;
    }
    // Fallback for Safari
    if (typeof safari !== 'undefined') {
      return safari.extension;
    }
    return null;
  })();

  // Promise wrapper for callback-based Chrome APIs
  function promisify(fn, context) {
    return function(...args) {
      return new Promise((resolve, reject) => {
        fn.call(context, ...args, (result) => {
          if (browserAPI.runtime && browserAPI.runtime.lastError) {
            reject(new Error(browserAPI.runtime.lastError.message));
          } else {
            resolve(result);
          }
        });
      });
    };
  }

  // Unified Storage API
  const storage = {
    local: {
      get: function(keys) {
        return new Promise((resolve, reject) => {
          try {
            if (isFirefox) {
              browser.storage.local.get(keys).then(resolve).catch(reject);
            } else if (browserAPI && browserAPI.storage) {
              browserAPI.storage.local.get(keys, (result) => {
                if (browserAPI.runtime.lastError) {
                  reject(new Error(browserAPI.runtime.lastError.message));
                } else {
                  resolve(result);
                }
              });
            } else {
              // Fallback to localStorage
              const result = {};
              const keyList = Array.isArray(keys) ? keys : (typeof keys === 'string' ? [keys] : Object.keys(keys));
              keyList.forEach(key => {
                const value = localStorage.getItem(`cowatch_${key}`);
                if (value !== null) {
                  try {
                    result[key] = JSON.parse(value);
                  } catch {
                    result[key] = value;
                  }
                }
              });
              resolve(result);
            }
          } catch (error) {
            reject(error);
          }
        });
      },
      set: function(items) {
        return new Promise((resolve, reject) => {
          try {
            if (isFirefox) {
              browser.storage.local.set(items).then(resolve).catch(reject);
            } else if (browserAPI && browserAPI.storage) {
              browserAPI.storage.local.set(items, () => {
                if (browserAPI.runtime.lastError) {
                  reject(new Error(browserAPI.runtime.lastError.message));
                } else {
                  resolve();
                }
              });
            } else {
              // Fallback to localStorage
              Object.entries(items).forEach(([key, value]) => {
                localStorage.setItem(`cowatch_${key}`, JSON.stringify(value));
              });
              resolve();
            }
          } catch (error) {
            reject(error);
          }
        });
      },
      remove: function(keys) {
        return new Promise((resolve, reject) => {
          try {
            if (isFirefox) {
              browser.storage.local.remove(keys).then(resolve).catch(reject);
            } else if (browserAPI && browserAPI.storage) {
              browserAPI.storage.local.remove(keys, () => {
                if (browserAPI.runtime.lastError) {
                  reject(new Error(browserAPI.runtime.lastError.message));
                } else {
                  resolve();
                }
              });
            } else {
              // Fallback to localStorage
              const keyList = Array.isArray(keys) ? keys : [keys];
              keyList.forEach(key => {
                localStorage.removeItem(`cowatch_${key}`);
              });
              resolve();
            }
          } catch (error) {
            reject(error);
          }
        });
      }
    }
  };

  // Unified Tabs API
  const tabs = {
    query: function(queryInfo) {
      return new Promise((resolve, reject) => {
        try {
          if (isFirefox) {
            browser.tabs.query(queryInfo).then(resolve).catch(reject);
          } else if (browserAPI && browserAPI.tabs) {
            browserAPI.tabs.query(queryInfo, (result) => {
              if (browserAPI.runtime.lastError) {
                reject(new Error(browserAPI.runtime.lastError.message));
              } else {
                resolve(result);
              }
            });
          } else {
            resolve([]);
          }
        } catch (error) {
          reject(error);
        }
      });
    },
    create: function(createProperties) {
      return new Promise((resolve, reject) => {
        try {
          if (isFirefox) {
            browser.tabs.create(createProperties).then(resolve).catch(reject);
          } else if (browserAPI && browserAPI.tabs) {
            browserAPI.tabs.create(createProperties, (tab) => {
              if (browserAPI.runtime.lastError) {
                reject(new Error(browserAPI.runtime.lastError.message));
              } else {
                resolve(tab);
              }
            });
          } else {
            window.open(createProperties.url, '_blank');
            resolve({});
          }
        } catch (error) {
          reject(error);
        }
      });
    },
    sendMessage: function(tabId, message) {
      return new Promise((resolve, reject) => {
        try {
          if (isFirefox) {
            browser.tabs.sendMessage(tabId, message).then(resolve).catch(reject);
          } else if (browserAPI && browserAPI.tabs) {
            browserAPI.tabs.sendMessage(tabId, message, (response) => {
              if (browserAPI.runtime.lastError) {
                // Don't reject, just resolve with undefined
                resolve(undefined);
              } else {
                resolve(response);
              }
            });
          } else {
            resolve(undefined);
          }
        } catch (error) {
          reject(error);
        }
      });
    }
  };

  // Unified Runtime API
  const runtime = {
    sendMessage: function(message) {
      return new Promise((resolve, reject) => {
        try {
          if (isFirefox) {
            browser.runtime.sendMessage(message).then(resolve).catch(reject);
          } else if (browserAPI && browserAPI.runtime) {
            browserAPI.runtime.sendMessage(message, (response) => {
              if (browserAPI.runtime.lastError) {
                reject(new Error(browserAPI.runtime.lastError.message));
              } else {
                resolve(response);
              }
            });
          } else {
            resolve(undefined);
          }
        } catch (error) {
          reject(error);
        }
      });
    },
    onMessage: {
      addListener: function(callback) {
        if (isFirefox && browser.runtime) {
          browser.runtime.onMessage.addListener((message, sender) => {
            return new Promise((resolve) => {
              callback(message, sender, resolve);
            });
          });
        } else if (browserAPI && browserAPI.runtime) {
          browserAPI.runtime.onMessage.addListener(callback);
        }
      }
    },
    onInstalled: {
      addListener: function(callback) {
        if (browserAPI && browserAPI.runtime && browserAPI.runtime.onInstalled) {
          browserAPI.runtime.onInstalled.addListener(callback);
        }
      }
    },
    getURL: function(path) {
      if (browserAPI && browserAPI.runtime && browserAPI.runtime.getURL) {
        return browserAPI.runtime.getURL(path);
      }
      return path;
    }
  };

  // Unified Action/BrowserAction API (MV3 vs MV2)
  const action = (function() {
    const actionAPI = browserAPI && (browserAPI.action || browserAPI.browserAction);
    
    return {
      setBadgeText: function(details) {
        if (actionAPI && actionAPI.setBadgeText) {
          actionAPI.setBadgeText(details);
        }
      },
      setBadgeBackgroundColor: function(details) {
        if (actionAPI && actionAPI.setBadgeBackgroundColor) {
          actionAPI.setBadgeBackgroundColor(details);
        }
      },
      onClicked: {
        addListener: function(callback) {
          if (actionAPI && actionAPI.onClicked) {
            actionAPI.onClicked.addListener(callback);
          }
        }
      }
    };
  })();

  // Feature detection utilities
  const features = {
    hasServiceWorker: typeof ServiceWorker !== 'undefined',
    hasBackgroundScripts: !!(browserAPI && browserAPI.runtime),
    hasContentScripts: !!(browserAPI && browserAPI.runtime),
    hasStorage: !!(browserAPI && browserAPI.storage) || typeof localStorage !== 'undefined',
    hasWebSocket: typeof WebSocket !== 'undefined',
    hasClipboard: !!(navigator.clipboard && navigator.clipboard.writeText),
    
    // Browser detection
    browser: {
      isChrome,
      isFirefox,
      isEdge,
      isSafari,
      isOpera,
      name: isFirefox ? 'Firefox' : (isEdge ? 'Edge' : (isSafari ? 'Safari' : (isOpera ? 'Opera' : 'Chrome')))
    }
  };

  // Clipboard API with fallback
  const clipboard = {
    writeText: async function(text) {
      if (navigator.clipboard && navigator.clipboard.writeText) {
        try {
          await navigator.clipboard.writeText(text);
          return true;
        } catch (e) {
          // Fall through to fallback
        }
      }
      
      // Fallback for older browsers
      const textArea = document.createElement('textarea');
      textArea.value = text;
      textArea.style.position = 'fixed';
      textArea.style.left = '-9999px';
      textArea.style.top = '-9999px';
      document.body.appendChild(textArea);
      textArea.focus();
      textArea.select();
      
      try {
        document.execCommand('copy');
        document.body.removeChild(textArea);
        return true;
      } catch (e) {
        document.body.removeChild(textArea);
        return false;
      }
    }
  };

  // Export unified API
  global.CoWatchBrowser = {
    storage,
    tabs,
    runtime,
    action,
    features,
    clipboard,
    
    // Original API access
    native: browserAPI
  };

})(typeof window !== 'undefined' ? window : (typeof self !== 'undefined' ? self : this));
