// CoWatch Content Script - Teleparty-like Watch Party Extension
// Provides synchronized video watching with friends
// Version 1.6.0

(function() {
  // Prevent multiple initializations
  if (window.coWatchInitialized) {
    console.log('🎬 CoWatch: Already initialized, skipping...');
    return;
  }
  window.coWatchInitialized = true;
  
  console.log('🎬 CoWatch Extension v1.7.0 - Content script loaded!');
  console.log('🎬 Current URL:', window.location.href);
  console.log('🎬 Document ready state:', document.readyState);

  // Add marker element for extension detection
  const extensionMarker = document.createElement('div');
  extensionMarker.id = 'cowatch-extension-installed';
  extensionMarker.style.display = 'none';
  document.documentElement.appendChild(extensionMarker);
  
  // Listen for extension check events from the page
  window.addEventListener('cowatch-extension-check', () => {
    console.log('🎬 Extension check received, responding...');
    window.dispatchEvent(new CustomEvent('cowatch-extension-response'));
  });

  const COWATCH_URL = 'https://cowatch.app';
  const SOCKET_URL = 'wss://cowatch.app/ws';

  // Supported languages
  const CHAT_LANGUAGES = [
    { code: 'en', name: 'English', flag: '🇺🇸' },
    { code: 'tr', name: 'Türkçe', flag: '🇹🇷' },
    { code: 'es', name: 'Español', flag: '🇪🇸' },
    { code: 'de', name: 'Deutsch', flag: '🇩🇪' },
    { code: 'fr', name: 'Français', flag: '🇫🇷' },
    { code: 'it', name: 'Italiano', flag: '🇮🇹' },
    { code: 'pt', name: 'Português', flag: '🇧🇷' },
    { code: 'ru', name: 'Русский', flag: '🇷🇺' },
    { code: 'ja', name: '日本語', flag: '🇯🇵' },
    { code: 'ko', name: '한국어', flag: '🇰🇷' },
    { code: 'zh', name: '中文', flag: '🇨🇳' },
    { code: 'ar', name: 'العربية', flag: '🇸🇦' },
  ];

  // Chat translations
  const CHAT_TRANSLATIONS = {
    en: {
      watchTogether: 'Watch Together',
      participants: 'Participants',
      typeMessage: 'Type a message...',
      searchGifs: 'Search GIFs...',
      editProfile: 'Edit Profile',
      yourName: 'Your name',
      saveChanges: 'Save Changes',
      uploadImage: 'Upload Image',
      selectAvatar: 'Select Avatar',
      setUpProfile: 'Set Up Your Profile',
      chooseNameAvatar: 'Choose a name and avatar to get started',
      createRoom: 'Create Room',
      joinRoom: 'Join Room',
      roomCode: 'Room code',
      or: 'or',
      enterRoomCode: 'Enter code to join a room',
      hosting: 'Hosting',
      watching: 'Watching',
      you: 'You',
      host: 'Host',
      joinedParty: 'You joined the party!',
      userJoined: 'joined the party',
      userLeft: 'left the party',
      resumedPlayback: 'resumed playback',
      pausedPlayback: 'paused playback',
      jumpedTo: 'jumped to',
      language: 'Language',
      close: 'Close',
      syncing: 'Syncing...',
      synced: 'Synced!',
    },
    tr: {
      watchTogether: 'Birlikte İzle',
      participants: 'Katılımcılar',
      typeMessage: 'Mesaj yaz...',
      searchGifs: 'GIF ara...',
      editProfile: 'Profili Düzenle',
      yourName: 'Adınız',
      saveChanges: 'Değişiklikleri Kaydet',
      uploadImage: 'Resim Yükle',
      selectAvatar: 'Avatar Seç',
      setUpProfile: 'Profilini Ayarla',
      chooseNameAvatar: 'Başlamak için bir isim ve avatar seç',
      createRoom: 'Oda Oluştur',
      joinRoom: 'Odaya Katıl',
      roomCode: 'Oda kodu',
      or: 'veya',
      enterRoomCode: 'Odaya katılmak için kodu gir',
      hosting: 'Yönetiyor',
      watching: 'İzliyor',
      you: 'Sen',
      host: 'Yönetici',
      joinedParty: 'Partiye katıldın!',
      userJoined: 'partiye katıldı',
      userLeft: 'partiden ayrıldı',
      resumedPlayback: 'oynatmaya devam etti',
      pausedPlayback: 'oynatmayı durdurdu',
      jumpedTo: 'atladı:',
      language: 'Dil',
      close: 'Kapat',
      syncing: 'Senkronize ediliyor...',
      synced: 'Senkronize edildi!',
    },
    es: {
      watchTogether: 'Ver Juntos',
      participants: 'Participantes',
      typeMessage: 'Escribe un mensaje...',
      searchGifs: 'Buscar GIFs...',
      editProfile: 'Editar Perfil',
      yourName: 'Tu nombre',
      saveChanges: 'Guardar Cambios',
      uploadImage: 'Subir Imagen',
      selectAvatar: 'Seleccionar Avatar',
      setUpProfile: 'Configura Tu Perfil',
      chooseNameAvatar: 'Elige un nombre y avatar para comenzar',
      createRoom: 'Crear Sala',
      joinRoom: 'Unirse a Sala',
      roomCode: 'Código de sala',
      or: 'o',
      enterRoomCode: 'Ingresa el código para unirte',
      hosting: 'Anfitrión',
      watching: 'Viendo',
      you: 'Tú',
      host: 'Anfitrión',
      joinedParty: '¡Te uniste a la fiesta!',
      userJoined: 'se unió a la fiesta',
      userLeft: 'dejó la fiesta',
      resumedPlayback: 'reanudó la reproducción',
      pausedPlayback: 'pausó la reproducción',
      jumpedTo: 'saltó a',
      language: 'Idioma',
      close: 'Cerrar',
    },
    de: {
      watchTogether: 'Gemeinsam Schauen',
      participants: 'Teilnehmer',
      typeMessage: 'Nachricht eingeben...',
      searchGifs: 'GIFs suchen...',
      editProfile: 'Profil Bearbeiten',
      yourName: 'Dein Name',
      saveChanges: 'Änderungen Speichern',
      uploadImage: 'Bild Hochladen',
      selectAvatar: 'Avatar Auswählen',
      setUpProfile: 'Profil Einrichten',
      chooseNameAvatar: 'Wähle einen Namen und Avatar zum Starten',
      createRoom: 'Raum Erstellen',
      joinRoom: 'Raum Beitreten',
      roomCode: 'Raumcode',
      or: 'oder',
      enterRoomCode: 'Code eingeben zum Beitreten',
      hosting: 'Hosting',
      watching: 'Schaut',
      you: 'Du',
      host: 'Host',
      joinedParty: 'Du bist der Party beigetreten!',
      userJoined: 'ist der Party beigetreten',
      userLeft: 'hat die Party verlassen',
      resumedPlayback: 'hat Wiedergabe fortgesetzt',
      pausedPlayback: 'hat Wiedergabe pausiert',
      jumpedTo: 'sprang zu',
      language: 'Sprache',
      close: 'Schließen',
    },
    fr: {
      watchTogether: 'Regarder Ensemble',
      participants: 'Participants',
      typeMessage: 'Écrire un message...',
      searchGifs: 'Rechercher des GIFs...',
      editProfile: 'Modifier le Profil',
      yourName: 'Votre nom',
      saveChanges: 'Enregistrer',
      uploadImage: 'Télécharger Image',
      selectAvatar: 'Choisir Avatar',
      setUpProfile: 'Configurer le Profil',
      chooseNameAvatar: 'Choisissez un nom et avatar pour commencer',
      createRoom: 'Créer une Salle',
      joinRoom: 'Rejoindre une Salle',
      roomCode: 'Code de salle',
      or: 'ou',
      enterRoomCode: 'Entrez le code pour rejoindre',
      hosting: 'Héberge',
      watching: 'Regarde',
      you: 'Vous',
      host: 'Hôte',
      joinedParty: 'Vous avez rejoint la fête!',
      userJoined: 'a rejoint la fête',
      userLeft: 'a quitté la fête',
      resumedPlayback: 'a repris la lecture',
      pausedPlayback: 'a mis en pause',
      jumpedTo: 'a sauté à',
      language: 'Langue',
      close: 'Fermer',
    },
    it: {
      watchTogether: 'Guarda Insieme',
      participants: 'Partecipanti',
      typeMessage: 'Scrivi un messaggio...',
      searchGifs: 'Cerca GIF...',
      editProfile: 'Modifica Profilo',
      yourName: 'Il tuo nome',
      saveChanges: 'Salva Modifiche',
      uploadImage: 'Carica Immagine',
      selectAvatar: 'Seleziona Avatar',
      setUpProfile: 'Configura il Profilo',
      chooseNameAvatar: 'Scegli un nome e avatar per iniziare',
      createRoom: 'Crea Stanza',
      joinRoom: 'Unisciti alla Stanza',
      roomCode: 'Codice stanza',
      or: 'o',
      enterRoomCode: 'Inserisci il codice per unirti',
      hosting: 'Ospita',
      watching: 'Guarda',
      you: 'Tu',
      host: 'Ospite',
      joinedParty: 'Ti sei unito alla festa!',
      userJoined: 'si è unito alla festa',
      userLeft: 'ha lasciato la festa',
      resumedPlayback: 'ha ripreso la riproduzione',
      pausedPlayback: 'ha messo in pausa',
      jumpedTo: 'saltato a',
      language: 'Lingua',
      close: 'Chiudi',
    },
    pt: {
      watchTogether: 'Assistir Juntos',
      participants: 'Participantes',
      typeMessage: 'Digite uma mensagem...',
      searchGifs: 'Pesquisar GIFs...',
      editProfile: 'Editar Perfil',
      yourName: 'Seu nome',
      saveChanges: 'Salvar Alterações',
      uploadImage: 'Enviar Imagem',
      selectAvatar: 'Selecionar Avatar',
      setUpProfile: 'Configurar Perfil',
      chooseNameAvatar: 'Escolha um nome e avatar para começar',
      createRoom: 'Criar Sala',
      joinRoom: 'Entrar na Sala',
      roomCode: 'Código da sala',
      or: 'ou',
      enterRoomCode: 'Digite o código para entrar',
      hosting: 'Hospedando',
      watching: 'Assistindo',
      you: 'Você',
      host: 'Anfitrião',
      joinedParty: 'Você entrou na festa!',
      userJoined: 'entrou na festa',
      userLeft: 'saiu da festa',
      resumedPlayback: 'retomou a reprodução',
      pausedPlayback: 'pausou a reprodução',
      jumpedTo: 'pulou para',
      language: 'Idioma',
      close: 'Fechar',
    },
    ru: {
      watchTogether: 'Смотреть Вместе',
      participants: 'Участники',
      typeMessage: 'Введите сообщение...',
      searchGifs: 'Поиск GIF...',
      editProfile: 'Редактировать Профиль',
      yourName: 'Ваше имя',
      saveChanges: 'Сохранить',
      uploadImage: 'Загрузить Изображение',
      selectAvatar: 'Выбрать Аватар',
      setUpProfile: 'Настроить Профиль',
      chooseNameAvatar: 'Выберите имя и аватар для начала',
      createRoom: 'Создать Комнату',
      joinRoom: 'Присоединиться',
      roomCode: 'Код комнаты',
      or: 'или',
      enterRoomCode: 'Введите код для входа',
      hosting: 'Хостинг',
      watching: 'Смотрит',
      you: 'Вы',
      host: 'Хост',
      joinedParty: 'Вы присоединились к вечеринке!',
      userJoined: 'присоединился к вечеринке',
      userLeft: 'покинул вечеринку',
      resumedPlayback: 'возобновил воспроизведение',
      pausedPlayback: 'приостановил воспроизведение',
      jumpedTo: 'перешёл к',
      language: 'Язык',
      close: 'Закрыть',
    },
    ja: {
      watchTogether: '一緒に見る',
      participants: '参加者',
      typeMessage: 'メッセージを入力...',
      searchGifs: 'GIF検索...',
      editProfile: 'プロフィール編集',
      yourName: 'あなたの名前',
      saveChanges: '変更を保存',
      uploadImage: '画像をアップロード',
      selectAvatar: 'アバターを選択',
      setUpProfile: 'プロフィールを設定',
      chooseNameAvatar: '名前とアバターを選択してください',
      createRoom: 'ルームを作成',
      joinRoom: 'ルームに参加',
      roomCode: 'ルームコード',
      or: 'または',
      enterRoomCode: 'コードを入力して参加',
      hosting: 'ホスト中',
      watching: '視聴中',
      you: 'あなた',
      host: 'ホスト',
      joinedParty: 'パーティーに参加しました！',
      userJoined: 'がパーティーに参加しました',
      userLeft: 'がパーティーを離れました',
      resumedPlayback: 'が再生を再開しました',
      pausedPlayback: 'が一時停止しました',
      jumpedTo: 'にジャンプ',
      language: '言語',
      close: '閉じる',
    },
    ko: {
      watchTogether: '함께 보기',
      participants: '참가자',
      typeMessage: '메시지를 입력하세요...',
      searchGifs: 'GIF 검색...',
      editProfile: '프로필 편집',
      yourName: '이름',
      saveChanges: '저장',
      uploadImage: '이미지 업로드',
      selectAvatar: '아바타 선택',
      setUpProfile: '프로필 설정',
      chooseNameAvatar: '시작하려면 이름과 아바타를 선택하세요',
      createRoom: '방 만들기',
      joinRoom: '방 참가',
      roomCode: '방 코드',
      or: '또는',
      enterRoomCode: '코드를 입력하여 참가',
      hosting: '호스팅',
      watching: '시청 중',
      you: '나',
      host: '호스트',
      joinedParty: '파티에 참가했습니다!',
      userJoined: '님이 파티에 참가했습니다',
      userLeft: '님이 파티를 떠났습니다',
      resumedPlayback: '님이 재생을 재개했습니다',
      pausedPlayback: '님이 일시정지했습니다',
      jumpedTo: '으로 이동',
      language: '언어',
      close: '닫기',
    },
    zh: {
      watchTogether: '一起观看',
      participants: '参与者',
      typeMessage: '输入消息...',
      searchGifs: '搜索GIF...',
      editProfile: '编辑个人资料',
      yourName: '你的名字',
      saveChanges: '保存更改',
      uploadImage: '上传图片',
      selectAvatar: '选择头像',
      setUpProfile: '设置个人资料',
      chooseNameAvatar: '选择名字和头像开始',
      createRoom: '创建房间',
      joinRoom: '加入房间',
      roomCode: '房间代码',
      or: '或',
      enterRoomCode: '输入代码加入',
      hosting: '主持',
      watching: '观看中',
      you: '你',
      host: '主持人',
      joinedParty: '你加入了派对！',
      userJoined: '加入了派对',
      userLeft: '离开了派对',
      resumedPlayback: '恢复了播放',
      pausedPlayback: '暂停了播放',
      jumpedTo: '跳转到',
      language: '语言',
      close: '关闭',
    },
    ar: {
      watchTogether: 'شاهد معًا',
      participants: 'المشاركون',
      typeMessage: 'اكتب رسالة...',
      searchGifs: 'بحث GIF...',
      editProfile: 'تعديل الملف الشخصي',
      yourName: 'اسمك',
      saveChanges: 'حفظ التغييرات',
      uploadImage: 'رفع صورة',
      selectAvatar: 'اختر الصورة الرمزية',
      setUpProfile: 'إعداد الملف الشخصي',
      chooseNameAvatar: 'اختر اسمًا وصورة للبدء',
      createRoom: 'إنشاء غرفة',
      joinRoom: 'الانضمام للغرفة',
      roomCode: 'رمز الغرفة',
      or: 'أو',
      enterRoomCode: 'أدخل الرمز للانضمام',
      hosting: 'مستضيف',
      watching: 'يشاهد',
      you: 'أنت',
      host: 'المضيف',
      joinedParty: 'انضممت إلى الحفلة!',
      userJoined: 'انضم إلى الحفلة',
      userLeft: 'غادر الحفلة',
      resumedPlayback: 'استأنف التشغيل',
      pausedPlayback: 'أوقف التشغيل مؤقتًا',
      jumpedTo: 'انتقل إلى',
      language: 'اللغة',
      close: 'إغلاق',
    },
  };

  // Browser API compatibility
  const browserAPI = typeof CoWatchBrowser !== 'undefined' ? CoWatchBrowser : null;
  const storage = browserAPI?.storage?.local || chrome.storage.local;

  // Animated Reaction GIFs (using Giphy - more reliable)
  const ANIMATED_REACTIONS = [
    // Row 1 - Classic Emotions
    { id: 'laugh', name: 'Laugh', emoji: '😂', gif: 'https://media.giphy.com/media/ZqlvCTNHpqrio/giphy.gif' },
    { id: 'love', name: 'Love', emoji: '❤️', gif: 'https://media.giphy.com/media/l0HlOBZcl7sbV6LnO/giphy.gif' },
    { id: 'wow', name: 'Wow', emoji: '😮', gif: 'https://media.giphy.com/media/l0MYGb1LuZ3n7dRnO/giphy.gif' },
    { id: 'sad', name: 'Sad', emoji: '😢', gif: 'https://media.giphy.com/media/d2lcHJTG5Tscg/giphy.gif' },
    { id: 'angry', name: 'Angry', emoji: '😠', gif: 'https://media.giphy.com/media/l1J9u3TZfpmeDLkD6/giphy.gif' },
    { id: 'fire', name: 'Fire', emoji: '🔥', gif: 'https://media.giphy.com/media/l378gODdla75xN0CA/giphy.gif' },
    // Row 2 - Celebrations
    { id: 'clap', name: 'Clap', emoji: '👏', gif: 'https://media.giphy.com/media/l0MYJnJQ4EiYLxvQ4/giphy.gif' },
    { id: 'party', name: 'Party', emoji: '🎉', gif: 'https://media.giphy.com/media/26tOZ42Mg6pbTUPHW/giphy.gif' },
    { id: 'thumbsup', name: 'Thumbs Up', emoji: '👍', gif: 'https://media.giphy.com/media/111ebonMs90YLu/giphy.gif' },
    { id: 'heart_eyes', name: 'Heart Eyes', emoji: '😍', gif: 'https://media.giphy.com/media/l4Ki2obCyAQS5WhFe/giphy.gif' },
    { id: 'shocked', name: 'Shocked', emoji: '😱', gif: 'https://media.giphy.com/media/l0MYC0LajbaPoEADu/giphy.gif' },
    { id: 'crying', name: 'Crying', emoji: '😭', gif: 'https://media.giphy.com/media/OPU6wzx8JrHna/giphy.gif' },
    // Row 3 - Fun Actions
    { id: 'mindblown', name: 'Mind Blown', emoji: '🤯', gif: 'https://media.giphy.com/media/xT0xeJpnrWC4XWblEk/giphy.gif' },
    { id: 'cool', name: 'Cool', emoji: '😎', gif: 'https://media.giphy.com/media/62PP2yEIAZF6g/giphy.gif' },
    { id: 'popcorn', name: 'Popcorn', emoji: '🍿', gif: 'https://media.giphy.com/media/tyqcJoNjNv0Fq/giphy.gif' },
    { id: 'sleepy', name: 'Sleepy', emoji: '😴', gif: 'https://media.giphy.com/media/xT9IgG50Fb7Mi0prBC/giphy.gif' },
    { id: 'scared', name: 'Scared', emoji: '😨', gif: 'https://media.giphy.com/media/bEVKYB487Lqxy/giphy.gif' },
    { id: 'thinking', name: 'Thinking', emoji: '🤔', gif: 'https://media.giphy.com/media/a5viI92PAF89q/giphy.gif' },
    // Row 4 - Special Effects
    { id: 'rainbow', name: 'Rainbow', emoji: '🌈', gif: 'https://media.giphy.com/media/3o7aD4GrHwn8vsGBTa/giphy.gif' },
    { id: 'star', name: 'Star', emoji: '⭐', gif: 'https://media.giphy.com/media/xT0BKL21U5nnucdpx6/giphy.gif' },
    { id: 'lightning', name: 'Lightning', emoji: '⚡', gif: 'https://media.giphy.com/media/3o7TKSjRrfIPjeiVyM/giphy.gif' },
    { id: 'rocket', name: 'Rocket', emoji: '🚀', gif: 'https://media.giphy.com/media/JzOyy8vKMCwvK/giphy.gif' },
    { id: 'sparkle', name: 'Sparkle', emoji: '✨', gif: 'https://media.giphy.com/media/l4FGni1RBAR2OWsGk/giphy.gif' },
    { id: 'confetti', name: 'Confetti', emoji: '🎊', gif: 'https://media.giphy.com/media/g9582DNuQppxC/giphy.gif' },
  ];

  // Avatar templates (Teleparty style)
  const AVATARS = [
    { id: 'cat', emoji: '🐱', color: '#FF6B6B' },
    { id: 'dog', emoji: '🐶', color: '#4ECDC4' },
    { id: 'fox', emoji: '🦊', color: '#FF9F43' },
    { id: 'lion', emoji: '🦁', color: '#F9CA24' },
    { id: 'panda', emoji: '🐼', color: '#2D3436' },
    { id: 'koala', emoji: '🐨', color: '#636E72' },
    { id: 'tiger', emoji: '🐯', color: '#E17055' },
    { id: 'bear', emoji: '🐻', color: '#A0522D' },
    { id: 'rabbit', emoji: '🐰', color: '#FDA7DF' },
    { id: 'unicorn', emoji: '🦄', color: '#A29BFE' },
    { id: 'dragon', emoji: '🐲', color: '#00B894' },
    { id: 'alien', emoji: '👽', color: '#6C5CE7' },
    { id: 'robot', emoji: '🤖', color: '#74B9FF' },
    { id: 'ghost', emoji: '👻', color: '#DFE6E9' },
    { id: 'skull', emoji: '💀', color: '#2D3436' },
    { id: 'devil', emoji: '😈', color: '#D63031' },
  ];

  // GIF categories with Tenor-style search terms (professional names)
  const GIF_CATEGORIES = [
    { id: 'trending', name: 'Trending', searchTerm: 'trending' },
    { id: 'reactions', name: 'Reactions', searchTerm: 'reaction' },
    { id: 'love', name: 'Love', searchTerm: 'love' },
    { id: 'happy', name: 'Happy', searchTerm: 'happy' },
    { id: 'sad', name: 'Sad', searchTerm: 'sad' },
    { id: 'angry', name: 'Angry', searchTerm: 'angry' },
    { id: 'surprised', name: 'Wow', searchTerm: 'surprised wow' },
    { id: 'dance', name: 'Dance', searchTerm: 'dance' },
    { id: 'celebrate', name: 'Party', searchTerm: 'celebrate party' },
    { id: 'agree', name: 'Agree', searchTerm: 'agree yes thumbs up' },
  ];

  // Comprehensive emoji list for reactions
  const REACTION_EMOJIS = [
    // Smileys
    '😀', '😃', '😄', '😁', '😅', '😂', '🤣', '😊', '😇', '🙂', '😉', '😌',
    '😍', '🥰', '😘', '😗', '😙', '😚', '😋', '😛', '😜', '🤪', '😝', '🤗',
    '🤭', '🤫', '🤔', '🤐', '🤨', '😐', '😑', '😶', '😏', '😒', '🙄', '😬',
    '😮', '🤯', '😱', '😨', '😰', '😥', '😢', '😭', '😤', '😡', '🤬', '😈',
    '👿', '💀', '☠️', '💩', '🤡', '👹', '👺', '👻', '👽', '🤖', '😺', '😸',
    // Gestures
    '👍', '👎', '👊', '✊', '🤛', '🤜', '🤞', '✌️', '🤟', '🤘', '👌', '🤌',
    '🤏', '👈', '👉', '👆', '👇', '☝️', '✋', '🤚', '🖐', '🖖', '👋', '🤙',
    '💪', '🦾', '🙏', '🤝', '👏', '🙌', '👐', '🤲',
    // Hearts & Love
    '❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '🤎', '💔', '❣️', '💕',
    '💞', '💓', '💗', '💖', '💘', '💝', '💟',
    // Celebration
    '🎉', '🎊', '🥳', '🎂', '🎁', '🎈', '🪅', '🎆', '🎇', '✨', '🌟', '💫',
    // Objects & Symbols
    '🔥', '💯', '💢', '💥', '💫', '💦', '💨', '🕳️', '💣', '💬', '👁️‍🗨️', '🗨️',
    '🗯️', '💭', '💤', '🎵', '🎶', '🎤', '🎧', '📱', '💻', '🎮', '🎬', '📸'
  ];

  // Quick reactions for the floating bar above input (Teleparty style)
  const QUICK_REACTIONS = ['😂', '😍', '😮', '😢', '😡', '👍', '👎', '🔥', '💯', '🎉'];

  // Tenor API key for GIF search (you should replace with your own)
  const TENOR_API_KEY = 'AIzaSyAyimkuYQYF_FXVALexPuGQctUWRURdCYQ'; // Google/Tenor API key
  const TENOR_API_URL = 'https://tenor.googleapis.com/v2';

class CoWatch {
  constructor() {
    this.video = null;
    this.socket = null;
    this.roomCode = null;
    this.isHost = false;
    this.userId = this.generateUserId();
    this.odaId = null; // Server-assigned ID (set on room:joined)
    this.visitorId = null; // Alias for odaId
    this.username = '';
    this.userAvatar = null;
    this.participants = [];
    this.messages = [];
    this.isConnected = false;
    this.ignoreVideoEvents = false;
    this.platform = this.detectPlatform();
    this.chatOpen = true; // Panel starts OPEN so users can see the UI immediately
    this.lastSyncTime = 0;
    this._hasJoinedRoom = false; // Prevent duplicate room:joined handling
    this.unreadCount = 0;
    this.emojiPickerOpen = false;
    this.gifPickerOpen = false;
    this.replyingTo = null;
    this.theme = 'dark'; // dark or light
    this.language = 'en'; // current language
    this.languagePickerOpen = false;
    this.currentPanel = 'chat'; // 'chat', 'participants', 'profile'
    this.isRecording = false;
    this.mediaRecorder = null;
    this.audioChunks = [];
    this.currentAudio = null; // For voice message playback
    this.recordingStream = null; // For recording stream
    this.hostOnlyControl = false; // Only host can control video
    this.reactionsDisabled = false; // Disable reactions for party
    this._lastRenderState = null; // For smart UI updates
    this._initComplete = false; // Track if init is complete
    this._pendingMessages = []; // Queue messages until init completes
    this._hasJoinedRoom = false; // Prevent duplicate room:joined handling
    
    // New features
    this.isSyncing = false; // Sync status indicator
    this.syncStatusTimeout = null; // For hiding sync status
    this.isDragging = false; // Panel drag state
    this.isResizing = false; // Panel resize state
    this.panelPosition = { x: null, y: null }; // Custom panel position
    this.panelSize = { width: 340, height: null }; // Custom panel size (Teleparty: 340px)
    this.messageDialogTimeout = null; // For auto-hiding message dialogs
    this.videoObserver = null; // DOM observer for video elements
    
    // DOM-based chat (Teleparty approach) - ENABLED
    // Chat UI is directly injected into the page for better compatibility
    // Teleparty uses direct DOM injection, NOT iframes - this avoids CSP issues
    this.useIframeChat = false;
    
    // === TELEPARTY-STYLE NEW FEATURES ===
    // Profile setup flag - only show setup when user tries to create room without profile
    this.showProfileSetup = false;
    
    // Typing indicator (who is typing)
    this.usersTyping = []; // Array of userIds currently typing
    this.typingTimeout = null; // Timeout to clear typing status
    this.typingTimeouts = {}; // Individual timeouts per user
    
    // Buffering indicator (who is buffering)
    this.usersBuffering = []; // Array of userIds currently buffering
    
    // On-screen reactions (Teleparty animated emojis)
    this.reactionContainer = null; // Container for floating reactions
    
    // GIF search state
    this.gifResults = []; // Current GIF search results
    this.gifSearchQuery = ''; // Current search query
    this.gifSearchTimeout = null; // Debounce for search
    
    // User presence
    this.userStatus = 'online'; // online, away, offline
    this.lastActivityTime = Date.now();
    
    // Name colors (Teleparty premium feature)
    this.nameColor = null; // Custom name color
    
    // === CRITICAL SYNC FEATURES (Teleparty-style) ===
    // Server time synchronization
    this.serverTimeOffset = 0; // Difference between server and client time
    this.pingHistory = []; // Last N ping measurements
    this.avgPing = 0; // Average ping in ms
    this.lastServerSync = 0; // Last server sync timestamp
    this.serverSyncInterval = null; // Interval for periodic sync
    
    // Task queue for sync operations (prevents conflicts)
    this.taskQueue = [];
    this.taskInProgress = false;
    this.taskNames = new Set();
    
    // WebSocket reconnection
    this.reconnectAttempts = 0;
    this.maxReconnectAttempts = 10;
    this.reconnectBackoff = 1000; // Start with 1 second
    this.reconnectTimeout = null;
    this.connectionLost = false;
    
    // Auto-rejoin party
    this.autoRejoinData = null;
    this.autoRejoinRetries = 0;
    this.maxAutoRejoinRetries = 3;
    
    // Expected video state (for sync validation)
    this.expectedState = {
      time: 0,
      playing: false,
      lastUpdate: 0
    };
    
    // === FRIENDS SYSTEM ===
    this.friends = []; // List of friends
    this.inviteModalOpen = false; // Invite friends modal state
    this.inviteActiveTab = 'link'; // 'link' or 'friends'
    this.invitedFriends = new Set(); // Friends already invited to this room
    this.friendsLoading = false; // Loading state
    
    // === SCHEDULED EVENTS SYSTEM ===
    this.scheduledEvents = []; // List of scheduled watch parties
    this.scheduledEventsModalOpen = false;
    this.createEventModalOpen = false;
    
    // === SOCKET.IO FALLBACK ===
    this.useSocketIo = false; // Whether to use Socket.io fallback
    this.socketIoAttempts = 0; // Number of failed WebSocket attempts before fallback
    this.maxWsFailuresBeforeFallback = 3;
    
    // === A/B TESTING ===
    this.experimentBuckets = {}; // User's experiment assignments
    this.permId = this.getOrCreatePermId(); // Permanent ID for experiments
    
    // === ANALYTICS ===
    this.analyticsEnabled = false; // User can opt-in
    this.analyticsQueue = []; // Queue of events to send
    this.sessionId = this.generateSessionId();
    this.sessionStartTime = Date.now();
    
    // === REVIEW PROMPT ===
    this.usageCount = 0;
    this.hasReviewed = false;
    this.reviewPromptShown = false;
    
    this.init();
  }

  // === PERMANENT ID FOR A/B TESTING ===
  getOrCreatePermId() {
    let permId = localStorage.getItem('cowatch_perm_id');
    if (!permId) {
      permId = 'perm_' + Math.random().toString(36).substr(2, 16) + Date.now().toString(36);
      localStorage.setItem('cowatch_perm_id', permId);
    }
    return permId;
  }

  generateSessionId() {
    return 'sess_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
  }

  // === A/B TESTING (Teleparty-style murmur hash bucketing) ===
  murmur3(str) {
    let h1 = 0xdeadbeef;
    for (let i = 0; i < str.length; i++) {
      const c = str.charCodeAt(i);
      h1 = Math.imul(h1 ^ c, 2654435761);
    }
    h1 = Math.imul(h1 ^ (h1 >>> 16), 2246822507);
    h1 = Math.imul(h1 ^ (h1 >>> 13), 3266489909);
    return (h1 ^ (h1 >>> 16)) >>> 0;
  }

  getExperimentBucket(experimentName, numBuckets = 100) {
    if (this.experimentBuckets[experimentName] !== undefined) {
      return this.experimentBuckets[experimentName];
    }
    const hash = this.murmur3(this.permId + experimentName);
    const bucket = hash % numBuckets;
    this.experimentBuckets[experimentName] = bucket;
    return bucket;
  }

  isInExperiment(experimentName, percentage = 50) {
    const bucket = this.getExperimentBucket(experimentName);
    return bucket < percentage;
  }

  // === ANALYTICS ===
  trackEvent(eventName, properties = {}) {
    if (!this.analyticsEnabled) return;
    
    const event = {
      event: eventName,
      timestamp: Date.now(),
      sessionId: this.sessionId,
      permId: this.permId,
      platform: this.platform?.name,
      roomCode: this.roomCode,
      ...properties
    };
    
    this.analyticsQueue.push(event);
    
    // Batch send every 10 events or every 30 seconds
    if (this.analyticsQueue.length >= 10) {
      this.flushAnalytics();
    }
  }

  async flushAnalytics() {
    if (this.analyticsQueue.length === 0) return;
    
    const events = [...this.analyticsQueue];
    this.analyticsQueue = [];
    
    try {
      // Send to analytics endpoint (can be PostHog, Mixpanel, or custom)
      await fetch('https://cowatch.tv/api/analytics', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ events })
      });
    } catch (e) {
      // Re-queue on failure
      this.analyticsQueue = [...events, ...this.analyticsQueue];
    }
  }

  // === REVIEW PROMPT SYSTEM ===
  async loadReviewState() {
    try {
      const storage = await browserAPI.storage.local.get(['cowatch_usage_count', 'cowatch_has_reviewed', 'cowatch_review_prompt_shown']);
      this.usageCount = storage.cowatch_usage_count || 0;
      this.hasReviewed = storage.cowatch_has_reviewed || false;
      this.reviewPromptShown = storage.cowatch_review_prompt_shown || false;
    } catch (e) {}
  }

  async incrementUsageCount() {
    this.usageCount++;
    try {
      await browserAPI.storage.local.set({ cowatch_usage_count: this.usageCount });
    } catch (e) {}
    
    // Show review prompt after 5 successful watch parties
    if (this.usageCount >= 5 && !this.hasReviewed && !this.reviewPromptShown) {
      this.showReviewPrompt();
    }
  }

  showReviewPrompt() {
    this.reviewPromptShown = true;
    browserAPI.storage.local.set({ cowatch_review_prompt_shown: true });
    
    // Create review prompt UI
    const prompt = document.createElement('div');
    prompt.className = 'cowatch-review-prompt';
    prompt.innerHTML = `
      <div class="cowatch-review-content">
        <div class="cowatch-review-icon">🎉</div>
        <h3>Enjoying CoWatch?</h3>
        <p>You've had ${this.usageCount} watch parties! Would you mind leaving us a review?</p>
        <div class="cowatch-review-buttons">
          <button class="cowatch-review-btn primary" id="cowatchReviewYes">
            ⭐ Leave Review
          </button>
          <button class="cowatch-review-btn secondary" id="cowatchReviewLater">
            Maybe Later
          </button>
        </div>
      </div>
    `;
    
    document.body.appendChild(prompt);
    
    document.getElementById('cowatchReviewYes')?.addEventListener('click', () => {
      this.hasReviewed = true;
      browserAPI.storage.local.set({ cowatch_has_reviewed: true });
      // Open Chrome Web Store review page
      window.open('https://chrome.google.com/webstore/detail/cowatch/YOUR_EXTENSION_ID/reviews', '_blank');
      prompt.remove();
    });
    
    document.getElementById('cowatchReviewLater')?.addEventListener('click', () => {
      prompt.remove();
    });
    
    // Auto-dismiss after 10 seconds
    setTimeout(() => prompt.remove(), 10000);
  }

  // Get translation for current language
  t(key) {
    const translations = CHAT_TRANSLATIONS[this.language] || CHAT_TRANSLATIONS['en'];
    return translations[key] || CHAT_TRANSLATIONS['en'][key] || key;
  }

  generateUserId() {
    return 'user_' + Math.random().toString(36).substr(2, 9) + '_' + Date.now();
  }

  detectPlatform() {
    const host = window.location.hostname;
    const platforms = {
      // ========== Primary Platforms ==========
      'netflix.com': { name: 'netflix', label: 'Netflix', color: '#E50914', icon: '🔴' },
      'youtube.com': { name: 'youtube', label: 'YouTube', color: '#FF0000', icon: '▶️' },
      'disneyplus.com': { name: 'disney', label: 'Disney+', color: '#113CCF', icon: '🏰' },
      'primevideo.com': { name: 'prime', label: 'Prime Video', color: '#00A8E1', icon: '📦' },
      'amazon.com': { name: 'prime', label: 'Prime Video', color: '#00A8E1', icon: '📦' },
      'hbomax.com': { name: 'hbo', label: 'Max', color: '#B535F6', icon: '💜' },
      'max.com': { name: 'max', label: 'Max', color: '#B535F6', icon: '💜' },
      'play.max.com': { name: 'max', label: 'Max', color: '#B535F6', icon: '💜' },
      'hulu.com': { name: 'hulu', label: 'Hulu', color: '#1CE783', icon: '💚' },
      
      // ========== Major Streaming Services ==========
      'peacocktv.com': { name: 'peacock', label: 'Peacock', color: '#000000', icon: '🦚' },
      'paramountplus.com': { name: 'paramount', label: 'Paramount+', color: '#0064FF', icon: '⭐' },
      'crunchyroll.com': { name: 'crunchyroll', label: 'Crunchyroll', color: '#F47521', icon: '🍥' },
      'tubitv.com': { name: 'tubi', label: 'Tubi', color: '#FA382F', icon: '📺' },
      'pluto.tv': { name: 'pluto', label: 'Pluto TV', color: '#FFD000', icon: '🪐' },
      'tv.apple.com': { name: 'appletv', label: 'Apple TV+', color: '#000000', icon: '🍎' },
      'mubi.com': { name: 'mubi', label: 'MUBI', color: '#0FA3B1', icon: '🎬' },
      'funimation.com': { name: 'funimation', label: 'Funimation', color: '#410099', icon: '🎌' },
      
      // ========== Sports Streaming ==========
      'espn.com': { name: 'espn', label: 'ESPN+', color: '#D00000', icon: '🏈' },
      'fubo.tv': { name: 'fubo', label: 'FuboTV', color: '#FA4616', icon: '⚽' },
      'fancode.com': { name: 'fancode', label: 'FanCode', color: '#E91E63', icon: '🏏' },
      'willow.tv': { name: 'willow', label: 'Willow TV', color: '#00796B', icon: '🏏' },
      
      // ========== North America Regional ==========
      'crave.ca': { name: 'crave', label: 'Crave', color: '#00B4E4', icon: '🇨🇦' },
      'sling.com': { name: 'sling', label: 'Sling TV', color: '#0F9DFB', icon: '📡' },
      
      // ========== Europe Regional ==========
      'canalplus.com': { name: 'canalplus', label: 'Canal+', color: '#000000', icon: '🇫🇷' },
      'mycanal.fr': { name: 'canalplus', label: 'Canal+', color: '#000000', icon: '🇫🇷' },
      
      // ========== Australia/NZ Regional ==========
      'stan.com.au': { name: 'stan', label: 'Stan', color: '#0066B2', icon: '🇦🇺' },
      
      // ========== Asia Regional ==========
      'hotstar.com': { name: 'hotstar', label: 'Hotstar', color: '#1F2D5C', icon: '⭐' },
      'viki.com': { name: 'viki', label: 'Viki', color: '#0CB6D6', icon: '🇰🇷' },
      'zee5.com': { name: 'zee5', label: 'ZEE5', color: '#6B3FA0', icon: '🇮🇳' },
      'sonyliv.com': { name: 'sonyliv', label: 'SonyLIV', color: '#0B65AF', icon: '🇮🇳' },
      'jiocinema.com': { name: 'jiocinema', label: 'JioCinema', color: '#E31E24', icon: '🇮🇳' },
      'viu.com': { name: 'viu', label: 'Viu', color: '#FFC300', icon: '🌏' },
      'vidio.com': { name: 'vidio', label: 'Vidio', color: '#FF0045', icon: '🇮🇩' },
      
      // ========== Japan Regional ==========
      'hulu.jp': { name: 'hulujp', label: 'Hulu Japan', color: '#1CE783', icon: '🇯🇵' },
      'unext.jp': { name: 'unext', label: 'U-NEXT', color: '#1F1F1F', icon: '🇯🇵' },
      'video.unext.jp': { name: 'unext', label: 'U-NEXT', color: '#1F1F1F', icon: '🇯🇵' },
      
      // ========== MENA Regional ==========
      'shahid.mbc.net': { name: 'shahid', label: 'Shahid', color: '#35C8A0', icon: '🇸🇦' },
      'disneyplus.com.mena': { name: 'disneymena', label: 'Disney+ MENA', color: '#113CCF', icon: '🏰' },
      
      // ========== LATAM Regional ==========
      'starplus.com': { name: 'starplus', label: 'Star+', color: '#FF0000', icon: '⭐' },
      
      // ========== Music ==========
      'spotify.com': { name: 'spotify', label: 'Spotify', color: '#1DB954', icon: '🎵' },
      'open.spotify.com': { name: 'spotify', label: 'Spotify', color: '#1DB954', icon: '🎵' },
      
      // ========== Video Platforms ==========
      'twitch.tv': { name: 'twitch', label: 'Twitch', color: '#9146FF', icon: '💬' },
      'vimeo.com': { name: 'vimeo', label: 'Vimeo', color: '#1AB7EA', icon: '🎥' },
      'dailymotion.com': { name: 'dailymotion', label: 'Dailymotion', color: '#00AAFF', icon: '📹' },
    };
    
    for (const [domain, info] of Object.entries(platforms)) {
      if (host.includes(domain.replace('.com', '').replace('.tv', '').replace('.ca', '').replace('.au', '').replace('.net', ''))) {
        return info;
      }
    }
    return { name: 'other', label: 'Video', color: '#FF3B30', icon: '🎬' };
  }

  // Check if current page is a watchable video page (not homepage/browse)
  isWatchablePage() {
    const url = window.location.href;
    const path = window.location.pathname;
    const host = window.location.hostname;
    
    // YouTube - only watch pages
    if (host.includes('youtube.com')) {
      return path.startsWith('/watch') || path.startsWith('/shorts') || path.startsWith('/live');
    }
    
    // Netflix - only watch pages
    if (host.includes('netflix.com')) {
      return path.startsWith('/watch');
    }
    
    // Disney+ - only video pages
    if (host.includes('disneyplus.com')) {
      return path.includes('/video/') || path.includes('/play/');
    }
    
    // Prime Video - only watch pages
    if (host.includes('primevideo.com') || host.includes('amazon.com')) {
      return path.includes('/detail/') || path.includes('/watch/') || url.includes('/gp/video/');
    }
    
    // HBO Max / Max - only play pages
    if (host.includes('max.com') || host.includes('hbomax.com')) {
      return path.includes('/play/') || path.includes('/video/') || path.startsWith('/player/');
    }
    
    // Hulu - only watch pages
    if (host.includes('hulu.com')) {
      return path.startsWith('/watch');
    }
    
    // Peacock - only watch pages
    if (host.includes('peacocktv.com')) {
      return path.includes('/watch/') || path.includes('/player');
    }
    
    // Paramount+ - only watch/video pages
    if (host.includes('paramountplus.com')) {
      return path.includes('/video/') || path.includes('/movies/') || path.includes('/shows/');
    }
    
    // Apple TV+ - only video pages
    if (host.includes('tv.apple.com')) {
      return path.includes('/episode/') || path.includes('/movie/') || path.includes('/show/');
    }
    
    // Crunchyroll - only watch pages
    if (host.includes('crunchyroll.com')) {
      return path.includes('/watch/');
    }
    
    // Twitch - only channel/video pages
    if (host.includes('twitch.tv')) {
      // Twitch channels and videos, not directory
      return !path.startsWith('/directory') && path !== '/' && path.length > 1;
    }
    
    // ESPN - watch pages
    if (host.includes('espn.com')) {
      return path.includes('/watch') || path.includes('/player');
    }
    
    // Fubo - watch pages
    if (host.includes('fubo.tv')) {
      return path.includes('/watch') || path.includes('/player');
    }
    
    // Tubi - watch pages
    if (host.includes('tubitv.com')) {
      return path.includes('/movies/') || path.includes('/tv-shows/') || path.includes('/video/');
    }
    
    // Pluto TV - watch pages
    if (host.includes('pluto.tv')) {
      return path.includes('/on-demand/') || path.includes('/live-tv/');
    }
    
    // Mubi - watch pages
    if (host.includes('mubi.com')) {
      return path.includes('/films/') && path.includes('/watch');
    }
    
    // Hotstar - watch pages
    if (host.includes('hotstar.com')) {
      return path.includes('/watch/') || path.includes('/play/');
    }
    
    // Viki - watch pages
    if (host.includes('viki.com')) {
      return path.includes('/videos/');
    }
    
    // ZEE5 - watch pages
    if (host.includes('zee5.com')) {
      return path.includes('/watch/') || path.includes('/video/');
    }
    
    // SonyLIV - watch pages
    if (host.includes('sonyliv.com')) {
      return path.includes('/shows/') || path.includes('/movies/');
    }
    
    // JioCinema - watch pages
    if (host.includes('jiocinema.com')) {
      return path.includes('/watch/');
    }
    
    // Vimeo - only video pages (numeric IDs)
    if (host.includes('vimeo.com')) {
      return /\/\d+/.test(path);
    }
    
    // Dailymotion - video pages
    if (host.includes('dailymotion.com')) {
      return path.includes('/video/');
    }
    
    // For other platforms, allow any page with video
    return true;
  }

  // Watch for URL changes to reinitialize when navigating to watchable page
  setupUrlWatcher() {
    let lastUrl = window.location.href;
    
    const checkAndReinit = () => {
      const currentUrl = window.location.href;
      if (currentUrl !== lastUrl) {
        lastUrl = currentUrl;
        console.log('🎬 URL changed to:', currentUrl);
        
        // If we're now on a watchable page and UI doesn't exist, do full init
        if (this.isWatchablePage() && !document.getElementById('cowatch-container')) {
          console.log('🎬 Now on watchable page, initializing...');
          this.fullInit();
        }
      }
    };
    
    // Check periodically
    setInterval(checkAndReinit, 1000);
    
    // Also listen for navigation events
    window.addEventListener('popstate', checkAndReinit);
    
    // Override history methods
    const originalPushState = history.pushState;
    const originalReplaceState = history.replaceState;
    
    history.pushState = function(...args) {
      originalPushState.apply(this, args);
      setTimeout(checkAndReinit, 100);
    };
    
    history.replaceState = function(...args) {
      originalReplaceState.apply(this, args);
      setTimeout(checkAndReinit, 100);
    };
  }

  // Inject platform-specific script for better player API access
  injectPlatformScript() {
    const platform = this.platform.name;
    // Map all 36 supported platforms to their injected scripts (Teleparty parity)
    const scriptMap = {
      // ========== Primary Platforms ==========
      'netflix': 'netflix.js',
      'prime': 'prime.js',         // Amazon Prime Video
      'disney': 'disney.js',       // Disney+
      'hulu': 'hulu.js',
      'hbo': 'hbo.js',             // Legacy HBO
      'max': 'max.js',             // HBO Max rebranded to Max
      'youtube': 'youtube.js',
      
      // ========== Major Streaming Services ==========
      'peacock': 'peacock.js',     // NBC Peacock
      'paramount': 'paramount.js', // Paramount+
      'crunchyroll': 'crunchyroll.js',
      'tubi': 'tubi.js',           // Tubi TV
      'pluto': 'pluto.js',         // Pluto TV
      'appletv': 'appletv.js',     // Apple TV+
      'mubi': 'mubi.js',
      'funimation': 'funimation.js',
      
      // ========== Sports Streaming ==========
      'espn': 'espn.js',           // ESPN+
      'fubo': 'fubo.js',           // FuboTV
      'fancode': 'fancode.js',     // FanCode India
      'willow': 'willow.js',       // Willow TV Cricket
      
      // ========== North America Regional ==========
      'crave': 'crave.js',         // Crave Canada
      'sling': 'sling.js',         // Sling TV
      
      // ========== Europe Regional ==========
      'canalplus': 'canalplus.js', // Canal+ France
      
      // ========== Australia/NZ Regional ==========
      'stan': 'stan.js',           // Stan Australia
      
      // ========== Asia Regional ==========
      'hotstar': 'hotstar.js',     // Disney+ Hotstar India
      'viki': 'viki.js',           // Viki Asian Drama
      'zee5': 'zee5.js',           // Zee5 India
      'sonyliv': 'sonyliv.js',     // SonyLiv India
      'jiocinema': 'jiocinema.js', // JioCinema India
      'viu': 'viu.js',             // Viu Asia
      'vidio': 'vidio.js',         // Vidio Indonesia
      
      // ========== Japan Regional ==========
      'hulujp': 'hulujp.js',       // Hulu Japan
      'unext': 'unext.js',         // U-Next Japan
      
      // ========== MENA Regional ==========
      'shahid': 'shahid.js',       // Shahid Arabic
      'disneymena': 'disneymena.js', // Disney+ MENA
      
      // ========== LATAM Regional ==========
      'starplus': 'starplus.js',   // Star+ Latin America
      
      // ========== Music ==========
      'spotify': 'spotify.js'      // Spotify (audio sync)
    };
    
    const scriptFile = scriptMap[platform];
    if (!scriptFile) {
      console.log('🎬 No platform-specific script for:', platform);
      return;
    }
    
    // Check if already injected
    if (document.getElementById('cowatch-platform-script')) {
      return;
    }
    
    try {
      // Get the script URL from extension
      const scriptUrl = typeof chrome !== 'undefined' && chrome.runtime 
        ? chrome.runtime.getURL(`injected/${scriptFile}`)
        : null;
      
      if (!scriptUrl) {
        console.log('🎬 Cannot get script URL for:', scriptFile);
        return;
      }
      
      // Create and inject script element
      const script = document.createElement('script');
      script.id = 'cowatch-platform-script';
      script.src = scriptUrl;
      script.type = 'text/javascript';
      
      // Use document_start injection method
      (document.head || document.documentElement).appendChild(script);
      
      console.log('🎬 Platform script injected:', scriptFile);
      
      // Setup message listener for responses from injected script
      this.setupInjectedMessageListener();
      
    } catch (e) {
      console.error('🎬 Failed to inject platform script:', e);
    }
  }
  
  // Listen for messages from injected script
  setupInjectedMessageListener() {
    if (this._injectedListenerSetup) return;
    this._injectedListenerSetup = true;
    
    // Listen for CoWatch style messages
    window.addEventListener('message', (event) => {
      if (event.data?.source !== 'cowatch-injected') return;
      
      const { action, data } = event.data;
      
      // Handle responses from platform scripts
      switch (action) {
        case 'getStateResponse':
          if (this._pendingStateCallback) {
            this._pendingStateCallback(data);
            this._pendingStateCallback = null;
          }
          break;
        case 'playResponse':
        case 'pauseResponse':
        case 'seekResponse':
          // These are fire-and-forget operations
          console.log('🎬 Platform action completed:', action, data);
          break;
      }
    });
    
    // Listen for Teleparty-style FromNode events (Netflix)
    window.addEventListener('FromNode', (event) => {
      const detail = event.detail;
      if (!detail?.type) return;
      
      // Update internal state based on response type
      switch (detail.type) {
        case 'UpdateState':
          // Store player state for sync
          this._playerState = {
            time: (detail.time || 0) / 1000, // Convert ms to seconds
            paused: detail.paused,
            loading: detail.loading,
            adState: detail.adState,
            actionsState: detail.actionsState,
            updatedAt: detail.updatedAt
          };
          // Resolve pending state callback if any
          if (this._pendingStateCallback) {
            this._pendingStateCallback({
              time: this._playerState.time,
              duration: this.video?.duration || 0,
              paused: this._playerState.paused,
              loading: this._playerState.loading,
              videoId: window.location.pathname.match(/watch\/(\d+)/)?.[1] || null,
              title: document.title?.replace(' - Netflix', '') || ''
            });
            this._pendingStateCallback = null;
          }
          break;
        case 'CurrentTime':
          this._currentTime = (detail.time || 0) / 1000;
          break;
        case 'GetTitle':
          this._pageTitle = detail.pageTitle;
          break;
        case 'GetType':
          this._videoType = detail.VideoType;
          break;
        case 'GetEpData':
          this._episodeData = detail.episodeData;
          break;
        case 'GetVideoData':
          this._videoLookupData = detail.videoLookupData;
          break;
        case 'CheckSkipSupplemental':
          console.log('🎬 Supplemental check completed');
          break;
      }
    }, false);
  }
  
  // Send command to platform script (supports both CoWatch and Teleparty protocols)
  sendToPlatform(action, data = {}) {
    // For Netflix, use Teleparty-style messages
    if (this.platform.name === 'netflix') {
      const typeMap = {
        'getState': 'GetState',
        'play': 'PLAY',
        'pause': 'PAUSE',
        'seek': 'SEEK'
      };
      const type = typeMap[action] || action.toUpperCase();
      window.postMessage({ type, ...data }, '*');
    } else {
      // For other platforms, use CoWatch style
      window.postMessage({
        source: 'cowatch-content',
        action,
        data
      }, '*');
    }
  }
  
  // Get playback state from platform script (async)
  async getPlatformState() {
    return new Promise((resolve) => {
      this._pendingStateCallback = resolve;
      
      // Send state request
      if (this.platform.name === 'netflix') {
        window.postMessage({ type: 'GetState' }, '*');
      } else {
        this.sendToPlatform('getState');
      }
      
      // Timeout after 500ms
      setTimeout(() => {
        if (this._pendingStateCallback) {
          this._pendingStateCallback = null;
          resolve(null);
        }
      }, 500);
    });
  }

  // Full initialization for watchable pages
  async fullInit() {
    try {
      await this.loadState();
      console.log('🎬 State loaded');
    } catch (e) {
      console.error('🎬 loadState error:', e);
    }
    
    try {
      this.injectStyles();
      console.log('🎬 Styles injected');
    } catch (e) {
      console.error('🎬 injectStyles error:', e);
    }
    
    // Inject platform-specific script for better API access
    try {
      this.injectPlatformScript();
      console.log('🎬 Platform script checked');
    } catch (e) {
      console.error('🎬 Platform script error:', e);
    }
    
    try {
      this.findVideo();
      console.log('🎬 Video found:', !!this.video);
    } catch (e) {
      console.error('🎬 findVideo error:', e);
    }
    
    // ALWAYS create UI on streaming platforms so users can create/join rooms
    // Video may load later - we'll find it via observer
    try {
      this.createUI();
      console.log('🎬 UI created (video:', !!this.video, ', roomCode:', !!this.roomCode, ')');
    } catch (e) {
      console.error('🎬 createUI error:', e);
    }
    
    try {
      this.observeDOM();
    } catch (e) {}
    
    try {
      this.setupUnloadListener();
    } catch (e) {}
    
    try {
      this.setupFullscreenListener();
    } catch (e) {}
    
    if (this.roomCode) {
      this.connectToRoom(this.roomCode);
    }
    
    this._initComplete = true;
    console.log('🎬 Full initialization complete');
  }

  async init() {
    console.log('🎬 CoWatch: Initializing on', this.platform.label);
    
    // Setup message listener FIRST - always needed for popup communication
    this.setupMessageListener();
    console.log('🎬 Message listener set up successfully');
    
    // Check if we're on a watchable page (not homepage/browse)
    if (!this.isWatchablePage()) {
      console.log('🎬 Not on a watchable page, skipping UI creation');
      this._initComplete = true; // Mark as complete so popup can still communicate
      this.setupUrlWatcher(); // Watch for URL changes to reinitialize
      return;
    }
    
    try {
      try {
        await this.loadState();
        console.log('🎬 State loaded');
      } catch (e) {
        console.error('🎬 loadState error:', e);
      }
      
      try {
        this.injectStyles();
        console.log('🎬 Styles injected');
      } catch (e) {
        console.error('🎬 injectStyles error:', e);
      }
      
      // Inject platform-specific script for better API access
      try {
        this.injectPlatformScript();
        console.log('🎬 Platform script checked');
      } catch (e) {
        console.error('🎬 Platform script error:', e);
      }
      
      try {
        this.findVideo();
        console.log('🎬 Video found:', !!this.video);
      } catch (e) {
        console.error('🎬 findVideo error:', e);
      }
      
      // ALWAYS create UI on streaming platforms so users can create/join rooms
      // Video may load later - we'll find it via observer
      try {
        this.createUI();
        console.log('🎬 UI created (video:', !!this.video, ', roomCode:', !!this.roomCode, ')');
      } catch (e) {
        console.error('🎬 createUI error:', e);
      }
      
      try {
        this.observeDOM();
        console.log('🎬 DOM observer set up');
      } catch (e) {
        console.error('🎬 observeDOM error:', e);
      }
      
      // Listen for page unload to leave room
      try {
        this.setupUnloadListener();
        console.log('🎬 Unload listener set up');
      } catch (e) {
        console.error('🎬 setupUnloadListener error:', e);
      }
      
      // Listen for fullscreen changes to ensure UI stays visible
      try {
        this.setupFullscreenListener();
        console.log('🎬 Fullscreen listener set up');
      } catch (e) {
        console.error('🎬 setupFullscreenListener error:', e);
      }
      
      // Initialize video observer for smart detection
      try {
        this.initVideoObserver();
        console.log('🎬 Video observer set up');
      } catch (e) {
        console.error('🎬 initVideoObserver error:', e);
      }
      
      if (this.roomCode) {
        this.connectToRoom(this.roomCode);
      } else {
        // Check for auto-rejoin
        this.checkAutoRejoin();
      }
      
      // Mark init as complete
      this._initComplete = true;
      console.log('🎬 CoWatch initialization complete');
    } catch (error) {
      console.error('🎬 CoWatch initialization error:', error);
      this._initComplete = true; // Still mark as complete so messages can be processed
    }
  }
  
  setupFullscreenListener() {
    console.log('🎬 Setting up adaptive fullscreen system');
    
    let lastFullscreenState = false;
    let originalParent = null;
    let isMovedToFullscreen = false;
    
    // Get the best fullscreen target element (platform specific)
    const getFullscreenTarget = () => {
      const fullscreenElement = document.fullscreenElement || document.webkitFullscreenElement || document.mozFullScreenElement || document.msFullscreenElement;
      if (fullscreenElement) return fullscreenElement;
      
      const host = window.location.hostname;
      
      // HBO Max / Max specific
      if (host.includes('max.com') || host.includes('hbomax.com')) {
        return document.querySelector('[data-testid="player"]') ||
               document.querySelector('.default-player') ||
               document.querySelector('.default-player-wrapper') ||
               document.querySelector('[class*="PlayerContainer"]') ||
               document.querySelector('[class*="FullPlayer"]');
      }
      
      // Netflix
      if (host.includes('netflix.com')) {
        return document.querySelector('.watch-video') ||
               document.querySelector('.player-video-wrapper') ||
               document.querySelector('[data-uia="video-canvas"]');
      }
      
      // YouTube
      if (host.includes('youtube.com')) {
        return document.querySelector('#movie_player') ||
               document.querySelector('.html5-video-player') ||
               document.querySelector('ytd-player');
      }
      
      // Disney+
      if (host.includes('disneyplus.com')) {
        return document.querySelector('.btm-media-client-element') ||
               document.querySelector('[data-testid="media-container"]') ||
               document.querySelector('.dss-media-player');
      }
      
      // Prime Video
      if (host.includes('primevideo.com') || host.includes('amazon.com')) {
        return document.querySelector('.webPlayerElement') ||
               document.querySelector('.cascadesContainer') ||
               document.querySelector('[data-testid="video-player"]') ||
               document.querySelector('.atvwebplayersdk-overlays-wrapper');
      }
      
      // Hulu
      if (host.includes('hulu.com')) {
        return document.querySelector('.Player') ||
               document.querySelector('.VideoPlayer') ||
               document.querySelector('[data-automationid="video-player"]');
      }
      
      // Peacock
      if (host.includes('peacocktv.com')) {
        return document.querySelector('.video-player') ||
               document.querySelector('[data-testid="video-container"]');
      }
      
      // Paramount+
      if (host.includes('paramountplus.com')) {
        return document.querySelector('.video-js') ||
               document.querySelector('[data-testid="video-player"]');
      }
      
      // Apple TV+
      if (host.includes('tv.apple.com')) {
        return document.querySelector('.video-container') ||
               document.querySelector('[data-testid="video"]');
      }
      
      // Crunchyroll
      if (host.includes('crunchyroll.com')) {
        return document.querySelector('#player') ||
               document.querySelector('.vilos-player');
      }
      
      // Twitch
      if (host.includes('twitch.tv')) {
        return document.querySelector('.video-player') ||
               document.querySelector('.video-ref');
      }
      
      return null;
    };
    
    // Lightweight fullscreen handler
    const ensureUIVisible = () => {
      const container = document.getElementById('cowatch-container');
      const toggle = document.getElementById('cowatch-toggle');
      if (!container) return;
      
      // Get the actual fullscreen element
      const fullscreenElement = document.fullscreenElement || document.webkitFullscreenElement || document.mozFullScreenElement || document.msFullscreenElement;
      
      // Check if we're in fullscreen using multiple methods
      const isFullscreenAPI = !!fullscreenElement;
      
      // Also check video size for platforms that don't use standard fullscreen API
      const video = document.querySelector('video');
      const isFullscreenBySize = video && (() => {
        const rect = video.getBoundingClientRect();
        return rect.width >= window.innerWidth * 0.95 && rect.height >= window.innerHeight * 0.95;
      })();
      
      // Platform-specific fullscreen checks
      const host = window.location.hostname;
      let platformFullscreenCheck = false;
      
      // HBO Max / Max - Check multiple indicators
      if (host.includes('max.com') || host.includes('hbomax.com')) {
        // HBO uses various methods for fullscreen
        const playerContainer = document.querySelector('[data-testid="player"]') || 
                               document.querySelector('.default-player') ||
                               document.querySelector('[class*="PlayerContainer"]');
        const videoEl = document.querySelector('video');
        
        platformFullscreenCheck = !!(
          document.querySelector('[data-testid="player"]:fullscreen') ||
          document.querySelector('.default-player:fullscreen') ||
          document.body.classList.contains('is-fullscreen') ||
          document.documentElement.classList.contains('fullscreen') ||
          // Check if player container or video is near full viewport size
          (playerContainer && playerContainer.getBoundingClientRect().width >= window.innerWidth * 0.95) ||
          (videoEl && videoEl.getBoundingClientRect().width >= window.innerWidth * 0.95 && 
           videoEl.getBoundingClientRect().height >= window.innerHeight * 0.9)
        );
      }
      
      // Netflix
      if (host.includes('netflix.com')) {
        platformFullscreenCheck = !!(
          document.querySelector('.watch-video:fullscreen') ||
          document.body.classList.contains('fullscreen') ||
          document.querySelector('[data-uia="player"]')?.classList.contains('fullscreen')
        );
      }
      
      // YouTube
      if (host.includes('youtube.com')) {
        platformFullscreenCheck = !!(
          document.querySelector('.ytp-fullscreen') ||
          document.querySelector('#movie_player.ytp-fullscreen') ||
          document.querySelector('.html5-video-player.ytp-fullscreen')
        );
      }
      
      // Disney+
      if (host.includes('disneyplus.com')) {
        platformFullscreenCheck = !!(
          document.body.classList.contains('is-fullscreen') ||
          document.querySelector('[data-testid="media-container"]:fullscreen')
        );
      }
      
      // Prime Video
      if (host.includes('primevideo.com') || host.includes('amazon.com')) {
        platformFullscreenCheck = !!(
          document.body.classList.contains('dv-player-fullscreen') ||
          document.querySelector('.webPlayerElement:fullscreen') ||
          document.querySelector('.atvwebplayersdk-hideabletopbuttons-active')
        );
      }
      
      // Hulu
      if (host.includes('hulu.com')) {
        platformFullscreenCheck = !!(
          document.body.classList.contains('fullscreen') ||
          document.querySelector('.Player:fullscreen')
        );
      }
      
      // Twitch
      if (host.includes('twitch.tv')) {
        platformFullscreenCheck = !!(
          document.body.classList.contains('theatre-mode') ||
          document.body.classList.contains('fullscreen-mode') ||
          document.querySelector('.video-player--fullscreen')
        );
      }
      
      const isFullscreen = isFullscreenAPI || isFullscreenBySize || platformFullscreenCheck;
      const isPanelOpen = container.classList.contains('open');
      
      // State changed
      if (isFullscreen !== lastFullscreenState) {
        lastFullscreenState = isFullscreen;
        console.log('🎬 Fullscreen state changed:', isFullscreen, 'element:', fullscreenElement?.tagName);
        
        // CRITICAL FIX: When browser fullscreen API is used, elements OUTSIDE the fullscreen
        // element are HIDDEN by the browser automatically. We MUST move UI INTO the fullscreen element.
        
        if (isFullscreen && fullscreenElement) {
          // Move UI INTO the fullscreen element so it remains visible
          if (container.parentElement !== fullscreenElement) {
            fullscreenElement.appendChild(container);
            console.log('🎬 Moved container INTO fullscreen element');
          }
          if (toggle && toggle.parentElement !== fullscreenElement) {
            fullscreenElement.appendChild(toggle);
            console.log('🎬 Moved toggle INTO fullscreen element');
          }
          originalParent = document.body;
          isMovedToFullscreen = true;
        } else if (!isFullscreen && isMovedToFullscreen) {
          // Exiting fullscreen - move UI back to body
          document.body.appendChild(container);
          if (toggle) document.body.appendChild(toggle);
          originalParent = null;
          isMovedToFullscreen = false;
          
          // Remove class from any element that might have it
          document.querySelectorAll('.cowatch-chat-open').forEach(el => el.classList.remove('cowatch-chat-open'));
          
          // Reset video styles
          if (video) video.style.cssText = '';
          
          // Reset platform-specific video container styles
          this.resetVideoContainerStyles();
          
          console.log('🎬 Moved container back to body');
        }
      }
      
      if (isFullscreen) {
        // Add class to fullscreen element for CSS targeting
        const targetElement = getFullscreenTarget();
        if (targetElement) {
          if (isPanelOpen) {
            targetElement.classList.add('cowatch-chat-open');
          } else {
            targetElement.classList.remove('cowatch-chat-open');
          }
        }
        
        // IMPORTANT: We do NOT resize video in fullscreen mode
        // Fullscreen API does not allow resizing the fullscreen element
        // All major watch party extensions (Teleparty, Scener) use overlay approach
        // Chat panel will overlay on top of the video - this is the industry standard
        
        // In fullscreen - force maximum visibility
        container.style.cssText = `
          position: fixed !important;
          top: 0 !important;
          right: 0 !important;
          width: 340px !important;
          height: 100vh !important;
          height: 100% !important;
          max-height: 100vh !important;
          max-height: 100% !important;
          z-index: 9999999999 !important;
          display: flex !important;
          flex-direction: column !important;
          visibility: visible !important;
          opacity: ${isPanelOpen ? '1' : '0'} !important;
          pointer-events: ${isPanelOpen ? 'auto' : 'none'} !important;
          background: rgba(10, 10, 10, 0.98) !important;
          transform: ${isPanelOpen ? 'translateX(0)' : 'translateX(100%)'} !important;
          transition: transform 0.3s ease, opacity 0.3s ease !important;
        `;
        
        // Toggle button in fullscreen - Teleparty style (round button)
        if (toggle) {
          if (!isPanelOpen && this.roomCode) {
            toggle.style.cssText = `
              position: fixed !important;
              top: 100px !important;
              right: 30px !important;
              width: 50px !important;
              height: 50px !important;
              z-index: 9999999998 !important;
              display: flex !important;
              align-items: center !important;
              justify-content: center !important;
              visibility: visible !important;
              opacity: 1 !important;
              pointer-events: auto !important;
              background: linear-gradient(135deg, #4ade80 0%, #22c55e 100%) !important;
              border: none !important;
              border-radius: 50% !important;
              cursor: pointer !important;
              box-shadow: 0 4px 15px rgba(74, 222, 128, 0.4) !important;
            `;
          } else {
            toggle.style.cssText = 'display: none !important;';
          }
        }
      } else {
        // Not in fullscreen - reset inline styles if they were set
        if (container.style.cssText.includes('9999999999')) {
          container.style.cssText = '';
        }
        // Keep toggle visible with Teleparty style (round button at top right)
        if (toggle && this.roomCode && !this.chatOpen) {
          toggle.style.cssText = `
            position: fixed !important;
            top: 100px !important;
            right: 30px !important;
            width: 50px !important;
            height: 50px !important;
            z-index: 9999999998 !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            visibility: visible !important;
            opacity: 1 !important;
            pointer-events: auto !important;
            border-radius: 50% !important;
          `;
        }
        if (video && video.style.cssText.includes('calc')) {
          video.style.cssText = '';
        }
      }
    };
    
    // Check frequently
    setInterval(ensureUIVisible, 300);
    
    // Also respond to standard fullscreen events
    document.addEventListener('fullscreenchange', () => setTimeout(ensureUIVisible, 100));
    document.addEventListener('webkitfullscreenchange', () => setTimeout(ensureUIVisible, 100));
    document.addEventListener('mozfullscreenchange', () => setTimeout(ensureUIVisible, 100));
    document.addEventListener('MSFullscreenChange', () => setTimeout(ensureUIVisible, 100));
    
    // Platform-specific stabilization
    const host = window.location.hostname;
    
    // HBO needs special handling due to aggressive DOM manipulation
    if (host.includes('max.com') || host.includes('hbomax.com')) {
      this.setupHBOStabilization();
    }
    
    // Inject universal chat styles for all platforms
    this.injectChatOverlayStyles();
  }
  
  // Simple overlay approach - chat panel on right side, doesn't touch video
  // Teleparty approach: 340px width chat overlay, z-index:9999999999
  injectChatOverlayStyles() {
    if (document.getElementById('cowatch-overlay-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'cowatch-overlay-styles';
    style.textContent = `
      /* CoWatch Chat Overlay - Teleparty-style fixed panel */
      /* --chatWidth: 340px (same as Teleparty) */
      :root {
        --cowatch-chat-width: 340px;
      }
      
      #cowatch-container {
        position: fixed !important;
        top: 0 !important;
        right: 0 !important;
        width: var(--cowatch-chat-width) !important;
        height: 100vh !important;
        height: 100% !important;
        max-height: 100vh !important;
        z-index: 9999999999 !important;
        display: flex !important;
        flex-direction: column !important;
        visibility: visible !important;
        pointer-events: auto !important;
        background: rgba(15, 15, 15, 0.98) !important;
        border-left: 1px solid rgba(255,255,255,0.1) !important;
        box-shadow: -4px 0 20px rgba(0,0,0,0.5) !important;
        transition: transform 0.3s ease, opacity 0.3s ease !important;
      }
      
      #cowatch-container:not(.open) {
        transform: translateX(100%) !important;
        opacity: 0 !important;
        pointer-events: none !important;
      }
      
      #cowatch-container.open {
        transform: translateX(0) !important;
        opacity: 1 !important;
        pointer-events: auto !important;
      }
      
      /* Toggle button - Teleparty style positioning */
      /* Teleparty uses top:50px-100px, right:30px, width:50px */
      #cowatch-toggle {
        position: fixed !important;
        top: 100px !important;
        right: 30px !important;
        z-index: 9999999998 !important;
        width: 50px !important;
        height: 50px !important;
        border-radius: 50% !important;
        visibility: visible !important;
        pointer-events: auto !important;
        transition: all 0.2s ease !important;
      }
      
      #cowatch-toggle:hover {
        transform: scale(1.1) !important;
        box-shadow: 0 4px 20px rgba(74, 222, 128, 0.5) !important;
      }
      
      /* Party active controls - Teleparty style */
      #cowatch-party-controls {
        position: fixed !important;
        top: 100px !important;
        right: 30px !important;
        z-index: 9999999998 !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
        transition: opacity 0.3s ease !important;
      }
      
      /* Unread message badge - Teleparty style */
      .unread-badge {
        position: absolute !important;
        top: -4px !important;
        right: -4px !important;
        background: #ff0000 !important;
        color: white !important;
        border-radius: 50% !important;
        width: 20px !important;
        height: 20px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 11px !important;
        font-weight: bold !important;
        z-index: 10 !important;
        box-shadow: 0 0 4px rgba(0,0,0,0.3) !important;
        animation: pulse 2s infinite !important;
      }
      
      @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.1); }
      }
    `;
    document.head.appendChild(style);
  }
  
  // HBO-specific fix: HBO manipulates DOM aggressively
  // We just need to ensure our UI stays visible and on top
  setupHBOStabilization() {
    console.log('🎬 Setting up HBO-specific stabilization');
    
    const stabilizeHBO = () => {
      const toggle = document.getElementById('cowatch-toggle');
      const container = document.getElementById('cowatch-container');
      
      if (!toggle && !container) return;
      
      // Ensure elements stay in body (HBO might move them)
      if (toggle && toggle.parentElement !== document.body) {
        document.body.appendChild(toggle);
      }
      if (container && container.parentElement !== document.body) {
        document.body.appendChild(container);
      }
    };
    
    // Run stabilization frequently for HBO
    setInterval(stabilizeHBO, 500);
    
    // Watch for HBO's DOM changes
    const observer = new MutationObserver(() => {
      if (this._hboStabilizeTimeout) return;
      this._hboStabilizeTimeout = setTimeout(() => {
        this._hboStabilizeTimeout = null;
        stabilizeHBO();
      }, 100);
    });
    
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
    
    console.log('🎬 HBO stabilization active');
  }

  setupUnloadListener() {
    // Leave room when page is closed or navigated away
    // Use multiple events for better reliability
    const leaveOnUnload = () => {
      if (this.roomCode && this.socket) {
        console.log('🎬 Page unloading, leaving room...');
        // Send leave message synchronously before page closes
        try {
          this.socket.send(JSON.stringify({
            type: 'room:leave',
            roomCode: this.roomCode,
            username: this.username
          }));
        } catch (e) {
          console.error('🎬 Failed to send leave message:', e);
        }
        // Also use sendBeacon as fallback for reliability
        try {
          navigator.sendBeacon('https://cowatch.app/api/rooms/leave', JSON.stringify({
            roomCode: this.roomCode,
            username: this.username
          }));
        } catch (e) {}
      }
    };
    
    window.addEventListener('beforeunload', leaveOnUnload);
    window.addEventListener('pagehide', leaveOnUnload);
    window.addEventListener('unload', leaveOnUnload);
    
    // Handle URL changes (SPA navigation) - leave room when navigating to different video
    let currentUrl = window.location.href;
    const checkUrlChange = () => {
      if (window.location.href !== currentUrl) {
        const oldUrl = currentUrl;
        currentUrl = window.location.href;
        console.log('🎬 URL changed from', oldUrl, 'to', currentUrl);
        
        // If we're in a room and URL changed significantly (not just hash/params), leave room
        if (this.roomCode && this.socket) {
          // Check if the base path changed (indicates different video)
          const oldPath = new URL(oldUrl).pathname;
          const newPath = new URL(currentUrl).pathname;
          
          if (oldPath !== newPath) {
            console.log('🎬 Path changed, leaving room...');
            this.leaveRoom();
          }
        }
      }
    };
    
    // Check for URL changes periodically and on popstate (every 3 seconds is enough)
    setInterval(checkUrlChange, 3000);
    window.addEventListener('popstate', checkUrlChange);
    
    // Also listen for pushState/replaceState
    const originalPushState = history.pushState;
    const originalReplaceState = history.replaceState;
    
    history.pushState = function(...args) {
      originalPushState.apply(this, args);
      setTimeout(checkUrlChange, 100);
    };
    
    history.replaceState = function(...args) {
      originalReplaceState.apply(this, args);
      setTimeout(checkUrlChange, 100);
    };
    
    // Also handle page visibility changes with timeout
    let hiddenTimeout = null;
    document.addEventListener('visibilitychange', () => {
      if (document.visibilityState === 'hidden' && this.roomCode) {
        console.log('🎬 Page visibility hidden, starting disconnect timeout...');
        // Give 30 seconds before considering user gone (tab might just be backgrounded)
        hiddenTimeout = setTimeout(() => {
          if (document.visibilityState === 'hidden' && this.roomCode && this.socket) {
            console.log('🎬 Page still hidden after timeout, leaving room...');
            leaveOnUnload();
          }
        }, 30000);
      } else if (document.visibilityState === 'visible' && hiddenTimeout) {
        console.log('🎬 Page visible again, canceling disconnect timeout');
        clearTimeout(hiddenTimeout);
        hiddenTimeout = null;
      }
    });
  }

  async loadState() {
    // Load state from storage to persist room across page refreshes
    try {
      const result = await new Promise((resolve) => {
        // Timeout after 2 seconds to prevent hanging
        const timeout = setTimeout(() => {
          console.log('🎬 loadState timeout, using defaults');
          resolve({});
        }, 2000);
        
        if (storage && storage.get) {
          storage.get(['roomCode', 'isHost', 'username', 'userAvatar', 'theme', 'language', 'messages'], (data) => {
            clearTimeout(timeout);
            resolve(data || {});
          });
        } else {
          clearTimeout(timeout);
          resolve({});
        }
      });
      
      if (result?.roomCode) {
        this.roomCode = result.roomCode;
        console.log('🎬 Restored room code:', this.roomCode);
      }
      if (result?.isHost !== undefined) this.isHost = result.isHost;
      if (result?.username) this.username = result.username;
      if (result?.userAvatar) {
        this.userAvatar = result.userAvatar;
      } else {
        // Assign random avatar if none exists
        this.userAvatar = AVATARS[Math.floor(Math.random() * AVATARS.length)];
      }
      if (result?.theme) {
        this.theme = result.theme;
      }
      if (result?.language) {
        this.language = result.language;
      } else {
        // Check localStorage for frontend sync
        try {
          const frontendLang = localStorage.getItem('cowatch-language');
          if (frontendLang && CHAT_TRANSLATIONS[frontendLang]) {
            this.language = frontendLang;
          }
        } catch (e) {}
      }
      if (result?.chatOpen !== undefined) {
        this.chatOpen = result.chatOpen;
      }
      // Restore last 50 messages
      if (result?.messages && Array.isArray(result.messages)) {
        this.messages = result.messages.slice(-50);
      }
    } catch (e) {
      console.log('🎬 loadState skipped:', e.message);
    }
  }

  saveState() {
    storage.set({
      roomCode: this.roomCode,
      isHost: this.isHost,
      username: this.username,
      userAvatar: this.userAvatar,
      theme: this.theme,
      language: this.language,
      chatOpen: this.chatOpen,
      messages: this.messages.slice(-50) // Keep last 50 messages
    });
  }

  clearState() {
    this.roomCode = null;
    this.isHost = false;
    this.messages = [];
    this.participants = [];
    storage.remove(['roomCode', 'isHost', 'messages']);
    // Keep username, avatar and theme for next session
  }

  findVideo(retryCount = 0) {
    // Platform-specific video selectors - ordered by specificity
    const selectors = {
      // Netflix uses encrypted media extensions
      netflix: ['video[src*="nflxvideo"]', '.watch-video video', '.player-video-wrapper video', 'video'],
      // YouTube has multiple video types
      youtube: ['video.html5-main-video', '.html5-video-player video', '#movie_player video', 'ytd-player video', 'video'],
      // Disney+ uses Bitmovin player
      disney: ['.btm-media-client-element video', '[data-testid="media-container"] video', '.dss-media-player video', 'video'],
      // Prime Video uses Shaka player
      prime: ['.webPlayerElement video', '.cascadesContainer video', '[data-testid="video-player"] video', '.atvwebplayersdk-overlays-wrapper video', 'video'],
      // HBO Max / Max uses custom player
      hbo: ['[data-testid="player"] video', '.default-player video', '.default-player-wrapper video', '[class*="VideoPlayer"] video', 'video'],
      // Hulu uses custom player
      hulu: ['.Player video', '.VideoPlayer video', '[data-automationid="video-player"] video', 'video'],
      // Peacock uses custom player
      peacock: ['.video-player video', '[data-testid="video-container"] video', 'video'],
      // Paramount+ uses Video.js
      paramount: ['.vjs-tech', '.video-js video', '[data-testid="video-player"] video', 'video'],
      // Apple TV+ uses custom player
      appletv: ['.video-container video', '[data-testid="video"] video', 'video'],
      // Crunchyroll uses Vilos player
      crunchyroll: ['#player video', '.vilos-player video', '[data-testid="vilos-player"] video', 'video'],
      // Mubi uses custom player
      mubi: ['.player-container video', '.mubi-player video', 'video'],
      // Live platforms
      twitch: ['video[playsinline]', '.video-ref video', 'video'],
      espn: ['.video-player video', '.plyr video', 'video'],
      fubo: ['.player-container video', 'video'],
      // Free streaming
      tubi: ['.bmpui-ui-container video', '[data-testid="video-player"] video', 'video'],
      pluto: ['.video-player video', '.player-container video', 'video'],
      // Video hosting
      vimeo: ['.vp-video video', '.player video', 'video'],
      dailymotion: ['.dmp_Player video', '.player video', 'video'],
      // Regional services
      hotstar: ['.player-wrapper video', '.bmpui-container video', 'video'],
      viki: ['.player-container video', '.viki-player video', 'video'],
      zee5: ['.player-wrapper video', 'video'],
      sonyliv: ['.video-player video', 'video'],
      jiocinema: ['.player-container video', 'video'],
      shahid: ['.shaka-video-container video', 'video'],
      viu: ['.vjs-tech', 'video'],
      vidio: ['.bitmovinplayer-container video', 'video'],
      stan: ['.bitmovinplayer-container video', 'video'],
      crave: ['.video-player video', 'video'],
      // Fallback
      other: ['video']
    };

    const platformSelectors = selectors[this.platform.name] || selectors.other;
    
    for (const selector of platformSelectors) {
      try {
        const videos = document.querySelectorAll(selector);
        for (const video of videos) {
          // Accept video if it has src, currentSrc (blob URLs), is ready to play, or has reasonable dimensions
          const hasSource = video.src || video.currentSrc || video.readyState > 0;
          const hasSize = video.videoWidth > 100 || video.offsetWidth > 100;
          const isVisible = video.offsetParent !== null || getComputedStyle(video).display !== 'none';
          
          if (hasSource || (hasSize && isVisible)) {
            this.video = video;
            this.attachVideoListeners();
            console.log('🎬 CoWatch: Video element found', { 
              selector,
              src: !!video.src, 
              currentSrc: !!video.currentSrc, 
              readyState: video.readyState,
              dimensions: `${video.videoWidth}x${video.videoHeight}`
            });
            return true;
          }
        }
      } catch (e) {
        // Selector might be invalid, continue to next
      }
    }
    
    // Retry max 5 times (10 seconds total), then rely on observer
    if (retryCount < 5) {
      setTimeout(() => {
        if (!this.video) this.findVideo(retryCount + 1);
      }, 2000);
    } else {
      console.log('🎬 No video found after retries, relying on observer');
    }
    
    return false;
  }
  observeDOM() {
    // Debounce to prevent excessive calls on DOM-heavy sites
    let debounceTimer = null;
    let lastVideoCheck = 0;
    
    const observer = new MutationObserver(() => {
      // Only check for video every 2 seconds max
      const now = Date.now();
      if (now - lastVideoCheck > 2000) {
        if (!this.video || !document.contains(this.video)) {
          const hadVideo = !!this.video;
          this.video = null;
          this.findVideo();
          
          // If we just found a video and UI doesn't exist, create it
          if (this.video && !hadVideo && !document.getElementById('cowatch-container')) {
            console.log('🎬 Video found via observer, creating UI');
            this.createUI();
          }
        }
        lastVideoCheck = now;
      }
      
      // Debounce UI updates - don't run more than once per second
      if (debounceTimer) return;
      debounceTimer = setTimeout(() => {
        debounceTimer = null;
        // Only update if panel is open - otherwise no point
        const container = document.getElementById('cowatch-container');
        if (container?.classList.contains('open')) {
          this.updateUI();
        }
      }, 1000);
    });
    
    // Only observe childList, not subtree - much lighter!
    observer.observe(document.body, { childList: true });
  }

  attachVideoListeners() {
    if (!this.video) return;

    const sendEvent = (type) => {
      if (this.ignoreVideoEvents || !this.socket) return;
      
      // Check if socket is connected (readyState 1 = OPEN)
      if (this.socket.readyState !== 1) {
        console.log('🎬 Socket not ready, skipping event:', type);
        return;
      }
      
      // If hostOnlyControl is enabled and user is not host, don't send events
      if (this.hostOnlyControl && !this.isHost) {
        console.log('🎬 Only host can control video playback');
        return;
      }
      
      this.socket.send(JSON.stringify({
        type,
        time: this.video.currentTime,
        username: this.username,
        timestamp: Date.now()
      }));
    };

    this.video.addEventListener('play', () => sendEvent('video:play'));
    this.video.addEventListener('pause', () => sendEvent('video:pause'));
    this.video.addEventListener('seeked', () => sendEvent('video:seek'));
    
    // Buffering events (Teleparty-style)
    this.video.addEventListener('waiting', () => {
      console.log('🎬 Video buffering...');
      this.sendBufferingStatus(true);
    });
    
    this.video.addEventListener('canplay', () => {
      console.log('🎬 Video can play');
      this.sendBufferingStatus(false);
    });
    
    this.video.addEventListener('playing', () => {
      this.sendBufferingStatus(false);
    });
  }

  injectStyles() {
    if (document.getElementById('cowatch-styles')) {
      console.log('🎬 Styles already injected');
      return;
    }

    console.log('🎬 Injecting styles...');
    const style = document.createElement('style');
    style.id = 'cowatch-styles';
    
    // Use a safe color fallback
    const platformColor = this.platform?.color || '#FF3B30';
    
    style.textContent = `
      @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');

      /* Main Container - Video Sidebar Style */
      #cowatch-container {
        position: fixed !important;
        top: 0 !important;
        right: 0 !important;
        width: 340px !important;
        height: 100vh !important;
        background: rgba(10, 10, 10, 0.98) !important;
        backdrop-filter: blur(20px) !important;
        -webkit-backdrop-filter: blur(20px) !important;
        border-left: 1px solid rgba(255,255,255,0.1) !important;
        z-index: 9999999999 !important;
        font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif !important;
        display: flex !important;
        flex-direction: column !important;
        transform: translateX(100%) !important;
        opacity: 0 !important;
        transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1), opacity 0.3s ease !important;
        box-shadow: -10px 0 40px rgba(0,0,0,0.5) !important;
        overflow: hidden !important;
      }

      #cowatch-container.open {
        transform: translateX(0) !important;
        opacity: 1 !important;
      }

      /* Panel open - adjust body margin for overlay space */
      body.cowatch-panel-open {
        overflow-x: hidden !important;
      }

      /* YouTube skip ad button - MUST keep it visible and clickable */
      .ytp-ad-skip-button-container,
      .ytp-ad-skip-button,
      .ytp-skip-ad-button,
      .ytp-ad-skip-button-modern,
      .ytp-ad-button,
      .ytp-ad-overlay-close-button,
      .ytp-ad-text,
      .video-ads .ytp-ad-skip-button-slot,
      .ytp-ad-player-overlay,
      .ytp-ad-player-overlay-skip-or-preview {
        width: auto !important;
        max-width: none !important;
        z-index: 2147483640 !important;
        pointer-events: auto !important;
        visibility: visible !important;
        opacity: 1 !important;
        display: flex !important;
      }

      /* Fullscreen mode - panel stays visible */
      :fullscreen #cowatch-container,
      :-webkit-full-screen #cowatch-container,
      :-moz-full-screen #cowatch-container,
      :-ms-fullscreen #cowatch-container,
      *:fullscreen #cowatch-container,
      *:-webkit-full-screen #cowatch-container {
        position: fixed !important;
        z-index: 9999999999 !important;
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
      }

      :fullscreen #cowatch-toggle,
      :-webkit-full-screen #cowatch-toggle,
      :-moz-full-screen #cowatch-toggle,
      :-ms-fullscreen #cowatch-toggle,
      *:fullscreen #cowatch-toggle,
      *:-webkit-full-screen #cowatch-toggle {
        position: fixed !important;
        z-index: 9999999998 !important;
        display: flex !important;
        visibility: visible !important;
      }

      /* When document is in fullscreen mode */
      html:fullscreen #cowatch-container,
      html:-webkit-full-screen #cowatch-container,
      body:fullscreen #cowatch-container,
      body:-webkit-full-screen #cowatch-container {
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
      }

      html:fullscreen #cowatch-toggle,
      html:-webkit-full-screen #cowatch-toggle,
      body:fullscreen #cowatch-toggle,
      body:-webkit-full-screen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
      }
      
      /* Fullscreen API fallback for HBO/Max */
      .fullscreen #cowatch-container,
      .full-screen #cowatch-container,
      [data-fullscreen="true"] #cowatch-container,
      .player-fullscreen #cowatch-container,
      .is-fullscreen #cowatch-container {
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999999 !important;
        position: fixed !important;
        top: 0 !important;
        right: 0 !important;
        height: 100vh !important;
      }
      
      .fullscreen #cowatch-toggle,
      .full-screen #cowatch-toggle,
      [data-fullscreen="true"] #cowatch-toggle,
      .player-fullscreen #cowatch-toggle,
      .is-fullscreen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999998 !important;
        position: fixed !important;
      }

      /* HBO Max/Max specific fullscreen - container MUST stay visible */
      [data-testid="player"]:fullscreen #cowatch-container,
      [data-testid="player"]:-webkit-full-screen #cowatch-container,
      .default-player:fullscreen #cowatch-container,
      .default-player:-webkit-full-screen #cowatch-container,
      .default-player-wrapper:fullscreen #cowatch-container,
      .default-player-wrapper:-webkit-full-screen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
        background: rgba(10, 10, 10, 0.98) !important;
      }
      
      /* HBO Max/Max fullscreen toggle visibility */
      [data-testid="player"]:fullscreen #cowatch-toggle,
      [data-testid="player"]:-webkit-full-screen #cowatch-toggle,
      .default-player:fullscreen #cowatch-toggle,
      .default-player:-webkit-full-screen #cowatch-toggle,
      .default-player-wrapper:fullscreen #cowatch-toggle,
      .default-player-wrapper:-webkit-full-screen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999998 !important;
        position: fixed !important;
      }
      
      /* HBO Max/Max alternative selectors - class-based */
      [class*="PlayerContainer"]:fullscreen #cowatch-container,
      [class*="FullPlayer"]:fullscreen #cowatch-container,
      [class*="VideoPlayer"]:fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
        background: rgba(10, 10, 10, 0.98) !important;
      }
      
      [class*="PlayerContainer"]:fullscreen #cowatch-toggle,
      [class*="FullPlayer"]:fullscreen #cowatch-toggle,
      [class*="VideoPlayer"]:fullscreen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999998 !important;
        position: fixed !important;
      }

      /* Netflix fullscreen */
      .watch-video:fullscreen #cowatch-container,
      .watch-video:-webkit-full-screen #cowatch-container,
      .NFPlayer:fullscreen #cowatch-container,
      .NFPlayer:-webkit-full-screen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      .watch-video:fullscreen #cowatch-toggle,
      .NFPlayer:fullscreen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999998 !important;
        position: fixed !important;
      }
      
      /* YouTube fullscreen */
      #movie_player.ytp-fullscreen #cowatch-container,
      .html5-video-player.ytp-fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        opacity: 1 !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      #movie_player.ytp-fullscreen #cowatch-toggle,
      .html5-video-player.ytp-fullscreen #cowatch-toggle {
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999998 !important;
        position: fixed !important;
      }
      
      /* Disney+ fullscreen */
      .btm-media-client-element:fullscreen #cowatch-container,
      .btm-media-client-element:-webkit-full-screen #cowatch-container,
      [data-testid="media-container"]:fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      /* Prime Video fullscreen */
      .webPlayerElement:fullscreen #cowatch-container,
      .cascadesContainer:fullscreen #cowatch-container,
      .atvwebplayersdk-overlays-wrapper:fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      /* Hulu fullscreen */
      .Player:fullscreen #cowatch-container,
      .VideoPlayer:fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      /* Twitch fullscreen */
      .video-player--fullscreen #cowatch-container,
      body.theatre-mode #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }
      
      /* Crunchyroll fullscreen */
      #player:fullscreen #cowatch-container,
      .vilos-player:fullscreen #cowatch-container {
        position: fixed !important;
        display: flex !important;
        visibility: visible !important;
        z-index: 9999999999 !important;
        right: 0 !important;
        top: 0 !important;
        height: 100vh !important;
        width: 340px !important;
      }

      /* ================ VIDEO RESIZE WHEN CHAT OPEN (Teleparty Style) ================ */
      /* Video shrinks to make room for chat panel instead of overlaying */
      /* Works in BOTH normal mode and fullscreen mode */
      
      /* NORMAL MODE - Body and HTML level adjustments */
      body.cowatch-chat-open,
      html.cowatch-chat-open {
        overflow-x: hidden !important;
      }
      
      /* YouTube Normal Mode - Theater and default mode */
      body.cowatch-chat-open ytd-watch-flexy {
        margin-right: 340px !important;
        transition: margin 0.3s ease !important;
      }
      
      body.cowatch-chat-open ytd-watch-flexy #primary {
        max-width: calc(100% - 340px) !important;
      }
      
      body.cowatch-chat-open #player-theater-container {
        margin-right: 340px !important;
        width: calc(100% - 340px) !important;
        transition: all 0.3s ease !important;
      }
      
      body.cowatch-chat-open #full-bleed-container {
        max-width: calc(100% - 340px) !important;
      }
      
      /* Netflix Normal Mode */
      body.cowatch-chat-open .watch-video {
        width: calc(100% - 340px) !important;
        margin-right: 340px !important;
        transition: all 0.3s ease !important;
      }
      
      /* Disney+ Normal Mode */
      body.cowatch-chat-open [data-testid="web-player"],
      body.cowatch-chat-open .dss-hls-player-container {
        width: calc(100% - 340px) !important;
        margin-right: 340px !important;
        transition: all 0.3s ease !important;
      }
      
      /* Prime Video Normal Mode */
      body.cowatch-chat-open [data-testid="video-player-wrapper"],
      body.cowatch-chat-open .webPlayerUIContainer {
        width: calc(100% - 340px) !important;
        margin-right: 340px !important;
        transition: all 0.3s ease !important;
      }
      
      /* HBO Max Normal Mode */
      body.cowatch-chat-open [data-testid="player"],
      body.cowatch-chat-open .default-player {
        width: calc(100% - 340px) !important;
        margin-right: 340px !important;
        transition: all 0.3s ease !important;
      }
      
      /* Generic video player normal mode */
      body.cowatch-chat-open [class*="player"]:not(#cowatch-container):not(#cowatch-toggle) {
        max-width: calc(100% - 340px) !important;
        transition: all 0.3s ease !important;
      }
      
      /* FULLSCREEN MODE - Video element adjustments */
      /* YouTube - resize video when chat is open in fullscreen */
      .cowatch-chat-open video,
      .cowatch-chat-open .html5-main-video,
      .cowatch-chat-open .video-stream {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
        margin-right: 340px !important;
      }
      
      /* YouTube player container */
      .cowatch-chat-open #movie_player,
      .cowatch-chat-open .html5-video-player,
      .cowatch-chat-open .html5-video-container {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* Netflix - resize video */
      .cowatch-chat-open .watch-video video,
      .cowatch-chat-open .NFPlayer video,
      .cowatch-chat-open .VideoContainer video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      .cowatch-chat-open .watch-video--player-view,
      .cowatch-chat-open .NFPlayer,
      .cowatch-chat-open .VideoContainer {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* Disney+ - resize video */
      .cowatch-chat-open .btm-media-client-element video,
      .cowatch-chat-open [data-testid="media-container"] video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* Prime Video - resize video */
      .cowatch-chat-open .webPlayerElement video,
      .cowatch-chat-open .cascadesContainer video,
      .cowatch-chat-open .atvwebplayersdk-overlays-wrapper video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      .cowatch-chat-open .webPlayerElement,
      .cowatch-chat-open .cascadesContainer,
      .cowatch-chat-open .atvwebplayersdk-overlays-wrapper {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* HBO Max/Max - resize video */
      .cowatch-chat-open [data-testid="player"] video,
      .cowatch-chat-open .default-player video,
      .cowatch-chat-open [class*="PlayerContainer"] video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* Hulu - resize video */
      .cowatch-chat-open .Player video,
      .cowatch-chat-open .VideoPlayer video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }
      
      /* Generic video resize for any platform */
      .cowatch-chat-open:fullscreen video,
      .cowatch-chat-open:-webkit-full-screen video {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
        object-fit: contain !important;
      }
      
      /* Ensure video controls stay visible and accessible */
      .cowatch-chat-open .ytp-chrome-bottom,
      .cowatch-chat-open .PlayerControlsNeo__all-controls,
      .cowatch-chat-open .watch-video--bottom-controls {
        width: calc(100% - 340px) !important;
        max-width: calc(100vw - 340px) !important;
      }

      /* Picture-in-Picture adjustments - keep toggle visible */
      body:has(video:picture-in-picture) #cowatch-toggle {
        display: flex !important;
      }

      /* Toggle Button */
      #cowatch-toggle {
        position: fixed !important;
        top: 50% !important;
        right: 0 !important;
        transform: translateY(-50%) !important;
        width: 48px !important;
        height: 100px !important;
        background: linear-gradient(135deg, #ff3b30 0%, #ff6b5b 100%) !important;
        border: none !important;
        border-radius: 12px 0 0 12px !important;
        cursor: pointer !important;
        z-index: 9999999998 !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        flex-direction: column !important;
        gap: 4px !important;
        transition: all 0.3s ease !important;
        box-shadow: -4px 0 20px rgba(255, 59, 48, 0.4) !important;
      }

      #cowatch-toggle:hover {
        width: 56px !important;
        box-shadow: -6px 0 30px rgba(255, 59, 48, 0.6) !important;
      }

      #cowatch-toggle.connected {
        background: linear-gradient(135deg, #4ade80 0%, #22c55e 100%) !important;
        box-shadow: -4px 0 20px rgba(74, 222, 128, 0.4) !important;
      }

      #cowatch-toggle.connected:hover {
        box-shadow: -6px 0 30px rgba(74, 222, 128, 0.6) !important;
      }

      #cowatch-toggle.hidden {
        transform: translateY(-50%) translateX(100%) !important;
      }

      #cowatch-toggle svg {
        width: 28px !important;
        height: 28px !important;
      }

      #cowatch-toggle .badge {
        position: absolute !important;
        bottom: 6px !important;
        right: 6px !important;
        background: #fff !important;
        color: #000 !important;
        font-size: 10px !important;
        font-weight: 700 !important;
        padding: 2px 5px !important;
        border-radius: 8px !important;
        min-width: 14px !important;
        height: 14px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      /* Header */
      .cowatch-header {
        padding: 12px 16px !important;
        background: rgba(20, 20, 20, 0.95) !important;
        border-bottom: 1px solid rgba(255,255,255,0.1) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
      }

      .cowatch-header-left {
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
      }

      .cowatch-logo {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        color: #fff !important;
        font-size: 15px !important;
        font-weight: 700 !important;
      }

      .cowatch-logo svg {
        color: #ff3b30 !important;
      }

      .cowatch-avatar-mini {
        width: 22px !important;
        height: 22px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 12px !important;
        border: 2px solid rgba(255,255,255,0.2) !important;
      }

      .cowatch-profile-btn-small {
        padding: 4px !important;
      }

      .cowatch-header h1 {
        color: #fff !important;
        font-size: 16px !important;
        font-weight: 700 !important;
        margin: 0 !important;
      }

      .cowatch-header-subtitle {
        color: rgba(255,255,255,0.5) !important;
        font-size: 11px !important;
        margin-top: 0 !important;
      }

      .cowatch-header-badge {
        background: rgba(255,255,255,0.1) !important;
        color: rgba(255,255,255,0.8) !important;
        font-size: 12px !important;
        font-weight: 600 !important;
        padding: 4px 10px !important;
        border-radius: 12px !important;
      }

      .cowatch-back-btn {
        width: 32px !important;
        height: 32px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 8px !important;
        color: rgba(255,255,255,0.7) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
      }

      .cowatch-back-btn:hover {
        background: rgba(255,255,255,0.2) !important;
        color: #fff !important;
      }

      .cowatch-profile-btn {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        background: rgba(255,255,255,0.06) !important;
        border: none !important;
        border-radius: 20px !important;
        padding: 4px 12px 4px 4px !important;
        cursor: pointer !important;
        transition: all 0.2s !important;
      }

      .cowatch-profile-btn:hover {
        background: rgba(255,255,255,0.1) !important;
      }

      .cowatch-profile-avatar {
        width: 28px !important;
        height: 28px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 14px !important;
      }

      .cowatch-profile-name {
        color: rgba(255,255,255,0.9) !important;
        font-size: 13px !important;
        font-weight: 500 !important;
        max-width: 80px !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      .cowatch-profile-btn svg {
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-participant-count {
        background: #ff3b30 !important;
        color: #fff !important;
        font-size: 9px !important;
        font-weight: 600 !important;
        padding: 1px 4px !important;
        border-radius: 8px !important;
        margin-left: 2px !important;
        min-width: 14px !important;
        text-align: center !important;
      }

      .cowatch-panel-content {
        padding: 16px !important;
        overflow-y: auto !important;
        flex: 1 !important;
      }

      .cowatch-panel-content::-webkit-scrollbar {
        width: 6px !important;
      }

      .cowatch-panel-content::-webkit-scrollbar-track {
        background: transparent !important;
      }

      .cowatch-panel-content::-webkit-scrollbar-thumb {
        background: rgba(255,255,255,0.2) !important;
        border-radius: 3px !important;
      }

      .cowatch-participants-list {
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
      }

      .cowatch-host-badge {
        color: #f59e0b !important;
      }

      .cowatch-profile-edit {
        display: flex !important;
        flex-direction: column !important;
        align-items: center !important;
        gap: 20px !important;
        padding: 24px 16px !important;
      }

      .cowatch-avatar-preview-large {
        width: 96px !important;
        height: 96px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 48px !important;
        box-shadow: 0 8px 32px rgba(0,0,0,0.4), 0 0 0 4px rgba(255,255,255,0.1) !important;
        position: relative !important;
        overflow: hidden !important;
      }

      .cowatch-avatar-preview-large.custom {
        background: transparent !important;
      }

      /* Custom Image Section */
      .cowatch-custom-image-section {
        display: flex !important;
        flex-direction: column !important;
        gap: 10px !important;
        width: 100% !important;
        padding: 12px !important;
        background: rgba(255,255,255,0.03) !important;
        border-radius: 12px !important;
        border: 1px solid rgba(255,255,255,0.06) !important;
      }

      .cowatch-image-size-control {
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
        color: rgba(255,255,255,0.6) !important;
        font-size: 12px !important;
      }

      .cowatch-image-size-control input[type="range"] {
        flex: 1 !important;
        -webkit-appearance: none !important;
        height: 4px !important;
        background: rgba(255,255,255,0.2) !important;
        border-radius: 2px !important;
        outline: none !important;
      }

      .cowatch-image-size-control input[type="range"]::-webkit-slider-thumb {
        -webkit-appearance: none !important;
        width: 16px !important;
        height: 16px !important;
        background: #ff3b30 !important;
        border-radius: 50% !important;
        cursor: pointer !important;
      }

      .cowatch-btn-text {
        background: none !important;
        border: none !important;
        color: rgba(255,255,255,0.5) !important;
        font-size: 12px !important;
        cursor: pointer !important;
        padding: 4px !important;
        text-decoration: underline !important;
        transition: color 0.15s !important;
      }

      .cowatch-btn-text:hover {
        color: #ff3b30 !important;
      }

      .cowatch-section-label {
        color: rgba(255,255,255,0.5) !important;
        font-size: 11px !important;
        font-weight: 600 !important;
        text-transform: uppercase !important;
        letter-spacing: 1px !important;
        align-self: flex-start !important;
        margin-top: 8px !important;
        margin-bottom: -8px !important;
      }

      .cowatch-avatar-grid {
        display: grid !important;
        grid-template-columns: repeat(4, 52px) !important;
        gap: 10px !important;
        width: 100% !important;
        padding: 4px !important;
        justify-content: center !important;
        justify-items: center !important;
      }

      .cowatch-close {
        width: 32px !important;
        height: 32px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 8px !important;
        color: rgba(255,255,255,0.7) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
      }

      .cowatch-close:hover {
        background: rgba(255,255,255,0.2) !important;
        color: #fff !important;
      }

      /* Reply Bar */
      .cowatch-reply-bar {
        display: flex !important;
        align-items: flex-start !important;
        gap: 10px !important;
        padding: 10px 14px !important;
        background: rgba(255,59,48,0.1) !important;
        border-radius: 12px !important;
        border-left: 3px solid #ff3b30 !important;
        font-size: 12px !important;
        color: rgba(255,255,255,0.7) !important;
      }

      .cowatch-reply-bar svg {
        color: #ff3b30 !important;
        flex-shrink: 0 !important;
        margin-top: 2px !important;
      }

      .cowatch-reply-info {
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 2px !important;
        overflow: hidden !important;
      }

      .cowatch-reply-to {
        font-weight: 500 !important;
        color: rgba(255,255,255,0.8) !important;
      }

      .cowatch-reply-text {
        color: rgba(255,255,255,0.5) !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
        font-size: 11px !important;
      }

      .cowatch-reply-bar button {
        background: none !important;
        border: none !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        padding: 4px !important;
        font-size: 14px !important;
        flex-shrink: 0 !important;
        transition: all 0.15s !important;
      }

      .cowatch-reply-bar button:hover {
        color: #fff !important;
      }

      .cowatch-reply-bar button:active {
        transform: scale(0.9) !important;
      }

      /* GIF Label Button */
      .cowatch-gif-label {
        font-size: 11px !important;
        font-weight: 700 !important;
        color: inherit !important;
        letter-spacing: 0.5px !important;
      }

      /* Content Sections */
      .cowatch-content {
        flex: 1 !important;
        overflow-y: auto !important;
        display: flex !important;
        flex-direction: column !important;
      }

      .cowatch-section {
        padding: 20px !important;
        border-bottom: 1px solid rgba(255,255,255,0.05) !important;
      }

      .cowatch-section-title {
        color: rgba(255,255,255,0.5) !important;
        font-size: 11px !important;
        font-weight: 600 !important;
        text-transform: uppercase !important;
        letter-spacing: 1px !important;
        margin-bottom: 12px !important;
      }

      /* Status Badge */
      .cowatch-status {
        display: inline-flex !important;
        align-items: center !important;
        gap: 8px !important;
        padding: 10px 16px !important;
        background: rgba(255,255,255,0.05) !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 12px !important;
        color: #fff !important;
        font-size: 13px !important;
        margin-bottom: 16px !important;
      }

      .cowatch-status.connected {
        background: rgba(74, 222, 128, 0.1) !important;
        border-color: rgba(74, 222, 128, 0.3) !important;
        color: #4ade80 !important;
      }

      .cowatch-status-dot {
        width: 8px !important;
        height: 8px !important;
        border-radius: 50% !important;
        background: currentColor !important;
        animation: pulse 2s infinite !important;
      }

      @keyframes pulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.5; }
      }

      /* ========== SYNC STATUS OVERLAY ========== */
      .cowatch-sync-overlay {
        position: fixed !important;
        bottom: 80px !important;
        left: 50% !important;
        transform: translateX(-50%) !important;
        background: rgba(10, 10, 10, 0.95) !important;
        border: 1px solid rgba(255, 255, 255, 0.15) !important;
        border-radius: 12px !important;
        padding: 12px 20px !important;
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        z-index: 2147483645 !important;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.05) !important;
        backdrop-filter: blur(20px) !important;
        -webkit-backdrop-filter: blur(20px) !important;
        animation: syncSlideUp 0.3s ease-out !important;
        pointer-events: none !important;
      }

      .cowatch-sync-overlay.hiding {
        animation: syncSlideDown 0.3s ease-in forwards !important;
      }

      @keyframes syncSlideUp {
        from {
          opacity: 0;
          transform: translateX(-50%) translateY(20px);
        }
        to {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
      }

      @keyframes syncSlideDown {
        from {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
        to {
          opacity: 0;
          transform: translateX(-50%) translateY(20px);
        }
      }

      .cowatch-sync-spinner {
        width: 20px !important;
        height: 20px !important;
        border: 2px solid rgba(255, 59, 48, 0.3) !important;
        border-top-color: #ff3b30 !important;
        border-radius: 50% !important;
        animation: syncSpin 0.8s linear infinite !important;
      }

      @keyframes syncSpin {
        to { transform: rotate(360deg); }
      }

      .cowatch-sync-text {
        color: #fff !important;
        font-size: 13px !important;
        font-weight: 500 !important;
      }

      .cowatch-sync-success {
        color: #4ade80 !important;
      }

      .cowatch-sync-success .cowatch-sync-spinner {
        border-color: rgba(74, 222, 128, 0.3) !important;
        border-top-color: #4ade80 !important;
        animation: none !important;
        background: #4ade80 !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .cowatch-sync-success .cowatch-sync-spinner::after {
        content: '✓' !important;
        color: #000 !important;
        font-size: 12px !important;
        font-weight: bold !important;
      }

      /* ========== MESSAGE DIALOG (In-Page Notifications) ========== */
      .cowatch-message-dialog {
        position: fixed !important;
        top: 80px !important;
        left: 50% !important;
        transform: translateX(-50%) !important;
        background: rgba(10, 10, 10, 0.95) !important;
        border: 1px solid rgba(255, 255, 255, 0.15) !important;
        border-radius: 14px !important;
        padding: 14px 20px !important;
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        z-index: 2147483645 !important;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.05) !important;
        backdrop-filter: blur(20px) !important;
        -webkit-backdrop-filter: blur(20px) !important;
        animation: dialogSlideDown 0.3s ease-out !important;
        max-width: 400px !important;
      }

      .cowatch-message-dialog.hiding {
        animation: dialogSlideUp 0.3s ease-in forwards !important;
      }

      @keyframes dialogSlideDown {
        from {
          opacity: 0;
          transform: translateX(-50%) translateY(-20px);
        }
        to {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
      }

      @keyframes dialogSlideUp {
        from {
          opacity: 1;
          transform: translateX(-50%) translateY(0);
        }
        to {
          opacity: 0;
          transform: translateX(-50%) translateY(-20px);
        }
      }

      .cowatch-message-dialog-avatar {
        width: 36px !important;
        height: 36px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 18px !important;
        flex-shrink: 0 !important;
      }

      .cowatch-message-dialog-content {
        display: flex !important;
        flex-direction: column !important;
        gap: 2px !important;
        flex: 1 !important;
        min-width: 0 !important;
      }

      .cowatch-message-dialog-title {
        color: #fff !important;
        font-size: 13px !important;
        font-weight: 600 !important;
      }

      .cowatch-message-dialog-text {
        color: rgba(255, 255, 255, 0.7) !important;
        font-size: 12px !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      .cowatch-message-dialog-close {
        width: 24px !important;
        height: 24px !important;
        background: rgba(255, 255, 255, 0.1) !important;
        border: none !important;
        border-radius: 50% !important;
        color: rgba(255, 255, 255, 0.6) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
        pointer-events: auto !important;
      }

      .cowatch-message-dialog-close:hover {
        background: rgba(255, 255, 255, 0.2) !important;
        color: #fff !important;
      }

      /* Message dialog types */
      .cowatch-message-dialog.join {
        border-left: 3px solid #4ade80 !important;
      }

      .cowatch-message-dialog.leave {
        border-left: 3px solid #f87171 !important;
      }

      .cowatch-message-dialog.sync {
        border-left: 3px solid #60a5fa !important;
      }

      .cowatch-message-dialog.chat {
        border-left: 3px solid #ff3b30 !important;
      }

      /* ========== DRAGGABLE/RESIZABLE PANEL ========== */
      .cowatch-drag-handle {
        position: absolute !important;
        top: 0 !important;
        left: 0 !important;
        right: 0 !important;
        height: 12px !important;
        cursor: move !important;
        z-index: 10 !important;
      }

      .cowatch-drag-handle::before {
        content: '' !important;
        position: absolute !important;
        top: 4px !important;
        left: 50% !important;
        transform: translateX(-50%) !important;
        width: 40px !important;
        height: 4px !important;
        background: rgba(255, 255, 255, 0.2) !important;
        border-radius: 2px !important;
        opacity: 0 !important;
        transition: opacity 0.2s !important;
      }

      #cowatch-container:hover .cowatch-drag-handle::before {
        opacity: 1 !important;
      }

      .cowatch-resize-handle {
        position: absolute !important;
        width: 12px !important;
        height: 12px !important;
        z-index: 10 !important;
      }

      .cowatch-resize-handle-left {
        left: 0 !important;
        top: 50% !important;
        transform: translateY(-50%) !important;
        cursor: ew-resize !important;
        height: 50px !important;
        width: 8px !important;
      }

      .cowatch-resize-handle-left::before {
        content: '' !important;
        position: absolute !important;
        left: 2px !important;
        top: 50% !important;
        transform: translateY(-50%) !important;
        width: 3px !important;
        height: 30px !important;
        background: rgba(255, 255, 255, 0.15) !important;
        border-radius: 2px !important;
        opacity: 0 !important;
        transition: opacity 0.2s !important;
      }

      #cowatch-container:hover .cowatch-resize-handle-left::before {
        opacity: 1 !important;
      }

      #cowatch-container.dragging,
      #cowatch-container.resizing {
        transition: none !important;
        user-select: none !important;
      }

      #cowatch-container.dragging * {
        pointer-events: none !important;
      }

      /* Room Code Display */
      .cowatch-room-code {
        text-align: center !important;
        padding: 24px !important;
        background: rgba(255, 59, 48, 0.1) !important;
        border: 2px dashed rgba(255, 59, 48, 0.3) !important;
        border-radius: 16px !important;
        margin-bottom: 16px !important;
      }

      .cowatch-room-code-label {
        color: rgba(255,255,255,0.6) !important;
        font-size: 12px !important;
        margin-bottom: 8px !important;
      }

      .cowatch-room-code-value {
        font-size: 36px !important;
        font-weight: 800 !important;
        letter-spacing: 8px !important;
        background: linear-gradient(135deg, #ff3b30, #ff6b5b) !important;
        -webkit-background-clip: text !important;
        -webkit-text-fill-color: transparent !important;
        background-clip: text !important;
      }

      /* Participants List */
      .cowatch-participants {
        display: flex !important;
        flex-wrap: wrap !important;
        gap: 8px !important;
      }

      .cowatch-participant {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        padding: 8px 12px !important;
        background: rgba(255,255,255,0.05) !important;
        border-radius: 20px !important;
        font-size: 13px !important;
        color: #fff !important;
      }

      .cowatch-participant-avatar {
        width: 28px !important;
        height: 28px !important;
        border-radius: 50% !important;
        background: linear-gradient(135deg, #ff3b30, #ff6b5b) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 11px !important;
        font-weight: 600 !important;
      }

      .cowatch-participant.host .cowatch-participant-avatar {
        background: linear-gradient(135deg, #fbbf24, #f59e0b) !important;
      }

      .cowatch-participant-name {
        max-width: 100px !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      /* Compact Section */
      .cowatch-section-compact {
        padding: 12px 16px !important;
        background: rgba(255,255,255,0.02) !important;
        border-bottom: 1px solid rgba(255,255,255,0.06) !important;
      }

      /* Room Info Row - Professional Design */
      .cowatch-room-info {
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        padding: 10px 14px !important;
        background: rgba(255,255,255,0.04) !important;
        border-radius: 10px !important;
        border: 1px solid rgba(255,255,255,0.06) !important;
      }

      .cowatch-room-code-inline {
        font-size: 14px !important;
        font-weight: 600 !important;
        letter-spacing: 2px !important;
        color: rgba(255,255,255,0.9) !important;
        flex: 1 !important;
        font-family: 'SF Mono', 'Monaco', 'Consolas', monospace !important;
      }

      /* Icon Buttons - Minimal */
      .cowatch-btn-icon {
        width: 28px !important;
        height: 28px !important;
        background: transparent !important;
        border: none !important;
        border-radius: 6px !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
      }

      .cowatch-btn-icon:hover {
        background: rgba(255,255,255,0.1) !important;
        color: rgba(255,255,255,0.9) !important;
      }

      .cowatch-btn-icon-danger {
        color: rgba(239, 68, 68, 0.7) !important;
      }

      .cowatch-btn-icon-danger:hover {
        background: rgba(239, 68, 68, 0.15) !important;
        color: #ef4444 !important;
      }

      /* Inline Participants - Hidden by default, shown via button */
      .cowatch-participants-inline {
        display: none !important;
      }

      .cowatch-participant-mini {
        width: 28px !important;
        height: 28px !important;
        border-radius: 50% !important;
        background: linear-gradient(135deg, #6366f1, #8b5cf6) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 10px !important;
        font-weight: 600 !important;
        color: #fff !important;
        cursor: default !important;
      }

      .cowatch-participant-mini.host {
        background: linear-gradient(135deg, #fbbf24, #f59e0b) !important;
      }

      /* Chat Area - Professional Design */
      .cowatch-chat {
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
        min-height: 0 !important;
        overflow: hidden !important;
        background: rgba(0,0,0,0.2) !important;
      }

      .cowatch-messages {
        flex: 1 !important;
        overflow-y: auto !important;
        padding: 8px !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 2px !important;
      }

      .cowatch-message {
        display: flex !important;
        gap: 8px !important;
        padding: 6px 12px !important;
        border-radius: 10px !important;
        transition: background 0.2s !important;
        position: relative !important;
        margin: 1px 0 !important;
      }

      .cowatch-message:hover {
        background: rgba(255,255,255,0.04) !important;
      }

      .cowatch-message:hover .cowatch-message-actions {
        opacity: 1 !important;
        visibility: visible !important;
      }

      .cowatch-message-avatar {
        width: 32px !important;
        height: 32px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 16px !important;
        flex-shrink: 0 !important;
        box-shadow: 0 2px 8px rgba(0,0,0,0.15) !important;
      }

      .cowatch-message-content {
        flex: 1 !important;
        min-width: 0 !important;
      }

      .cowatch-message-header {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        margin-bottom: 4px !important;
      }

      .cowatch-message-username {
        color: rgba(255,255,255,0.95) !important;
        font-weight: 600 !important;
        font-size: 13px !important;
      }

      .cowatch-message-time {
        color: rgba(255,255,255,0.35) !important;
        font-size: 11px !important;
      }

      .cowatch-message-text {
        color: rgba(255,255,255,0.85) !important;
        font-size: 14px !important;
        line-height: 1.5 !important;
        word-break: break-word !important;
        background: rgba(255,255,255,0.04) !important;
        padding: 10px 14px !important;
        border-radius: 0 16px 16px 16px !important;
        display: inline-block !important;
        max-width: 100% !important;
      }

      .cowatch-message-actions {
        position: absolute !important;
        top: 8px !important;
        right: 12px !important;
        opacity: 0 !important;
        visibility: hidden !important;
        transition: all 0.2s !important;
        display: flex !important;
        gap: 4px !important;
        background: rgba(28,28,30,0.95) !important;
        border-radius: 8px !important;
        padding: 4px !important;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3) !important;
      }

      .cowatch-message-action-btn {
        width: 28px !important;
        height: 28px !important;
        background: transparent !important;
        border: none !important;
        border-radius: 6px !important;
        color: rgba(255,255,255,0.6) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.15s !important;
      }

      .cowatch-message-action-btn:hover {
        background: rgba(255,255,255,0.1) !important;
        color: #fff !important;
      }

      .cowatch-message.system {
        justify-content: center !important;
        padding: 12px 16px !important;
        margin: 12px 0 !important;
      }

      .cowatch-message.system .cowatch-message-text {
        color: rgba(255,255,255,0.7) !important;
        font-size: 13px !important;
        font-style: normal !important;
        font-weight: 500 !important;
        text-align: center !important;
        background: rgba(255, 59, 48, 0.1) !important;
        padding: 10px 20px !important;
        border-radius: 20px !important;
        border: 1px solid rgba(255, 59, 48, 0.2) !important;
        display: inline-block !important;
        backdrop-filter: blur(8px) !important;
      }

      /* Chat Input - Professional */
      .cowatch-chat-input {
        padding: 16px !important;
        background: rgba(20,20,20,0.95) !important;
        border-top: 1px solid rgba(255,255,255,0.06) !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 10px !important;
      }

      /* Quick Reaction Bar - Teleparty Style */
      .cowatch-quick-reaction-bar {
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        gap: 2px !important;
        padding: 4px 8px !important;
        background: rgba(30,30,30,0.9) !important;
        border-radius: 16px !important;
        margin: 0 auto 4px auto !important;
      }

      .cowatch-quick-reaction {
        width: 28px !important;
        height: 28px !important;
        background: transparent !important;
        border: none !important;
        border-radius: 50% !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 16px !important;
        transition: all 0.15s ease !important;
        padding: 0 !important;
      }

      .cowatch-quick-reaction:hover {
        background: rgba(255,255,255,0.1) !important;
        transform: scale(1.2) !important;
      }

      .cowatch-quick-reaction:active {
        transform: scale(0.95) !important;
      }

      .cowatch-quick-reaction img {
        width: 24px !important;
        height: 24px !important;
        object-fit: contain !important;
      }

      /* Reconnection Status */
      .cowatch-reconnect-status,
      .cowatch-reconnect-failed {
        position: absolute !important;
        top: 0 !important;
        left: 0 !important;
        right: 0 !important;
        bottom: 0 !important;
        background: rgba(0,0,0,0.85) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        z-index: 10000 !important;
      }
      
      .cowatch-reconnect-content {
        display: flex !important;
        flex-direction: column !important;
        align-items: center !important;
        gap: 12px !important;
        color: #fff !important;
        font-size: 14px !important;
      }
      
      .cowatch-reconnect-spinner {
        width: 32px !important;
        height: 32px !important;
        border: 3px solid rgba(255,255,255,0.2) !important;
        border-top-color: #FF3B30 !important;
        border-radius: 50% !important;
        animation: cowatch-spin 1s linear infinite !important;
      }
      
      @keyframes cowatch-spin {
        to { transform: rotate(360deg); }
      }
      
      .cowatch-reconnect-failed button {
        background: #FF3B30 !important;
        color: #fff !important;
        border: none !important;
        padding: 8px 20px !important;
        border-radius: 6px !important;
        cursor: pointer !important;
        font-size: 14px !important;
        font-weight: 500 !important;
      }
      
      .cowatch-reconnect-failed button:hover {
        background: #E5342B !important;
      }

      /* Typing Indicator - Teleparty Style */
      .cowatch-typing-indicator {
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
        padding: 2px 10px !important;
        font-size: 11px !important;
        color: rgba(255,255,255,0.5) !important;
        margin-bottom: 2px !important;
      }

      .cowatch-typing-dots {
        display: flex !important;
        gap: 2px !important;
      }

      .cowatch-typing-dots span {
        width: 4px !important;
        height: 4px !important;
        background: rgba(255,255,255,0.5) !important;
        border-radius: 50% !important;
        animation: cowatch-typing-bounce 1.2s ease-in-out infinite !important;
      }

      .cowatch-typing-dots span:nth-child(1) {
        animation-delay: 0s !important;
      }

      .cowatch-typing-dots span:nth-child(2) {
        animation-delay: 0.2s !important;
      }

      .cowatch-typing-dots span:nth-child(3) {
        animation-delay: 0.4s !important;
      }

      @keyframes cowatch-typing-bounce {
        0%, 60%, 100% {
          transform: translateY(0);
        }
        30% {
          transform: translateY(-4px);
        }
      }

      .cowatch-typing-text {
        font-style: italic !important;
      }

      /* Buffering Indicator */
      .cowatch-buffering-indicator {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        padding: 4px 12px !important;
        font-size: 12px !important;
        color: rgba(255,200,100,0.8) !important;
        margin-bottom: 4px !important;
      }
      
      .cowatch-buffering-icon {
        animation: cowatch-pulse 1.5s ease-in-out infinite !important;
      }
      
      @keyframes cowatch-pulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.4; }
      }
      
      .cowatch-buffering-text {
        font-style: italic !important;
      }

      .cowatch-input-row {
        display: flex !important;
        align-items: center !important;
        gap: 4px !important;
        padding: 6px !important;
      }

      .cowatch-input-actions {
        display: flex !important;
        align-items: center !important;
        gap: 4px !important;
        flex-shrink: 0 !important;
      }

      .cowatch-input-wrapper {
        display: flex !important;
        align-items: center !important;
        gap: 4px !important;
        flex: 1 !important;
        background: rgba(255,255,255,0.06) !important;
        border: 1px solid rgba(255,255,255,0.08) !important;
        border-radius: 20px !important;
        padding: 2px 2px 2px 10px !important;
        transition: all 0.2s !important;
        min-width: 0 !important;
      }

      .cowatch-input-wrapper:focus-within {
        border-color: rgba(255,255,255,0.2) !important;
        background: rgba(255,255,255,0.08) !important;
      }

      .cowatch-chat-input input {
        flex: 1 !important;
        background: transparent !important;
        border: none !important;
        padding: 6px 0 !important;
        color: #fff !important;
        font-size: 13px !important;
        font-family: inherit !important;
        outline: none !important;
        min-width: 0 !important;
      }

      .cowatch-chat-input input::placeholder {
        color: rgba(255,255,255,0.4) !important;
      }

      /* Action Buttons in Input Row */
      .cowatch-input-action {
        width: 30px !important;
        height: 30px !important;
        background: rgba(255,255,255,0.06) !important;
        border: 1px solid rgba(255,255,255,0.08) !important;
        border-radius: 50% !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-input-action:hover {
        color: rgba(255,255,255,0.9) !important;
        background: rgba(255,255,255,0.12) !important;
      }

      .cowatch-input-action:active {
        transform: scale(0.92) !important;
      }

      .cowatch-input-action.active {
        color: #ff3b30 !important;
        border-color: rgba(255,59,48,0.3) !important;
      }

      .cowatch-input-action.recording {
        color: #ff3b30 !important;
        background: rgba(255,59,48,0.2) !important;
        animation: recordingPulse 1s ease-in-out infinite !important;
      }

      @keyframes recordingPulse {
        0%, 100% { box-shadow: 0 0 0 0 rgba(255,59,48,0.4); }
        50% { box-shadow: 0 0 0 8px rgba(255,59,48,0); }
      }

      /* Recording Bar - shown while recording */
      .cowatch-recording-bar {
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
        width: 100% !important;
        padding: 8px 12px !important;
        background: rgba(255,59,48,0.15) !important;
        border: 1px solid rgba(255,59,48,0.3) !important;
        border-radius: 24px !important;
      }

      .cowatch-recording-indicator {
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
      }

      .cowatch-recording-dot {
        width: 10px !important;
        height: 10px !important;
        background: #ff3b30 !important;
        border-radius: 50% !important;
        animation: recordingBlink 1s ease-in-out infinite !important;
      }

      @keyframes recordingBlink {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.3; }
      }

      .cowatch-recording-time {
        color: #fff !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        font-variant-numeric: tabular-nums !important;
      }

      .cowatch-recording-stop {
        width: 40px !important;
        height: 40px !important;
        background: #ff3b30 !important;
        border: none !important;
        border-radius: 50% !important;
        color: #fff !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
      }

      .cowatch-recording-stop:hover {
        background: #ff5545 !important;
        transform: scale(1.05) !important;
      }

      /* Recorded Audio Bar - shown after recording, before sending */
      .cowatch-recorded-audio {
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
        width: 100% !important;
        padding: 8px 12px !important;
        background: rgba(74,222,128,0.15) !important;
        border: 1px solid rgba(74,222,128,0.3) !important;
        border-radius: 24px !important;
        gap: 12px !important;
      }

      .cowatch-recorded-cancel {
        width: 36px !important;
        height: 36px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 50% !important;
        color: rgba(255,255,255,0.7) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-recorded-cancel:hover {
        background: rgba(255,59,48,0.3) !important;
        color: #ff3b30 !important;
      }

      .cowatch-recorded-info {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        flex: 1 !important;
        color: #4ade80 !important;
      }

      .cowatch-recorded-duration {
        font-size: 14px !important;
        font-weight: 600 !important;
        color: #fff !important;
      }

      .cowatch-recorded-send {
        width: 40px !important;
        height: 40px !important;
        background: #4ade80 !important;
        border: none !important;
        border-radius: 50% !important;
        color: #000 !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-recorded-send:hover {
        background: #22c55e !important;
        transform: scale(1.05) !important;
      }

      /* Send Button */
      .cowatch-send-btn {
        width: 36px !important;
        height: 36px !important;
        background: #ff3b30 !important;
        border: none !important;
        border-radius: 50% !important;
        color: #fff !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-send-btn:hover {
        background: #ff5545 !important;
        transform: scale(1.05) !important;
      }

      .cowatch-send-btn:active {
        transform: scale(0.92) !important;
      }

      .cowatch-send-btn svg {
        margin: 0 !important;
      }

      .cowatch-chat-input button:hover {
        transform: scale(1.05) !important;
      }

      /* Buttons */
      .cowatch-btn {
        width: 100% !important;
        padding: 14px 20px !important;
        border: none !important;
        border-radius: 12px !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        font-family: inherit !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        gap: 10px !important;
        transition: all 0.2s !important;
        margin-bottom: 10px !important;
      }

      .cowatch-btn:active {
        transform: scale(0.96) !important;
      }

      .cowatch-btn-primary {
        background: linear-gradient(135deg, #ff3b30, #ff6b5b) !important;
        color: #fff !important;
        box-shadow: 0 4px 20px rgba(255, 59, 48, 0.3) !important;
      }

      .cowatch-btn-primary:hover {
        transform: translateY(-2px) !important;
        box-shadow: 0 6px 30px rgba(255, 59, 48, 0.4) !important;
      }

      .cowatch-btn-primary:active {
        transform: translateY(0) scale(0.98) !important;
      }

      .cowatch-btn-secondary {
        background: rgba(255,255,255,0.1) !important;
        border: 1px solid rgba(255,255,255,0.2) !important;
        color: #fff !important;
      }

      .cowatch-btn-secondary:hover {
        background: rgba(255,255,255,0.15) !important;
      }

      .cowatch-btn-secondary:active {
        transform: scale(0.98) !important;
      }

      .cowatch-btn-danger {
        background: rgba(239, 68, 68, 0.2) !important;
        border: 1px solid rgba(239, 68, 68, 0.3) !important;
        color: #ef4444 !important;
      }

      .cowatch-btn-danger:hover {
        background: rgba(239, 68, 68, 0.3) !important;
      }

      .cowatch-btn-danger:active {
        transform: scale(0.98) !important;
      }

      .cowatch-btn-copy {
        background: rgba(74, 222, 128, 0.2) !important;
        border: 1px solid rgba(74, 222, 128, 0.3) !important;
        color: #4ade80 !important;
      }

      .cowatch-btn-copy:hover {
        background: rgba(74, 222, 128, 0.3) !important;
      }

      /* Input Fields */
      .cowatch-input {
        width: 100% !important;
        padding: 14px 16px !important;
        background: rgba(255,255,255,0.1) !important;
        border: 1px solid rgba(255,255,255,0.15) !important;
        border-radius: 12px !important;
        color: #fff !important;
        font-size: 14px !important;
        font-family: inherit !important;
        outline: none !important;
        margin-bottom: 12px !important;
        box-sizing: border-box !important;
        transition: all 0.2s !important;
      }

      .cowatch-input:focus {
        border-color: #ff3b30 !important;
        background: rgba(255,255,255,0.15) !important;
      }

      .cowatch-input::placeholder {
        color: rgba(255,255,255,0.4) !important;
      }

      /* Divider */
      .cowatch-divider {
        display: flex !important;
        align-items: center !important;
        gap: 16px !important;
        margin: 16px 0 !important;
        color: rgba(255,255,255,0.4) !important;
        font-size: 12px !important;
      }

      .cowatch-divider::before,
      .cowatch-divider::after {
        content: '' !important;
        flex: 1 !important;
        height: 1px !important;
        background: rgba(255,255,255,0.1) !important;
      }

      /* Empty State */
      .cowatch-empty {
        text-align: center !important;
        padding: 40px 20px !important;
        color: rgba(255,255,255,0.5) !important;
      }

      .cowatch-empty svg {
        width: 64px !important;
        height: 64px !important;
        margin-bottom: 16px !important;
        opacity: 0.5 !important;
      }

      .cowatch-empty h3 {
        color: #fff !important;
        font-size: 16px !important;
        margin-bottom: 8px !important;
      }

      .cowatch-empty p {
        font-size: 13px !important;
        line-height: 1.5 !important;
      }

      /* Platform Badge */
      .cowatch-platform {
        display: inline-flex !important;
        align-items: center !important;
        gap: 6px !important;
        padding: 6px 12px !important;
        background: ${this.platform.color}20 !important;
        border: 1px solid ${this.platform.color}40 !important;
        border-radius: 8px !important;
        color: ${this.platform.color} !important;
        font-size: 12px !important;
        font-weight: 500 !important;
      }

      /* Scrollbar */
      .cowatch-messages::-webkit-scrollbar {
        width: 6px;
      }

      .cowatch-messages::-webkit-scrollbar-track {
        background: transparent;
      }

      .cowatch-messages::-webkit-scrollbar-thumb {
        background: rgba(255,255,255,0.2);
        border-radius: 3px;
      }

      .cowatch-messages::-webkit-scrollbar-thumb:hover {
        background: rgba(255,255,255,0.3);
      }

      /* Animations */
      @keyframes slideIn {
        from {
          opacity: 0;
          transform: translateX(20px);
        }
        to {
          opacity: 1;
          transform: translateX(0);
        }
      }

      .cowatch-message {
        animation: slideIn 0.3s ease;
      }

      /* Floating Emoji Animation - Video Overlay */
      @keyframes floatUpEnhanced {
        0% {
          opacity: 0;
          transform: translateY(0) translateX(var(--drift, 0)) scale(0.5);
        }
        10% {
          opacity: 1;
          transform: translateY(-40px) translateX(calc(var(--drift, 0) * 0.8)) scale(1);
        }
        90% {
          opacity: 0.8;
          transform: translateY(-400px) translateX(calc(var(--drift, 0) * -0.3)) scale(1);
        }
        100% {
          opacity: 0;
          transform: translateY(-450px) translateX(var(--drift, 0)) scale(0.8);
        }
      }

      @keyframes emojiWiggle {
        0%, 100% { transform: rotate(0deg); }
        25% { transform: rotate(-12deg); }
        75% { transform: rotate(12deg); }
      }

      @keyframes pulseGlow {
        0%, 100% {
          filter: drop-shadow(0 0 15px rgba(255,255,255,0.4));
        }
        50% {
          filter: drop-shadow(0 0 35px rgba(255,255,255,0.8));
        }
      }

      .cowatch-floating-emoji {
        position: fixed !important;
        font-size: 36px !important;
        pointer-events: none !important;
        z-index: 9999999998 !important;
        animation: floatUpEnhanced 3s ease-out forwards !important;
        text-shadow: 0 2px 10px rgba(0,0,0,0.3) !important;
      }

      .cowatch-floating-emoji.wiggle {
        animation: floatUpEnhanced 3s ease-out forwards,
                   emojiWiggle 0.15s ease-in-out infinite !important;
      }

      /* Emoji Picker - Professional Overlay */
      .cowatch-emoji-picker {
        position: absolute !important;
        bottom: 70px !important;
        left: 16px !important;
        right: 16px !important;
        background: #1c1c1e !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 16px !important;
        padding: 12px !important;
        display: none !important;
        flex-direction: column !important;
        gap: 8px !important;
        max-height: 360px !important;
        overflow-y: auto !important;
        box-shadow: 0 8px 32px rgba(0,0,0,0.4) !important;
        opacity: 0 !important;
        transform: translateY(10px) scale(0.95) !important;
        transform-origin: bottom center !important;
        transition: opacity 0.25s ease, transform 0.25s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
      }

      .cowatch-emoji-picker.open {
        display: flex !important;
        opacity: 1 !important;
        transform: translateY(0) scale(1) !important;
      }

      .cowatch-reaction-section {
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
      }

      .cowatch-reaction-label {
        font-size: 11px !important;
        color: rgba(255,255,255,0.4) !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
        font-weight: 600 !important;
      }

      .cowatch-animated-reactions {
        display: grid !important;
        grid-template-columns: repeat(6, 48px) !important;
        gap: 6px !important;
        max-height: 220px !important;
        overflow-y: auto !important;
        padding: 4px 0 !important;
        justify-content: start !important;
      }

      .cowatch-animated-reactions::-webkit-scrollbar {
        width: 4px !important;
      }

      .cowatch-animated-reactions::-webkit-scrollbar-thumb {
        background: rgba(255,255,255,0.2) !important;
        border-radius: 2px !important;
      }

      .cowatch-animated-reaction {
        width: 48px !important;
        height: 48px !important;
        background: rgba(255,255,255,0.05) !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 10px !important;
        cursor: pointer !important;
        transition: all 0.2s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        padding: 4px !important;
        overflow: hidden !important;
        position: relative !important;
      }

      .cowatch-animated-reaction:hover {
        background: rgba(255, 59, 48, 0.2) !important;
        border-color: rgba(255, 59, 48, 0.5) !important;
        transform: scale(1.15) translateY(-2px) !important;
        box-shadow: 0 4px 12px rgba(255, 59, 48, 0.3) !important;
        z-index: 10 !important;
      }

      .cowatch-animated-reaction:active {
        transform: scale(0.95) !important;
      }

      .cowatch-animated-reaction img {
        width: 36px !important;
        height: 36px !important;
        object-fit: contain !important;
        border-radius: 4px !important;
      }

      .cowatch-emoji-divider {
        height: 1px !important;
        background: rgba(255,255,255,0.1) !important;
        margin: 4px 0 !important;
      }

      .cowatch-quick-emojis {
        display: grid !important;
        grid-template-columns: repeat(6, 48px) !important;
        gap: 6px !important;
        justify-content: start !important;
      }

      .cowatch-emoji-btn {
        width: 48px !important;
        height: 48px !important;
        background: rgba(255,255,255,0.03) !important;
        border: 1px solid rgba(255,255,255,0.08) !important;
        border-radius: 10px !important;
        font-size: 24px !important;
        cursor: pointer !important;
        transition: all 0.15s !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .cowatch-emoji-btn:hover {
        background: rgba(255,255,255,0.1) !important;
        transform: scale(1.2) !important;
      }

      /* Language Picker */
      .cowatch-language-picker {
        position: absolute !important;
        bottom: 70px !important;
        left: 16px !important;
        right: 16px !important;
        background: #1c1c1e !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 16px !important;
        display: none !important;
        flex-direction: column !important;
        box-shadow: 0 8px 32px rgba(0,0,0,0.4) !important;
        opacity: 0 !important;
        transform: translateY(10px) scale(0.95) !important;
        transform-origin: bottom center !important;
        transition: opacity 0.25s ease, transform 0.25s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
        overflow: hidden !important;
      }

      .cowatch-language-picker.open {
        display: flex !important;
        opacity: 1 !important;
        transform: translateY(0) scale(1) !important;
      }

      .cowatch-language-header {
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
        padding: 12px 16px !important;
        border-bottom: 1px solid rgba(255,255,255,0.1) !important;
        font-weight: 600 !important;
        font-size: 14px !important;
        color: #fff !important;
      }

      .cowatch-language-close {
        background: transparent !important;
        border: none !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        padding: 4px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: color 0.2s !important;
      }

      .cowatch-language-close:hover {
        color: #fff !important;
      }

      .cowatch-language-list {
        display: flex !important;
        flex-direction: column !important;
        max-height: 300px !important;
        overflow-y: auto !important;
      }

      .cowatch-language-item {
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        padding: 12px 16px !important;
        background: transparent !important;
        border: none !important;
        color: #fff !important;
        font-size: 14px !important;
        cursor: pointer !important;
        transition: background 0.2s !important;
        text-align: left !important;
      }

      .cowatch-language-item:hover {
        background: rgba(255,255,255,0.05) !important;
      }

      .cowatch-language-item.active {
        background: rgba(255, 59, 48, 0.15) !important;
      }

      .cowatch-language-item.active:hover {
        background: rgba(255, 59, 48, 0.2) !important;
      }

      .cowatch-language-flag {
        font-size: 20px !important;
      }

      .cowatch-language-name {
        flex: 1 !important;
      }

      .cowatch-language-item svg {
        color: #ff3b30 !important;
      }

      /* Floating Animated Reaction */
      .cowatch-floating-reaction {
        position: fixed !important;
        z-index: 9999999999 !important;
        pointer-events: none !important;
        animation: floatUpReaction 5s ease-out forwards !important;
      }

      .cowatch-floating-reaction img {
        width: 120px !important;
        height: 120px !important;
        filter: drop-shadow(0 4px 20px rgba(0,0,0,0.4)) !important;
      }

      @keyframes floatUpReaction {
        0% {
          opacity: 0;
          transform: translateY(0) translateX(var(--drift, 0)) scale(0.3);
        }
        10% {
          opacity: 1;
          transform: translateY(-50px) translateX(calc(var(--drift, 0) * 0.3)) scale(1.2);
        }
        30% {
          opacity: 1;
          transform: translateY(-150px) translateX(calc(var(--drift, 0) * 0.6)) scale(1.1);
        }
        70% {
          opacity: 0.9;
          transform: translateY(-400px) translateX(calc(var(--drift, 0) * 1.1)) scale(1);
        }
        100% {
          opacity: 0;
          transform: translateY(-600px) translateX(var(--drift, 0)) scale(0.8);
        }
      }

      /* Emoji Toggle Button */
      .cowatch-emoji-toggle {
        width: 38px !important;
        height: 38px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 10px !important;
        font-size: 18px !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-emoji-toggle:hover {
        background: rgba(255,255,255,0.2) !important;
      }

      /* Minimize Button */
      .cowatch-minimize {
        width: 32px !important;
        height: 32px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 8px !important;
        color: rgba(255,255,255,0.7) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        margin-right: 8px !important;
      }

      .cowatch-minimize:hover {
        background: rgba(255,255,255,0.2) !important;
        color: #fff !important;
      }

      /* Unread Badge on Toggle */
      #cowatch-toggle .unread-badge {
        position: absolute !important;
        top: 8px !important;
        right: 8px !important;
        background: #ff3b30 !important;
        color: #fff !important;
        font-size: 10px !important;
        font-weight: 700 !important;
        min-width: 16px !important;
        height: 16px !important;
        border-radius: 8px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        padding: 0 4px !important;
      }

      /* GIF Picker - Professional Design like Reference */
      .cowatch-gif-picker {
        position: absolute !important;
        bottom: 70px !important;
        left: 16px !important;
        right: 16px !important;
        background: #1c1c1e !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 16px !important;
        display: none !important;
        flex-direction: column !important;
        max-height: 350px !important;
        overflow: hidden !important;
        box-shadow: 0 8px 32px rgba(0,0,0,0.4) !important;
        opacity: 0 !important;
        transform: translateY(10px) scale(0.95) !important;
        transform-origin: bottom center !important;
        transition: opacity 0.25s ease, transform 0.25s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
      }

      .cowatch-gif-picker.open {
        display: flex !important;
        opacity: 1 !important;
        transform: translateY(0) scale(1) !important;
      }

      /* GIF Search Input */
      .cowatch-gif-search {
        padding: 12px !important;
        border-bottom: 1px solid rgba(255,255,255,0.08) !important;
      }

      .cowatch-gif-search input {
        width: 100% !important;
        padding: 10px 14px !important;
        background: rgba(255,255,255,0.08) !important;
        border: none !important;
        border-radius: 10px !important;
        color: #fff !important;
        font-size: 14px !important;
        outline: none !important;
      }

      .cowatch-gif-search input::placeholder {
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-gif-categories {
        display: flex !important;
        gap: 6px !important;
        padding: 10px 12px !important;
        overflow-x: auto !important;
        border-bottom: 1px solid rgba(255,255,255,0.06) !important;
        flex-shrink: 0 !important;
      }

      .cowatch-gif-categories::-webkit-scrollbar {
        height: 0 !important;
      }

      .cowatch-gif-category {
        padding: 6px 14px !important;
        background: rgba(255,255,255,0.08) !important;
        border: none !important;
        border-radius: 20px !important;
        color: rgba(255,255,255,0.7) !important;
        font-size: 12px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        white-space: nowrap !important;
        transition: all 0.2s !important;
      }

      .cowatch-gif-category:hover {
        background: rgba(255,255,255,0.12) !important;
        color: #fff !important;
      }

      .cowatch-gif-category.active {
        background: #ff3b30 !important;
        color: #fff !important;
      }

      .cowatch-gif-grid {
        display: grid !important;
        grid-template-columns: repeat(3, 1fr) !important;
        gap: 6px !important;
        padding: 10px !important;
        overflow-y: auto !important;
        max-height: 220px !important;
        flex: 1 !important;
      }

      .cowatch-gif-item {
        aspect-ratio: 1 !important;
        width: 100% !important;
        height: auto !important;
        border-radius: 8px !important;
        overflow: hidden !important;
        cursor: pointer !important;
        background: rgba(255,255,255,0.05) !important;
        transition: transform 0.2s, border-color 0.2s !important;
        border: 2px solid transparent !important;
        position: relative !important;
      }

      .cowatch-gif-item:hover {
        transform: scale(1.02) !important;
        border-color: #ff3b30 !important;
      }

      .cowatch-gif-item img {
        width: 100% !important;
        height: 100% !important;
        object-fit: cover !important;
        display: block !important;
      }

      .cowatch-gif-toggle {
        width: 38px !important;
        height: 38px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 10px !important;
        color: #fff !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
        font-size: 16px !important;
      }

      .cowatch-gif-toggle:hover {
        background: rgba(255,255,255,0.2) !important;
      }

      /* Reply UI */
      .cowatch-reply-bar {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        padding: 8px 12px !important;
        background: rgba(255,255,255,0.05) !important;
        border-radius: 8px 8px 0 0 !important;
        font-size: 12px !important;
        color: rgba(255,255,255,0.7) !important;
      }

      .cowatch-reply-bar .reply-text {
        flex: 1 !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      .cowatch-reply-bar .reply-close {
        background: none !important;
        border: none !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        padding: 4px !important;
      }

      .cowatch-message-reply {
        font-size: 11px !important;
        color: rgba(255,255,255,0.5) !important;
        padding: 4px 8px !important;
        background: rgba(255,255,255,0.05) !important;
        border-left: 2px solid #ff3b30 !important;
        border-radius: 4px !important;
        margin-bottom: 4px !important;
        max-width: 200px !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      .cowatch-message-actions {
        display: none !important;
        position: absolute !important;
        right: 0 !important;
        top: 0 !important;
        background: #1a1a1a !important;
        border-radius: 6px !important;
        padding: 2px !important;
      }

      .cowatch-message:hover .cowatch-message-actions {
        display: flex !important;
      }

      .cowatch-message-action-btn {
        width: 24px !important;
        height: 24px !important;
        background: transparent !important;
        border: none !important;
        color: rgba(255,255,255,0.6) !important;
        cursor: pointer !important;
        border-radius: 4px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .cowatch-message-action-btn:hover {
        background: rgba(255,255,255,0.1) !important;
        color: #fff !important;
      }

      /* GIF Message */
      .cowatch-message-gif {
        max-width: 200px !important;
        border-radius: 8px !important;
        margin-top: 4px !important;
      }

      /* Voice Message */
      .cowatch-voice-message {
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
        background: rgba(255,255,255,0.06) !important;
        border-radius: 20px !important;
        padding: 8px 14px !important;
        margin-top: 4px !important;
        min-width: 200px !important;
        max-width: 280px !important;
      }

      .cowatch-voice-message.playing {
        background: rgba(255, 59, 48, 0.15) !important;
      }

      .cowatch-voice-play {
        width: 36px !important;
        height: 36px !important;
        background: #ff3b30 !important;
        border: none !important;
        border-radius: 50% !important;
        color: #fff !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.2s !important;
        flex-shrink: 0 !important;
      }

      .cowatch-voice-play:hover {
        background: #ff5545 !important;
        transform: scale(1.05) !important;
      }

      .cowatch-voice-play:active {
        transform: scale(0.95) !important;
      }

      .cowatch-voice-progress-container {
        flex: 1 !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 4px !important;
      }

      .cowatch-voice-progress {
        width: 100% !important;
        height: 32px !important;
        position: relative !important;
        display: flex !important;
        align-items: center !important;
        cursor: pointer !important;
      }

      .cowatch-voice-progress-bar {
        position: absolute !important;
        left: 0 !important;
        top: 50% !important;
        transform: translateY(-50%) !important;
        height: 3px !important;
        background: #ff3b30 !important;
        border-radius: 2px !important;
        width: 0% !important;
        z-index: 2 !important;
        transition: width 0.1s linear !important;
      }

      .cowatch-voice-waveform {
        width: 100% !important;
        height: 100% !important;
        display: flex !important;
        align-items: center !important;
        gap: 2px !important;
      }

      .cowatch-voice-waveform span {
        flex: 1 !important;
        max-width: 4px !important;
        min-height: 4px !important;
        background: rgba(255,255,255,0.2) !important;
        border-radius: 2px !important;
        transition: background 0.2s !important;
      }

      .cowatch-voice-message.playing .cowatch-voice-waveform span {
        animation: voiceWave 0.5s ease-in-out infinite alternate !important;
      }

      .cowatch-voice-message.playing .cowatch-voice-waveform span:nth-child(odd) {
        animation-delay: 0.15s !important;
      }

      @keyframes voiceWave {
        from { background: rgba(255,255,255,0.2); }
        to { background: rgba(255, 59, 48, 0.6); }
      }

      .cowatch-voice-times {
        display: flex !important;
        justify-content: space-between !important;
        padding: 0 2px !important;
      }

      .cowatch-voice-current,
      .cowatch-voice-duration {
        color: rgba(255,255,255,0.5) !important;
        font-size: 10px !important;
        font-weight: 500 !important;
      }

      .cowatch-voice-current {
        color: rgba(255, 59, 48, 0.9) !important;
      }

      /* Avatar Selector */
      .cowatch-avatar-selector {
        display: grid !important;
        grid-template-columns: repeat(4, 52px) !important;
        gap: 10px !important;
        margin: 12px 0 !important;
        justify-content: center !important;
        justify-items: center !important;
      }

      .cowatch-avatar-option {
        width: 52px !important;
        height: 52px !important;
        border-radius: 16px !important;
        border: 2px solid rgba(255,255,255,0.1) !important;
        background: rgba(255,255,255,0.05) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 26px !important;
        transition: all 0.15s !important;
      }

      .cowatch-avatar-option:hover {
        transform: scale(1.08) !important;
        border-color: rgba(255,255,255,0.3) !important;
        background: rgba(255,255,255,0.1) !important;
      }

      .cowatch-avatar-option:active {
        transform: scale(0.95) !important;
      }

      .cowatch-avatar-option.selected {
        border-color: #ff3b30 !important;
        background: rgba(255,59,48,0.15) !important;
        box-shadow: 0 0 16px rgba(255, 59, 48, 0.4) !important;
      }

      /* User Setup Screen */
      .cowatch-setup {
        padding: 20px !important;
        text-align: center !important;
      }

      .cowatch-setup-title {
        color: #fff !important;
        font-size: 18px !important;
        font-weight: 600 !important;
        margin-bottom: 8px !important;
      }

      .cowatch-setup-subtitle {
        color: rgba(255,255,255,0.5) !important;
        font-size: 13px !important;
        margin-bottom: 20px !important;
      }

      .cowatch-avatar-preview {
        width: 80px !important;
        height: 80px !important;
        border-radius: 50% !important;
        margin: 0 auto 16px !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 40px !important;
      }

      /* Toolbar in Header */
      .cowatch-toolbar {
        display: flex !important;
        align-items: center !important;
        gap: 2px !important;
      }

      .cowatch-toolbar-btn {
        width: 28px !important;
        height: 28px !important;
        background: transparent !important;
        border: none !important;
        border-radius: 6px !important;
        color: rgba(255,255,255,0.5) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        transition: all 0.15s !important;
      }

      .cowatch-toolbar-btn:hover {
        background: rgba(255,255,255,0.1) !important;
        color: rgba(255,255,255,0.9) !important;
      }

      .cowatch-toolbar-btn:active {
        transform: scale(0.9) !important;
        background: rgba(255,255,255,0.15) !important;
      }

      .cowatch-toolbar-btn.active {
        color: #ff3b30 !important;
      }

      /* Participants Modal */
      .cowatch-modal-overlay {
        position: absolute !important;
        top: 0 !important;
        left: 0 !important;
        right: 0 !important;
        bottom: 0 !important;
        background: rgba(0,0,0,0.8) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        z-index: 100 !important;
        backdrop-filter: blur(4px) !important;
      }

      .cowatch-modal {
        background: #1c1c1e !important;
        border-radius: 16px !important;
        width: 90% !important;
        max-width: 320px !important;
        max-height: 70% !important;
        overflow: hidden !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        box-shadow: 0 20px 60px rgba(0,0,0,0.5) !important;
      }

      .cowatch-modal-header {
        padding: 16px !important;
        border-bottom: 1px solid rgba(255,255,255,0.08) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: space-between !important;
      }

      .cowatch-modal-title {
        color: #fff !important;
        font-size: 16px !important;
        font-weight: 600 !important;
      }

      .cowatch-modal-close {
        width: 28px !important;
        height: 28px !important;
        background: rgba(255,255,255,0.1) !important;
        border: none !important;
        border-radius: 50% !important;
        color: rgba(255,255,255,0.6) !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .cowatch-modal-content {
        padding: 16px !important;
        overflow-y: auto !important;
        max-height: 300px !important;
      }

      .cowatch-participant-item {
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        padding: 10px !important;
        border-radius: 10px !important;
        margin-bottom: 6px !important;
        transition: background 0.2s !important;
      }

      .cowatch-participant-item:hover {
        background: rgba(255,255,255,0.05) !important;
      }

      .cowatch-participant-item .avatar {
        width: 40px !important;
        height: 40px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 20px !important;
      }

      .cowatch-participant-item .info {
        flex: 1 !important;
      }

      .cowatch-participant-item .name {
        color: #fff !important;
        font-weight: 500 !important;
        font-size: 14px !important;
      }

      .cowatch-participant-item .role {
        color: rgba(255,255,255,0.4) !important;
        font-size: 12px !important;
      }

      .cowatch-participant-item.host .role {
        color: #f59e0b !important;
      }

      /* Invite Friends Button */
      .cowatch-btn-invite {
        color: #10b981 !important;
      }
      .cowatch-btn-invite:hover {
        background: rgba(16, 185, 129, 0.15) !important;
        color: #10b981 !important;
      }

      /* Invite Friends Modal */
      .cowatch-invite-modal {
        background: #1c1c1e !important;
        border-radius: 16px !important;
        width: 90% !important;
        max-width: 320px !important;
        max-height: 80% !important;
        overflow: hidden !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        box-shadow: 0 20px 60px rgba(0,0,0,0.5) !important;
      }

      .cowatch-invite-tabs {
        display: flex !important;
        border-bottom: 1px solid rgba(255,255,255,0.08) !important;
        padding: 0 16px !important;
      }

      .cowatch-invite-tab {
        flex: 1 !important;
        padding: 12px !important;
        background: none !important;
        border: none !important;
        color: rgba(255,255,255,0.5) !important;
        font-size: 13px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        border-bottom: 2px solid transparent !important;
        transition: all 0.2s !important;
      }

      .cowatch-invite-tab:hover {
        color: rgba(255,255,255,0.8) !important;
      }

      .cowatch-invite-tab.active {
        color: #ff3b30 !important;
        border-bottom-color: #ff3b30 !important;
      }

      .cowatch-invite-content {
        padding: 16px !important;
        overflow-y: auto !important;
        max-height: 350px !important;
      }

      .cowatch-invite-section {
        margin-bottom: 16px !important;
      }

      .cowatch-invite-section-title {
        font-size: 12px !important;
        color: rgba(255,255,255,0.5) !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
        margin-bottom: 10px !important;
      }

      .cowatch-invite-link-box {
        background: rgba(255,255,255,0.05) !important;
        border-radius: 10px !important;
        padding: 12px !important;
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
      }

      .cowatch-invite-link {
        flex: 1 !important;
        color: rgba(255,255,255,0.7) !important;
        font-size: 13px !important;
        font-family: monospace !important;
        overflow: hidden !important;
        text-overflow: ellipsis !important;
        white-space: nowrap !important;
      }

      .cowatch-invite-copy-btn {
        background: #ff3b30 !important;
        color: white !important;
        border: none !important;
        border-radius: 8px !important;
        padding: 8px 14px !important;
        font-size: 12px !important;
        font-weight: 600 !important;
        cursor: pointer !important;
        transition: all 0.2s !important;
      }

      .cowatch-invite-copy-btn:hover {
        background: #e0352b !important;
        transform: scale(1.02) !important;
      }

      .cowatch-invite-copy-btn.copied {
        background: #10b981 !important;
      }

      .cowatch-friends-search {
        width: 100% !important;
        background: rgba(255,255,255,0.05) !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 10px !important;
        padding: 10px 14px !important;
        color: #fff !important;
        font-size: 14px !important;
        margin-bottom: 12px !important;
      }

      .cowatch-friends-search::placeholder {
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-friends-list {
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
      }

      .cowatch-friend-item {
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
        padding: 10px !important;
        background: rgba(255,255,255,0.03) !important;
        border-radius: 10px !important;
        cursor: pointer !important;
        transition: all 0.2s !important;
      }

      .cowatch-friend-item:hover {
        background: rgba(255,255,255,0.08) !important;
      }

      .cowatch-friend-item .avatar {
        width: 36px !important;
        height: 36px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 18px !important;
      }

      .cowatch-friend-item .info {
        flex: 1 !important;
      }

      .cowatch-friend-item .name {
        color: #fff !important;
        font-weight: 500 !important;
        font-size: 14px !important;
      }

      .cowatch-friend-item .status {
        font-size: 12px !important;
        display: flex !important;
        align-items: center !important;
        gap: 4px !important;
      }

      .cowatch-friend-item .status.online {
        color: #10b981 !important;
      }

      .cowatch-friend-item .status.offline {
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-friend-item .status-dot {
        width: 6px !important;
        height: 6px !important;
        border-radius: 50% !important;
        background: currentColor !important;
      }

      .cowatch-invite-btn {
        background: #ff3b30 !important;
        color: white !important;
        border: none !important;
        border-radius: 8px !important;
        padding: 6px 12px !important;
        font-size: 12px !important;
        font-weight: 600 !important;
        cursor: pointer !important;
        transition: all 0.2s !important;
      }

      .cowatch-invite-btn:hover {
        background: #e0352b !important;
      }

      .cowatch-invite-btn:disabled {
        background: rgba(255,255,255,0.2) !important;
        cursor: not-allowed !important;
      }

      .cowatch-invite-btn.invited {
        background: #10b981 !important;
      }

      .cowatch-no-friends {
        text-align: center !important;
        padding: 30px 20px !important;
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-no-friends svg {
        width: 48px !important;
        height: 48px !important;
        margin-bottom: 12px !important;
        opacity: 0.3 !important;
      }

      .cowatch-no-friends p {
        font-size: 13px !important;
        margin: 0 !important;
      }

      .cowatch-share-buttons {
        display: flex !important;
        gap: 10px !important;
        margin-top: 16px !important;
      }

      .cowatch-share-btn {
        flex: 1 !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        gap: 8px !important;
        padding: 12px !important;
        border: none !important;
        border-radius: 10px !important;
        font-size: 13px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        transition: all 0.2s !important;
      }

      .cowatch-share-btn:hover {
        transform: translateY(-1px) !important;
      }

      .cowatch-share-btn.whatsapp {
        background: #25D366 !important;
        color: white !important;
      }

      .cowatch-share-btn.telegram {
        background: #0088cc !important;
        color: white !important;
      }

      .cowatch-share-btn.twitter {
        background: #1DA1F2 !important;
        color: white !important;
      }

      /* Light Theme Support */
      #cowatch-container.light {
        background: linear-gradient(180deg, #f5f5f7 0%, #ffffff 100%) !important;
      }

      #cowatch-container.light .cowatch-header {
        background: rgba(255, 255, 255, 0.95) !important;
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-header h1 {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-header-subtitle {
        color: rgba(0,0,0,0.5) !important;
      }

      /* Logo stays red in both themes */
      #cowatch-container.light .cowatch-logo svg circle {
        stroke: #ff3b30 !important;
        fill: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-logo svg circle:first-child {
        stroke: #ff3b30 !important;
        fill: none !important;
      }

      #cowatch-container.light .cowatch-avatar-mini {
        border-color: rgba(0,0,0,0.15) !important;
      }

      #cowatch-container.light .cowatch-close {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-close:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-minimize {
        background: rgba(0,0,0,0.06) !important;
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-minimize:hover {
        background: rgba(0,0,0,0.1) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-section-compact {
        background: rgba(0,0,0,0.02) !important;
        border-color: rgba(0,0,0,0.05) !important;
      }

      #cowatch-container.light .cowatch-room-info {
        background: rgba(0,0,0,0.03) !important;
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-room-code-inline {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-btn-icon {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-btn-icon:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-chat {
        background: rgba(0,0,0,0.02) !important;
      }

      #cowatch-container.light .cowatch-message:hover {
        background: rgba(0,0,0,0.03) !important;
      }

      #cowatch-container.light .cowatch-message-username {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-message-text {
        color: rgba(0,0,0,0.8) !important;
      }

      #cowatch-container.light .cowatch-message-time {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-message.system .cowatch-message-text {
        color: rgba(0,0,0,0.6) !important;
        background: rgba(255, 59, 48, 0.08) !important;
        border-color: rgba(255, 59, 48, 0.15) !important;
      }

      #cowatch-container.light .cowatch-chat-input {
        background: rgba(255,255,255,0.95) !important;
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-input-row {
        background: transparent !important;
      }

      #cowatch-container.light .cowatch-input-wrapper {
        background: rgba(0,0,0,0.04) !important;
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-input-wrapper:focus-within {
        background: rgba(0,0,0,0.06) !important;
        border-color: rgba(0,0,0,0.15) !important;
      }

      #cowatch-container.light .cowatch-input-action {
        background: rgba(0,0,0,0.04) !important;
        border-color: rgba(0,0,0,0.08) !important;
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-chat-input input {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-chat-input input::placeholder {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-input-action:hover {
        color: #1d1d1f !important;
        background: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-emoji-picker,
      #cowatch-container.light .cowatch-gif-picker {
        background: #ffffff !important;
        border-color: rgba(0,0,0,0.1) !important;
      }

      #cowatch-container.light .cowatch-gif-category {
        background: rgba(0,0,0,0.05) !important;
        color: rgba(0,0,0,0.7) !important;
      }

      #cowatch-container.light .cowatch-gif-category:hover {
        background: rgba(0,0,0,0.1) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-modal {
        background: #ffffff !important;
      }

      #cowatch-container.light .cowatch-modal-header {
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-modal-title {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-modal-close {
        background: rgba(0,0,0,0.08) !important;
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-modal-close:hover {
        background: rgba(0,0,0,0.12) !important;
        color: #1d1d1f !important;
      }

      /* Light Theme - Invite Modal */
      #cowatch-container.light .cowatch-invite-modal {
        background: #ffffff !important;
      }

      #cowatch-container.light .cowatch-invite-tabs {
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-invite-tab {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-invite-tab:hover {
        color: rgba(0,0,0,0.8) !important;
      }

      #cowatch-container.light .cowatch-invite-tab.active {
        color: #ff3b30 !important;
        border-bottom-color: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-invite-section-title {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-invite-link-box {
        background: rgba(0,0,0,0.04) !important;
      }

      #cowatch-container.light .cowatch-invite-link {
        color: rgba(0,0,0,0.7) !important;
      }

      #cowatch-container.light .cowatch-friends-search {
        background: rgba(0,0,0,0.04) !important;
        border-color: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-friends-search::placeholder {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-friend-item {
        background: rgba(0,0,0,0.02) !important;
      }

      #cowatch-container.light .cowatch-friend-item:hover {
        background: rgba(0,0,0,0.06) !important;
      }

      #cowatch-container.light .cowatch-friend-item .name {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-friend-item .status.offline {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-no-friends {
        color: rgba(0,0,0,0.4) !important;
      }

      /* Toast animations */
      @keyframes cowatchToastIn {
        from {
          transform: translateX(100%);
          opacity: 0;
        }
        to {
          transform: translateX(0);
          opacity: 1;
        }
      }

      @keyframes cowatchToastOut {
        from {
          transform: translateX(0);
          opacity: 1;
        }
        to {
          transform: translateX(100%);
          opacity: 0;
        }
      }

      /* Review Prompt */
      .cowatch-review-prompt {
        position: fixed !important;
        bottom: 20px !important;
        right: 360px !important;
        z-index: 2147483647 !important;
        background: linear-gradient(135deg, #1c1c1e 0%, #2c2c2e 100%) !important;
        border-radius: 16px !important;
        padding: 20px !important;
        box-shadow: 0 10px 40px rgba(0,0,0,0.4) !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        animation: cowatchToastIn 0.3s ease !important;
        max-width: 300px !important;
      }

      .cowatch-review-content {
        text-align: center !important;
      }

      .cowatch-review-icon {
        font-size: 48px !important;
        margin-bottom: 12px !important;
      }

      .cowatch-review-content h3 {
        color: #fff !important;
        font-size: 18px !important;
        font-weight: 600 !important;
        margin: 0 0 8px 0 !important;
      }

      .cowatch-review-content p {
        color: rgba(255,255,255,0.6) !important;
        font-size: 14px !important;
        margin: 0 0 16px 0 !important;
        line-height: 1.4 !important;
      }

      .cowatch-review-buttons {
        display: flex !important;
        gap: 10px !important;
      }

      .cowatch-review-btn {
        flex: 1 !important;
        padding: 10px 16px !important;
        border-radius: 10px !important;
        font-size: 14px !important;
        font-weight: 600 !important;
        cursor: pointer !important;
        border: none !important;
        transition: all 0.2s !important;
      }

      .cowatch-review-btn.primary {
        background: linear-gradient(135deg, #ff3b30 0%, #ff6b5b 100%) !important;
        color: white !important;
      }

      .cowatch-review-btn.primary:hover {
        transform: translateY(-2px) !important;
        box-shadow: 0 4px 15px rgba(255, 59, 48, 0.4) !important;
      }

      .cowatch-review-btn.secondary {
        background: rgba(255,255,255,0.1) !important;
        color: rgba(255,255,255,0.7) !important;
      }

      .cowatch-review-btn.secondary:hover {
        background: rgba(255,255,255,0.15) !important;
      }

      /* Scheduled Events */
      .cowatch-schedule-modal {
        background: #1c1c1e !important;
        border-radius: 16px !important;
        width: 90% !important;
        max-width: 400px !important;
        max-height: 80% !important;
        overflow: hidden !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
      }

      .cowatch-schedule-header {
        padding: 16px !important;
        border-bottom: 1px solid rgba(255,255,255,0.08) !important;
        display: flex !important;
        justify-content: space-between !important;
        align-items: center !important;
      }

      .cowatch-schedule-title {
        font-size: 18px !important;
        font-weight: 600 !important;
        color: #fff !important;
      }

      .cowatch-schedule-content {
        padding: 16px !important;
        max-height: 400px !important;
        overflow-y: auto !important;
      }

      .cowatch-schedule-empty {
        text-align: center !important;
        padding: 40px 20px !important;
        color: rgba(255,255,255,0.4) !important;
      }

      .cowatch-schedule-empty svg {
        width: 64px !important;
        height: 64px !important;
        margin-bottom: 16px !important;
        opacity: 0.3 !important;
      }

      .cowatch-event-item {
        background: rgba(255,255,255,0.03) !important;
        border-radius: 12px !important;
        padding: 14px !important;
        margin-bottom: 10px !important;
        border: 1px solid rgba(255,255,255,0.06) !important;
      }

      .cowatch-event-item:hover {
        background: rgba(255,255,255,0.06) !important;
      }

      .cowatch-event-title {
        font-size: 15px !important;
        font-weight: 600 !important;
        color: #fff !important;
        margin-bottom: 6px !important;
      }

      .cowatch-event-time {
        font-size: 13px !important;
        color: #ff3b30 !important;
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
        margin-bottom: 8px !important;
      }

      .cowatch-event-platform {
        font-size: 12px !important;
        color: rgba(255,255,255,0.5) !important;
        display: flex !important;
        align-items: center !important;
        gap: 6px !important;
      }

      .cowatch-event-actions {
        display: flex !important;
        gap: 8px !important;
        margin-top: 12px !important;
      }

      .cowatch-event-btn {
        flex: 1 !important;
        padding: 8px 12px !important;
        border-radius: 8px !important;
        font-size: 13px !important;
        font-weight: 500 !important;
        cursor: pointer !important;
        border: none !important;
        transition: all 0.2s !important;
      }

      .cowatch-event-btn.join {
        background: #ff3b30 !important;
        color: white !important;
      }

      .cowatch-event-btn.reminder {
        background: rgba(255,255,255,0.1) !important;
        color: rgba(255,255,255,0.8) !important;
      }

      .cowatch-create-event-form {
        display: flex !important;
        flex-direction: column !important;
        gap: 14px !important;
      }

      .cowatch-form-group {
        display: flex !important;
        flex-direction: column !important;
        gap: 6px !important;
      }

      .cowatch-form-label {
        font-size: 13px !important;
        color: rgba(255,255,255,0.6) !important;
        font-weight: 500 !important;
      }

      .cowatch-form-input {
        background: rgba(255,255,255,0.05) !important;
        border: 1px solid rgba(255,255,255,0.1) !important;
        border-radius: 10px !important;
        padding: 12px 14px !important;
        color: #fff !important;
        font-size: 14px !important;
      }

      .cowatch-form-input:focus {
        outline: none !important;
        border-color: #ff3b30 !important;
      }

      .cowatch-datetime-row {
        display: flex !important;
        gap: 10px !important;
      }

      .cowatch-datetime-row .cowatch-form-group {
        flex: 1 !important;
      }

      #cowatch-container.light .cowatch-participant-item .name {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-participant-item .role {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-toolbar-btn {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-toolbar-btn:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-toolbar-btn svg {
        stroke: currentColor !important;
      }

      #cowatch-container.light .cowatch-btn-icon {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-btn-icon:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-btn-icon svg {
        stroke: currentColor !important;
      }

      #cowatch-container.light .cowatch-send-btn {
        background: #ff3b30 !important;
        color: #fff !important;
      }

      #cowatch-container.light .cowatch-voice-message {
        background: rgba(0,0,0,0.06) !important;
      }

      #cowatch-container.light .cowatch-voice-message.playing {
        background: rgba(255, 59, 48, 0.1) !important;
      }

      #cowatch-container.light .cowatch-voice-waveform span {
        background: rgba(0,0,0,0.2) !important;
      }

      #cowatch-container.light .cowatch-voice-current,
      #cowatch-container.light .cowatch-voice-duration {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-voice-current {
        color: rgba(255, 59, 48, 0.9) !important;
      }

      #cowatch-container.light .cowatch-message-actions {
        background: rgba(255,255,255,0.9) !important;
      }

      #cowatch-container.light .cowatch-message-action-btn {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-message-action-btn:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-message-reply {
        background: rgba(0,0,0,0.04) !important;
        border-left-color: rgba(0,0,0,0.2) !important;
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-reply-bar {
        background: rgba(255,59,48,0.08) !important;
        color: rgba(0,0,0,0.7) !important;
      }

      #cowatch-container.light .cowatch-reply-bar button {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-gif-search input {
        background: rgba(0,0,0,0.06) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-gif-search input::placeholder {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-emoji-btn:hover {
        background: rgba(0,0,0,0.08) !important;
      }

      /* Light theme - Setup screen */
      #cowatch-container.light .cowatch-setup-title {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-setup-subtitle {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-section {
        border-color: rgba(0,0,0,0.05) !important;
      }

      #cowatch-container.light .cowatch-section-title {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-input {
        background: rgba(0,0,0,0.04) !important;
        border-color: rgba(0,0,0,0.1) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-input:focus {
        border-color: #ff3b30 !important;
        background: rgba(0,0,0,0.02) !important;
      }

      #cowatch-container.light .cowatch-input::placeholder {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-status {
        background: rgba(0,0,0,0.04) !important;
        border-color: rgba(0,0,0,0.1) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-status.connected {
        background: rgba(74, 222, 128, 0.1) !important;
        border-color: rgba(74, 222, 128, 0.3) !important;
      }

      #cowatch-container.light .cowatch-avatar-option {
        border-color: rgba(0,0,0,0.1) !important;
      }

      #cowatch-container.light .cowatch-avatar-option.selected {
        border-color: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-btn {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-btn-secondary {
        background: rgba(0,0,0,0.06) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-btn-secondary:hover {
        background: rgba(0,0,0,0.1) !important;
      }

      #cowatch-container.light .cowatch-gif-item:hover {
        border-color: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-profile-btn {
        background: rgba(0,0,0,0.04) !important;
      }

      #cowatch-container.light .cowatch-profile-btn:hover {
        background: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-profile-name {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-profile-btn svg {
        color: rgba(0,0,0,0.4) !important;
      }

      #cowatch-container.light .cowatch-back-btn {
        background: rgba(0,0,0,0.06) !important;
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-back-btn:hover {
        background: rgba(0,0,0,0.1) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-header-badge {
        background: rgba(0,0,0,0.06) !important;
        color: rgba(0,0,0,0.7) !important;
      }

      #cowatch-container.light .cowatch-section-label {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-message-text {
        background: rgba(0,0,0,0.04) !important;
        color: rgba(0,0,0,0.85) !important;
      }

      #cowatch-container.light .cowatch-reply-bar {
        background: rgba(255,59,48,0.08) !important;
        color: rgba(0,0,0,0.7) !important;
      }

      /* Light mode - Additional fixes */
      #cowatch-container.light .cowatch-toolbar-btn {
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-toolbar-btn:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-toolbar-btn.active {
        background: rgba(255,59,48,0.1) !important;
        color: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-participant-count {
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-avatar-mini {
        border-color: rgba(0,0,0,0.1) !important;
      }

      #cowatch-container.light .cowatch-content {
        background: transparent !important;
      }

      #cowatch-container.light .cowatch-room-code-inline {
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-voice-message {
        background: rgba(0,0,0,0.04) !important;
      }

      #cowatch-container.light .cowatch-voice-play {
        background: #ff3b30 !important;
        color: #fff !important;
      }

      #cowatch-container.light .cowatch-voice-progress-bar {
        background: rgba(0,0,0,0.2) !important;
      }

      #cowatch-container.light .cowatch-voice-times {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-recorded-audio {
        background: rgba(0,0,0,0.04) !important;
      }

      #cowatch-container.light .cowatch-recorded-cancel,
      #cowatch-container.light .cowatch-recorded-info {
        color: rgba(0,0,0,0.6) !important;
      }

      #cowatch-container.light .cowatch-recording-bar {
        background: rgba(239,68,68,0.08) !important;
      }

      #cowatch-container.light .cowatch-language-picker {
        background: rgba(255,255,255,0.98) !important;
        border-color: rgba(0,0,0,0.1) !important;
      }

      #cowatch-container.light .cowatch-language-header {
        color: #1d1d1f !important;
        border-color: rgba(0,0,0,0.08) !important;
      }

      #cowatch-container.light .cowatch-language-close {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-language-close:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-language-item {
        color: rgba(0,0,0,0.7) !important;
      }

      #cowatch-container.light .cowatch-language-item:hover {
        background: rgba(0,0,0,0.04) !important;
        color: #1d1d1f !important;
      }

      #cowatch-container.light .cowatch-language-item.active {
        background: rgba(255,59,48,0.1) !important;
        color: #ff3b30 !important;
      }

      #cowatch-container.light .cowatch-message-actions {
        background: rgba(255,255,255,0.9) !important;
      }

      #cowatch-container.light .cowatch-message-action-btn {
        color: rgba(0,0,0,0.5) !important;
      }

      #cowatch-container.light .cowatch-message-action-btn:hover {
        background: rgba(0,0,0,0.08) !important;
        color: #1d1d1f !important;
      }

      /* Voice Message Recording */
      .cowatch-recording-indicator {
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
        padding: 8px 12px !important;
        background: rgba(239, 68, 68, 0.1) !important;
        border-radius: 20px !important;
        color: #ef4444 !important;
        font-size: 12px !important;
        font-weight: 500 !important;
      }

      .cowatch-recording-dot {
        width: 8px !important;
        height: 8px !important;
        background: #ef4444 !important;
        border-radius: 50% !important;
        animation: pulse 1s infinite !important;
      }

      @keyframes pulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.3; }
      }

      /* Audio Message */
      .cowatch-audio-message {
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
        padding: 8px 12px !important;
        background: rgba(255,255,255,0.05) !important;
        border-radius: 20px !important;
        margin-top: 4px !important;
      }

      .cowatch-audio-play {
        width: 32px !important;
        height: 32px !important;
        background: #ff3b30 !important;
        border: none !important;
        border-radius: 50% !important;
        color: #fff !important;
        cursor: pointer !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
      }

      .cowatch-audio-wave {
        flex: 1 !important;
        height: 24px !important;
        display: flex !important;
        align-items: center !important;
        gap: 2px !important;
      }

      .cowatch-audio-bar {
        width: 3px !important;
        background: rgba(255,255,255,0.4) !important;
        border-radius: 2px !important;
      }
    `;

    document.head.appendChild(style);
  }

  createUI() {
    // ALWAYS inject styles first
    this.injectStyles();
    
    // Remove existing UI
    document.getElementById('cowatch-container')?.remove();
    document.getElementById('cowatch-toggle')?.remove();
    document.getElementById('cowatch-iframe')?.remove();

    // ALWAYS create toggle button when panel is closed (gives users a way to reopen)
    if (!this.chatOpen) {
      this.createToggleButton();
    }

    // TELEPARTY APPROACH: Always use iframe for chat
    // The iframe loads cowatch.app/sidebar which handles all chat UI
    // This provides better isolation from platform CSS and a consistent experience
    if (this.useIframeChat) {
      this.createIframeChatPanel();
      return;
    }

    // Create main container (fallback to original approach - disabled)
    const container = document.createElement('div');
    container.id = 'cowatch-container';
    // Apply theme class on creation
    if (this.theme === 'light') {
      container.classList.add('light');
    }
    // Open panel if chatOpen is true (restored from state)
    if (this.chatOpen) {
      container.classList.add('open');
      document.body.classList.add('cowatch-panel-open');
    }
    container.innerHTML = this.renderPanel();
    document.body.appendChild(container);

    this.bindEvents();
    
    // Initialize draggable/resizable functionality
    this.initDraggablePanel();
    
    // If we have a room code and chatOpen, reconnect to the room
    if (this.roomCode && this.chatOpen && !this.isConnected) {
      console.log('🎬 Reconnecting to room:', this.roomCode);
      this.connectToRoom(this.roomCode);
    }
  }

  // Create iframe-based chat panel (Teleparty approach)
  // The chat UI is loaded from cowatch.app/sidebar for better isolation
  createIframeChatPanel() {
    console.log('🎬 Creating iframe-based chat panel (Teleparty approach)');
    
    // Get extension ID for communication
    const extensionId = chrome.runtime?.id || '';
    const timestamp = Date.now();
    
    // Get avatar id from userAvatar object
    const avatarId = this.userAvatar?.id || (typeof this.userAvatar === 'string' ? this.userAvatar : '');
    
    // Build sidebar URL with parameters
    const sidebarUrl = new URL(`${COWATCH_URL}/sidebar`);
    sidebarUrl.searchParams.set('t', timestamp.toString());
    sidebarUrl.searchParams.set('id', `chrome-extension://${extensionId}`);
    sidebarUrl.searchParams.set('room', this.roomCode || '');
    sidebarUrl.searchParams.set('username', this.username || '');
    sidebarUrl.searchParams.set('avatar', avatarId);
    sidebarUrl.searchParams.set('lang', this.language || 'en');
    sidebarUrl.searchParams.set('isHost', this.isHost ? 'true' : 'false');
    sidebarUrl.searchParams.set('platform', this.platform?.name || '');
    sidebarUrl.searchParams.set('videoUrl', this.video?.src || window.location.href);
    
    // Create container for iframe (340px width like Teleparty)
    const container = document.createElement('div');
    container.id = 'cowatch-container';
    container.style.cssText = `
      position: fixed !important;
      top: 0 !important;
      right: 0 !important;
      width: 340px !important;
      height: 100vh !important;
      z-index: 9999999999 !important;
      background: #1a1a2e !important;
      box-shadow: -4px 0 20px rgba(0,0,0,0.5) !important;
      transform: translateX(${this.chatOpen ? '0' : '100%'}) !important;
      transition: transform 0.3s ease !important;
      display: flex !important;
      flex-direction: column !important;
    `;
    
    if (this.chatOpen) {
      container.classList.add('open');
      document.body.classList.add('cowatch-panel-open');
    }
    
    // Create iframe
    const iframe = document.createElement('iframe');
    iframe.id = 'cowatch-iframe';
    iframe.src = sidebarUrl.toString();
    iframe.style.cssText = `
      width: 100% !important;
      height: 100% !important;
      border: none !important;
      background: transparent !important;
      flex: 1 !important;
    `;
    iframe.allow = 'microphone; camera';
    
    container.appendChild(iframe);
    document.body.appendChild(container);
    
    // Notify video resize when chat opens
    if (this.chatOpen) {
      this.notifyChatVisibility(true);
    }
    
    // Listen for messages from iframe
    this.setupIframeMessageListener();
    
    console.log('🎬 Iframe chat panel created:', sidebarUrl.toString());
  }

  // Listen for messages from sidebar iframe
  setupIframeMessageListener() {
    window.addEventListener('message', (event) => {
      // Only accept messages from our sidebar
      if (!event.origin.includes('cowatch.app')) return;
      if (event.data?.source !== 'cowatch-sidebar') return;
      
      const { type, data } = event.data;
      console.log('🎬 Received message from sidebar:', type, data);
      
      switch (type) {
        case 'close':
          this.togglePanel(false);
          break;
          
        case 'toggle-play':
          if (this.video) {
            if (this.video.paused) {
              // Use platform-specific method for play
              if (this.platform.name === 'netflix') {
                window.postMessage({ type: 'PLAY' }, '*');
              } else {
                this.video.play();
              }
            } else {
              // Use platform-specific method for pause
              if (this.platform.name === 'netflix') {
                window.postMessage({ type: 'PAUSE' }, '*');
              } else {
                this.video.pause();
              }
            }
          }
          break;
          
        case 'seek':
          if (this.video && typeof data === 'number') {
            // Use platform-specific method for seeking
            if (this.platform.name === 'netflix') {
              const newTime = (this.video.currentTime + data) * 1000;
              window.postMessage({ type: 'SEEK', time: newTime }, '*');
            } else {
              this.video.currentTime += data;
            }
          }
          break;
          
        case 'video-state':
          // Handle video state sync from iframe
          if (this.video && data) {
            if (data.isPlaying !== undefined) {
              if (this.platform.name === 'netflix') {
                window.postMessage({ type: data.isPlaying ? 'PLAY' : 'PAUSE' }, '*');
              } else {
                data.isPlaying ? this.video.play() : this.video.pause();
              }
            }
            if (data.currentTime !== undefined) {
              if (this.platform.name === 'netflix') {
                window.postMessage({ type: 'SEEK', time: data.currentTime * 1000 }, '*');
              } else {
                this.video.currentTime = data.currentTime;
              }
            }
          }
          break;
          
        case 'room-created':
          // Room was created in sidebar, update local state
          if (data && data.code) {
            this.roomCode = data.code;
            this.saveState();
          }
          break;
          
        case 'room-joined':
          // Room was joined in sidebar, update local state
          if (data && data.code) {
            this.roomCode = data.code;
            this.saveState();
          }
          break;
          
        case 'room-left':
          // User left room, clear local state
          this.roomCode = null;
          this.saveState();
          this.removeToggleButton();
          break;
      }
    });
  }

  // Send message to sidebar iframe
  sendToIframe(type, data) {
    const iframe = document.getElementById('cowatch-iframe');
    if (iframe && iframe.contentWindow) {
      iframe.contentWindow.postMessage({
        source: 'cowatch-extension',
        type,
        data
      }, '*');
    }
  }

  createToggleButton() {
    // Remove existing toggle if any
    document.getElementById('cowatch-toggle')?.remove();
    
    // TELEPARTY STYLE: Round button at top:100px, right:30px
    // Shows party controls (copy link, show chat, leave) when in session
    const toggle = document.createElement('button');
    toggle.id = 'cowatch-toggle';
    toggle.className = this.isConnected ? 'connected' : '';
    
    // Teleparty uses circular button with icon
    toggle.innerHTML = `
      <svg viewBox="0 0 24 24" fill="none" width="24" height="24">
        <circle cx="12" cy="12" r="10" stroke="white" stroke-width="2" fill="none"/>
        <circle cx="12" cy="12" r="4" fill="white"/>
        <circle cx="9" cy="10" r="1" fill="currentColor"/>
      </svg>
      ${this.participants.length > 1 ? `<span class="badge">${this.participants.length}</span>` : ''}
      ${this.unreadCount > 0 ? `<span class="unread-badge">${this.unreadCount > 9 ? '9+' : this.unreadCount}</span>` : ''}
    `;
    toggle.addEventListener('click', () => this.togglePanel());
    
    // Teleparty style: top:100px, right:30px, width:50px, circular
    toggle.style.cssText = `
      position: fixed !important;
      top: 100px !important;
      right: 30px !important;
      width: 50px !important;
      height: 50px !important;
      z-index: 9999999998 !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      visibility: visible !important;
      opacity: 1 !important;
      pointer-events: auto !important;
      background: linear-gradient(135deg, ${this.isConnected ? '#4ade80' : '#6366f1'} 0%, ${this.isConnected ? '#22c55e' : '#4f46e5'} 100%) !important;
      border: none !important;
      border-radius: 50% !important;
      cursor: pointer !important;
      box-shadow: 0 4px 15px ${this.isConnected ? 'rgba(74, 222, 128, 0.4)' : 'rgba(99, 102, 241, 0.4)'} !important;
      transition: all 0.2s ease !important;
    `;
    
    // Add tooltip
    toggle.title = this.isConnected ? 'Open CoWatch Chat' : 'Start Watch Party';
    
    document.body.appendChild(toggle);
  }

  removeToggleButton() {
    document.getElementById('cowatch-toggle')?.remove();
  }

  renderPanel() {
    // Show connected state if we have a room code (even if WebSocket not yet connected)
    if (this.roomCode) {
      return this.renderConnectedState();
    }
    return this.renderDisconnectedState();
  }

  renderDisconnectedState() {
    const hasVideo = !!this.video;
    // Only show setup if explicitly requested (showProfileSetup flag)
    const needsSetup = this.showProfileSetup && (!this.username || this.username === 'Guest');
    const selectedAvatar = this.userAvatar || AVATARS[0];
    
    // If user needs to set up profile first (only when showProfileSetup is true)
    if (needsSetup) {
      return `
        <div class="cowatch-header">
          <div class="cowatch-header-left">
            <div class="cowatch-logo">
              <svg viewBox="0 0 24 24" fill="none" width="24" height="24">
                <circle cx="12" cy="12" r="9" stroke="white" stroke-width="2" fill="none"/>
                <circle cx="12" cy="12" r="4" fill="white"/>
              </svg>
            </div>
            <div>
              <h1>CoWatch</h1>
              <div class="cowatch-header-subtitle">Watch Together</div>
            </div>
          </div>
          <button class="cowatch-close" id="cowatchClose">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M18 6L6 18M6 6l12 12"/>
            </svg>
          </button>
        </div>

        <div class="cowatch-content">
          <div class="cowatch-setup">
            <div class="cowatch-setup-title">Set Up Your Profile</div>
            <div class="cowatch-setup-subtitle">Choose a name and avatar to get started</div>
            
            <div class="cowatch-avatar-preview" id="avatarPreview" style="background: ${selectedAvatar.customImage ? 'transparent' : (selectedAvatar.color || '#6366f1')};">
              ${selectedAvatar.customImage ? `<img src="${selectedAvatar.customImage}" style="width:100%;height:100%;object-fit:cover;border-radius:50%;">` : (selectedAvatar.emoji || '😊')}
            </div>
            
            <input type="text" class="cowatch-input" id="cowatchSetupUsername" 
              placeholder="Enter your name" 
              maxlength="20" 
              style="text-align: center;">
            
            <div class="cowatch-avatar-selector" id="avatarSelector">
              ${AVATARS.map((avatar, index) => `
                <button class="cowatch-avatar-option ${index === 0 ? 'selected' : ''}" 
                  data-avatar-index="${index}"
                  style="background: ${avatar.color};">
                  ${avatar.emoji}
                </button>
              `).join('')}
            </div>
            
            <button class="cowatch-btn cowatch-btn-primary" id="cowatchContinue" disabled>
              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M5 12h14M12 5l7 7-7 7"/>
              </svg>
              Continue
            </button>
          </div>
        </div>
      `;
    }
    
    return `
      <div class="cowatch-header">
        <div class="cowatch-header-left">
          <div class="cowatch-logo">
            <svg viewBox="0 0 24 24" fill="none" width="24" height="24">
              <circle cx="12" cy="12" r="9" stroke="white" stroke-width="2" fill="none"/>
              <circle cx="12" cy="12" r="4" fill="white"/>
            </svg>
          </div>
          <div>
            <h1>CoWatch</h1>
            <div class="cowatch-header-subtitle">Watch Together</div>
          </div>
        </div>
        <button class="cowatch-close" id="cowatchClose">
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M18 6L6 18M6 6l12 12"/>
          </svg>
        </button>
      </div>

      <div class="cowatch-content">
        <div class="cowatch-section">
          <div class="cowatch-status ${hasVideo ? 'connected' : ''}">
            <span class="cowatch-status-dot"></span>
            ${hasVideo ? `${this.platform.label} video detected` : 'Looking for video...'}
          </div>

          ${hasVideo ? `
            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 16px; padding: 10px; background: rgba(255,255,255,0.05); border-radius: 10px;">
              <div style="width: 40px; height: 40px; border-radius: 50%; background: ${selectedAvatar.customImage ? 'transparent' : (selectedAvatar.color || '#6366f1')}; display: flex; align-items: center; justify-content: center; font-size: 20px; overflow: hidden;">
                ${selectedAvatar.customImage ? `<img src="${selectedAvatar.customImage}" style="width:100%;height:100%;object-fit:cover;">` : (selectedAvatar.emoji || '😊')}
              </div>
              <div style="flex: 1;">
                <div style="color: #fff; font-weight: 500;">${this.username}</div>
                <button id="cowatchEditProfile" style="background: none; border: none; color: rgba(255,255,255,0.5); font-size: 12px; cursor: pointer; padding: 0;">Edit profile</button>
              </div>
            </div>

            <button class="cowatch-btn cowatch-btn-primary" id="cowatchCreate">
              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M12 5v14M5 12h14"/>
              </svg>
              Start Watch Party
            </button>

            <div class="cowatch-divider">or join existing party</div>

            <input type="text" class="cowatch-input" id="cowatchRoomCode" 
              placeholder="Enter room code (e.g. ABC123)" 
              maxlength="6" 
              style="text-transform: uppercase; letter-spacing: 3px; text-align: center; font-size: 18px;">

            <button class="cowatch-btn cowatch-btn-secondary" id="cowatchJoin">
              <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M15 3h4a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2h-4M10 17l5-5-5-5M15 12H3"/>
              </svg>
              Join Party
            </button>
          ` : `
            <div class="cowatch-empty">
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                <rect x="2" y="4" width="20" height="16" rx="2"/>
                <path d="M10 9l5 3-5 3V9z"/>
              </svg>
              <h3>No Video Found</h3>
              <p>Navigate to a video on ${this.platform.label} to start a watch party with friends.</p>
            </div>
          `}
        </div>
      </div>
    `;
  }

  renderConnectedState() {
    const selectedAvatar = this.userAvatar || AVATARS[0];
    const participantCount = this.participants.length || 1;
    
    // Profile edit panel
    if (this.currentPanel === 'profile') {
      return this.renderProfileEditPanel();
    }
    
    // Participants full panel
    if (this.currentPanel === 'participants') {
      return this.renderParticipantsPanel();
    }
    
    return `
      <div class="cowatch-header">
        <div class="cowatch-header-left">
          <div class="cowatch-logo">
            <svg width="20" height="20" viewBox="0 0 32 32" fill="none">
              <circle cx="16" cy="16" r="12" stroke="currentColor" stroke-width="2" fill="none"/>
              <circle cx="16" cy="16" r="5" fill="currentColor"/>
            </svg>
            <span>CoWatch</span>
          </div>
        </div>
        <div class="cowatch-toolbar">
          <button class="cowatch-toolbar-btn" id="cowatchParticipants" title="Participants">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
              <circle cx="9" cy="7" r="4"/>
              <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
              <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
            </svg>
            <span class="cowatch-participant-count">${participantCount}</span>
          </button>
          <button class="cowatch-toolbar-btn ${this.theme === 'light' ? 'active' : ''}" id="cowatchTheme" title="Toggle theme">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              ${this.theme === 'dark' ? `
                <circle cx="12" cy="12" r="5"/>
                <line x1="12" y1="1" x2="12" y2="3"/>
                <line x1="12" y1="21" x2="12" y2="23"/>
                <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/>
                <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/>
                <line x1="1" y1="12" x2="3" y2="12"/>
                <line x1="21" y1="12" x2="23" y2="12"/>
                <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/>
                <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/>
              ` : `
                <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"/>
              `}
            </svg>
          </button>
          <button class="cowatch-toolbar-btn cowatch-profile-btn-small" id="cowatchProfileEdit" title="Settings">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <circle cx="12" cy="12" r="3"/>
              <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/>
            </svg>
          </button>
          <button class="cowatch-toolbar-btn" id="cowatchMinimize" title="Minimize">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <polyline points="15 18 9 12 15 6"/>
            </svg>
          </button>
        </div>
      </div>

      <div class="cowatch-content">
        <div class="cowatch-section cowatch-section-compact">
          <div class="cowatch-room-info">
            <span class="cowatch-room-code-inline">${this.roomCode}</span>
            <button class="cowatch-btn-icon" id="cowatchResetVideo" title="Return to video">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polygon points="5 3 19 12 5 21 5 3"/>
              </svg>
            </button>
            <button class="cowatch-btn-icon cowatch-btn-invite" id="cowatchInviteFriends" title="Invite Friends">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M16 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                <circle cx="8.5" cy="7" r="4"/>
                <line x1="20" y1="8" x2="20" y2="14"/>
                <line x1="23" y1="11" x2="17" y2="11"/>
              </svg>
            </button>
            <button class="cowatch-btn-icon" id="cowatchCopy" title="Copy link">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <rect x="9" y="9" width="13" height="13" rx="2"/>
                <path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"/>
              </svg>
            </button>
            <button class="cowatch-btn-icon cowatch-btn-icon-danger" id="cowatchLeave" title="Leave">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4M16 17l5-5-5-5M21 12H9"/>
              </svg>
            </button>
          </div>
        </div>

        <div class="cowatch-chat">
          <div class="cowatch-messages" id="cowatchMessages">
            ${this.messages.length === 0 ? `
              <div class="cowatch-message system">
                <div class="cowatch-message-text">Welcome to the watch party!</div>
              </div>
            ` : ''}
            ${this.messages.map(m => this.renderMessage(m)).join('')}
          </div>

          <div class="cowatch-chat-input">
            ${this.replyingTo ? `
              <div class="cowatch-reply-bar">
                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                  <path d="M3 10h10a8 8 0 0 1 8 8v4M3 10l6 6M3 10l6-6"/>
                </svg>
                <div class="cowatch-reply-info">
                  <span class="cowatch-reply-to">Replying to <strong>${this.replyingTo.username}</strong></span>
                  <span class="cowatch-reply-text">${this.replyingTo.text || ''}</span>
                </div>
                <button id="cowatchCancelReply">✕</button>
              </div>
            ` : ''}
            
            <div class="cowatch-gif-picker ${this.gifPickerOpen ? 'open' : ''}" id="cowatchGifPicker">
              <div class="cowatch-gif-search">
                <input type="text" id="cowatchGifSearch" placeholder="Search GIFs...">
              </div>
              <div class="cowatch-gif-categories">
                ${GIF_CATEGORIES.map((cat, i) => `
                  <button class="cowatch-gif-category ${i === 0 ? 'active' : ''}" data-category="${cat.searchTerm}">
                    ${cat.name}
                  </button>
                `).join('')}
              </div>
              <div class="cowatch-gif-grid" id="cowatchGifGrid">
                <div style="grid-column: 1/-1; text-align: center; padding: 20px; color: rgba(255,255,255,0.4); font-size: 13px;">
                  Select a category or search
                </div>
              </div>
            </div>
            
            <div class="cowatch-emoji-picker ${this.emojiPickerOpen ? 'open' : ''}" id="cowatchEmojiPicker">
              <div class="cowatch-reaction-section">
                <div class="cowatch-reaction-label">Animated Reactions</div>
                <div class="cowatch-animated-reactions">
                  ${ANIMATED_REACTIONS.map(r => `
                    <button class="cowatch-animated-reaction" data-reaction-id="${r.id}" data-reaction-gif="${r.gif}" title="${r.name}">
                      <img src="${r.gif}" alt="${r.name}" onerror="this.style.display='none';this.nextElementSibling.style.display='flex';">
                      <span class="cowatch-reaction-fallback" style="display:none;font-size:24px;">${r.emoji}</span>
                    </button>
                  `).join('')}
                </div>
              </div>
              <div class="cowatch-emoji-divider"></div>
              <div class="cowatch-reaction-section">
                <div class="cowatch-reaction-label">Quick Emojis</div>
                <div class="cowatch-quick-emojis">
                  ${REACTION_EMOJIS.map(e => `<button class="cowatch-emoji-btn" data-emoji="${e}">${e}</button>`).join('')}
                </div>
              </div>
            </div>
            
            <div class="cowatch-language-picker ${this.languagePickerOpen ? 'open' : ''}" id="cowatchLanguagePicker">
              <div class="cowatch-language-header">
                <span>${this.t('language')}</span>
                <button class="cowatch-language-close" id="cowatchLanguageClose">
                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="18" y1="6" x2="6" y2="18"/>
                    <line x1="6" y1="6" x2="18" y2="18"/>
                  </svg>
                </button>
              </div>
              <div class="cowatch-language-list">
                ${CHAT_LANGUAGES.map(lang => `
                  <button class="cowatch-language-item ${lang.code === this.language ? 'active' : ''}" data-lang="${lang.code}">
                    <span class="cowatch-language-flag">${lang.flag}</span>
                    <span class="cowatch-language-name">${lang.name}</span>
                    ${lang.code === this.language ? '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="20 6 9 17 4 12"/></svg>' : ''}
                  </button>
                `).join('')}
              </div>
            </div>
            
            <!-- Quick Reaction Bar - Teleparty Style -->
            <div class="cowatch-quick-reaction-bar" id="cowatchQuickReactions">
              ${QUICK_REACTIONS.map(emoji => `
                <button class="cowatch-quick-reaction" data-quick-emoji="${emoji}" title="Send ${emoji} reaction">
                  ${emoji}
                </button>
              `).join('')}
            </div>
            
            <!-- Typing Indicator -->
            <div class="cowatch-typing-indicator" id="cowatchTypingIndicator" style="display: none;">
              <span class="cowatch-typing-dots">
                <span></span><span></span><span></span>
              </span>
              <span class="cowatch-typing-text"></span>
            </div>
            
            <!-- Buffering Indicator -->
            <div class="cowatch-buffering-indicator" id="cowatchBufferingIndicator" style="display: none;">
              <span class="cowatch-buffering-icon">⏳</span>
              <span class="cowatch-buffering-text"></span>
            </div>
            
            <div class="cowatch-input-row">
              ${this.recordedAudio ? `
                <!-- Recorded audio ready to send -->
                <div class="cowatch-recorded-audio">
                  <button class="cowatch-recorded-cancel" id="cowatchCancelRecording" title="Cancel">
                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <line x1="18" y1="6" x2="6" y2="18"/>
                      <line x1="6" y1="6" x2="18" y2="18"/>
                    </svg>
                  </button>
                  <div class="cowatch-recorded-info">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/>
                    </svg>
                    <span class="cowatch-recorded-duration">${this.formatDuration(this.recordedDuration || 0)}</span>
                  </div>
                  <button class="cowatch-recorded-send" id="cowatchSendRecording" title="Send voice message">
                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/>
                    </svg>
                  </button>
                </div>
              ` : this.isRecording ? `
                <!-- Currently recording -->
                <div class="cowatch-recording-bar">
                  <div class="cowatch-recording-indicator">
                    <span class="cowatch-recording-dot"></span>
                    <span class="cowatch-recording-time" id="cowatchRecordingTime">0:00</span>
                  </div>
                  <button class="cowatch-recording-stop" id="cowatchStopRecording" title="Stop recording">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                      <rect x="6" y="6" width="12" height="12" rx="2"/>
                    </svg>
                  </button>
                </div>
              ` : `
                <!-- Normal input -->
                <div class="cowatch-input-actions">
                  <button class="cowatch-input-action ${this.emojiPickerOpen ? 'active' : ''}" id="cowatchEmojiToggle" title="Reactions">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <circle cx="12" cy="12" r="10"/>
                      <path d="M8 14s1.5 2 4 2 4-2 4-2"/>
                      <line x1="9" y1="9" x2="9.01" y2="9"/>
                      <line x1="15" y1="9" x2="15.01" y2="9"/>
                    </svg>
                  </button>
                  <button class="cowatch-input-action ${this.gifPickerOpen ? 'active' : ''}" id="cowatchGifToggle" title="GIFs">
                    <span class="cowatch-gif-label">GIF</span>
                  </button>
                  <button class="cowatch-input-action" id="cowatchMic" title="Voice message">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/>
                      <path d="M19 10v2a7 7 0 0 1-14 0v-2"/>
                      <line x1="12" y1="19" x2="12" y2="23"/>
                      <line x1="8" y1="23" x2="16" y2="23"/>
                    </svg>
                  </button>
                </div>
                <div class="cowatch-input-wrapper">
                  <input type="text" id="cowatchChatInput" placeholder="Type a message..." maxlength="500" autocomplete="off" autocorrect="off" autocapitalize="off" spellcheck="false" data-form-type="other" data-lpignore="true" data-1p-ignore="true">
                  <button class="cowatch-send-btn" id="cowatchSendBtn">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                      <path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/>
                    </svg>
                  </button>
                </div>
              `}
            </div>
          </div>
        </div>
      </div>
      ${this.renderInviteModal()}
      ${this.renderScheduledEventsModal()}
      ${this.renderCreateEventModal()}
    `;
  }

  renderParticipantsPanel() {
    const selectedAvatar = this.userAvatar || AVATARS[0];
    
    // Build participants list with current user's updated info
    let participantsList = this.participants.length > 0 ? this.participants.map(p => {
      // If this is the current user, use their current username and avatar
      if (p.odaId === this.odaId || p.odaKey === this.odaKey || p.userId === this.odaId) {
        return {
          ...p,
          username: this.username,
          avatar: this.userAvatar || p.avatar
        };
      }
      return p;
    }) : [
      { username: this.username, avatar: selectedAvatar, isHost: this.isHost }
    ];
    
    return `
      <div class="cowatch-header">
        <div class="cowatch-header-left">
          <button class="cowatch-back-btn" id="cowatchBack">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M19 12H5M12 19l-7-7 7-7"/>
            </svg>
          </button>
          <h1>Participants</h1>
        </div>
        <span class="cowatch-header-badge">${participantsList.length}</span>
      </div>
      
      <div class="cowatch-content cowatch-panel-content">
        <div class="cowatch-participants-list">
          ${participantsList.map(p => {
            // Check if this is current user by odaId match
            const isCurrentUser = p.odaId === this.odaId || p.odaKey === this.odaKey || p.userId === this.odaId;
            const displayName = isCurrentUser ? this.username : (p.username || 'Guest');
            const displayAvatar = isCurrentUser ? (this.userAvatar || p.avatar || AVATARS[0]) : (p.avatar || AVATARS[0]);
            const avatarColor = (displayAvatar && displayAvatar.color) || '#6366f1';
            const avatarEmoji = (displayAvatar && displayAvatar.emoji) || '😊';
            const hasCustomImage = displayAvatar && displayAvatar.customImage;
            
            return `
              <div class="cowatch-participant-item ${p.isHost ? 'host' : ''}">
                <div class="avatar" style="background: ${hasCustomImage ? 'transparent' : avatarColor}; overflow: hidden;">${hasCustomImage ? `<img src="${displayAvatar.customImage}" style="width:100%;height:100%;object-fit:cover;border-radius:50%;">` : avatarEmoji}</div>
                <div class="info">
                  <div class="name">${displayName}${isCurrentUser ? ' (You)' : ''}</div>
                  <div class="role">${p.isHost ? 'Host' : 'Watching'}</div>
                </div>
                ${p.isHost ? `
                  <div class="cowatch-host-badge">
                    <svg width="14" height="14" viewBox="0 0 24 24" fill="currentColor">
                      <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                    </svg>
                  </div>
                ` : ''}
              </div>
            `;
          }).join('')}
        </div>
      </div>
    `;
  }

  renderProfileEditPanel() {
    const selectedAvatar = this.userAvatar || AVATARS[0];
    const hasCustomImage = selectedAvatar.customImage;
    // Show current username if set, otherwise leave empty for guest
    const displayUsername = (this.username && this.username !== 'Guest') ? this.username : '';
    
    return `
      <div class="cowatch-header">
        <div class="cowatch-header-left">
          <button class="cowatch-back-btn" id="cowatchBack">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M19 12H5M12 19l-7-7 7-7"/>
            </svg>
          </button>
          <h1>${this.t('editProfile')}</h1>
        </div>
      </div>
      
      <div class="cowatch-content cowatch-panel-content">
        <div class="cowatch-profile-edit">
          <div class="cowatch-avatar-preview-large ${hasCustomImage ? 'custom' : ''}" 
               id="avatarPreviewLarge"
               style="background: ${hasCustomImage ? 'transparent' : (selectedAvatar.color || '#6366f1')};">
            ${hasCustomImage ? `<img src="${selectedAvatar.customImage}" alt="Avatar" style="width: 100%; height: 100%; object-fit: cover; border-radius: inherit;">` : (selectedAvatar.emoji || '😊')}
          </div>
          
          <input type="text" class="cowatch-input" id="cowatchEditUsername" 
            placeholder="${this.t('yourName')}" 
            value="${displayUsername}"
            maxlength="20">
          
          <div class="cowatch-section-label">${this.t('uploadImage')}</div>
          
          <div class="cowatch-custom-image-section">
            <input type="file" id="cowatchCustomImage" accept="image/*" style="display: none;">
            <button class="cowatch-btn cowatch-btn-secondary" id="cowatchUploadImage">
              <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                <polyline points="17 8 12 3 7 8"/>
                <line x1="12" y1="3" x2="12" y2="15"/>
              </svg>
              ${this.t('uploadImage')}
            </button>
            ${hasCustomImage ? `
              <div class="cowatch-image-size-control">
                <label>Size:</label>
                <input type="range" id="cowatchImageSize" min="50" max="150" value="${selectedAvatar.imageSize || 100}">
                <span id="cowatchImageSizeValue">${selectedAvatar.imageSize || 100}%</span>
              </div>
              <button class="cowatch-btn-text" id="cowatchRemoveImage">Remove custom image</button>
            ` : ''}
          </div>
          
          <div class="cowatch-section-label">${this.t('selectAvatar')}</div>
          
          <div class="cowatch-avatar-grid" id="avatarSelector">
            ${AVATARS.map((avatar, index) => `
              <button class="cowatch-avatar-option ${!hasCustomImage && avatar.emoji === selectedAvatar.emoji ? 'selected' : ''}" 
                data-avatar-index="${index}"
                style="background: ${avatar.color};">
                ${avatar.emoji}
              </button>
            `).join('')}
          </div>
          
          <button class="cowatch-btn cowatch-btn-primary" id="cowatchSaveProfile">
            ${this.t('saveChanges')}
          </button>
        </div>
      </div>
    `;
  }

  renderParticipantsModal() {
    // Deprecated - using panel instead
    return '';
  }

  renderMessage(msg) {
    if (msg.type === 'system') {
      return `
        <div class="cowatch-message system">
          <div class="cowatch-message-text">${msg.text}</div>
        </div>
      `;
    }

    // Use current user's avatar and username if this is their message
    const isOwnMessage = msg.userId === this.userId;
    const displayAvatar = isOwnMessage ? (this.userAvatar || msg.avatar || AVATARS[0]) : (msg.avatar || AVATARS[0]);
    const displayUsername = isOwnMessage ? this.username : (msg.username || 'Guest');
    const hasCustomImage = displayAvatar && displayAvatar.customImage;
    
    const msgId = msg.id || Date.now();
    
    // Check if message is a GIF
    const isGif = msg.text && msg.text.startsWith('GIF:');
    const gifUrl = isGif ? msg.text.substring(4) : null;
    
    // Check if message is an audio
    const isAudio = msg.text && msg.text.startsWith('AUDIO:');
    const audioUrl = isAudio ? msg.text.substring(6) : null;
    
    // Check for reply
    const hasReply = msg.replyTo;

    // Build avatar HTML with fallback
    const avatarColor = (displayAvatar && displayAvatar.color) || '#6366f1';
    const avatarEmoji = (displayAvatar && displayAvatar.emoji) || '😊';
    const avatarContent = hasCustomImage 
      ? `<img src="${displayAvatar.customImage}" alt="Avatar" style="width: 100%; height: 100%; object-fit: cover; border-radius: inherit; transform: scale(${(displayAvatar.imageSize || 100) / 100});">`
      : avatarEmoji;

    return `
      <div class="cowatch-message ${isOwnMessage ? 'own' : ''}" data-msg-id="${msgId}">
        <div class="cowatch-message-avatar ${hasCustomImage ? 'custom' : ''}" style="background: ${hasCustomImage ? 'transparent' : avatarColor}; overflow: hidden;">
          ${avatarContent}
        </div>
        <div class="cowatch-message-content">
          <div class="cowatch-message-header">
            <span class="cowatch-message-username">${displayUsername}${isOwnMessage ? ' (You)' : ''}</span>
            <span class="cowatch-message-time">${this.formatTime(msg.timestamp)}</span>
          </div>
          ${hasReply ? `
            <div class="cowatch-message-reply">
              ↩ ${msg.replyTo.userId === this.userId ? this.username : msg.replyTo.username}: ${(msg.replyTo.text || '').substring(0, 40)}${(msg.replyTo.text || '').length > 40 ? '...' : ''}
            </div>
          ` : ''}
          ${isGif ? `
            <img src="${gifUrl}" class="cowatch-message-gif" alt="GIF" loading="lazy">
          ` : isAudio ? `
            <div class="cowatch-voice-message" data-audio="${audioUrl}" data-duration="${msg.audioDuration || 0}">
              <button class="cowatch-voice-play">
                <svg class="play-icon" width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                  <polygon points="5 3 19 12 5 21 5 3"/>
                </svg>
                <svg class="pause-icon" width="16" height="16" viewBox="0 0 24 24" fill="currentColor" style="display:none;">
                  <rect x="6" y="4" width="4" height="16"/>
                  <rect x="14" y="4" width="4" height="16"/>
                </svg>
              </button>
              <div class="cowatch-voice-progress-container">
                <div class="cowatch-voice-progress">
                  <div class="cowatch-voice-progress-bar"></div>
                  <div class="cowatch-voice-waveform">
                    ${Array(20).fill(0).map(() => `<span style="height: ${Math.random() * 80 + 20}%"></span>`).join('')}
                  </div>
                </div>
                <div class="cowatch-voice-times">
                  <span class="cowatch-voice-current">0:00</span>
                  <span class="cowatch-voice-duration">${this.formatDuration(msg.audioDuration || 0)}</span>
                </div>
              </div>
            </div>
          ` : `
            <div class="cowatch-message-text">${this.escapeHtml(msg.text)}</div>
          `}
        </div>
        <div class="cowatch-message-actions">
          <button class="cowatch-message-action-btn" data-action="reply" data-msg-id="${msgId}" data-user-id="${msg.userId || ''}" data-username="${displayUsername}" data-text="${this.escapeHtml(msg.text || '').substring(0, 50)}" title="Reply">
            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <path d="M3 10h10a8 8 0 0 1 8 8v4M3 10l6 6M3 10l6-6"/>
            </svg>
          </button>
        </div>
      </div>
    `;
  }

  bindEvents() {
    const container = document.getElementById('cowatch-container');
    if (!container) return;

    // Prevent duplicate event listeners
    if (container._cowatchEventsbound) {
      // Only rebind non-delegated events that need fresh references
      this.bindNonDelegatedEvents(container);
      return;
    }
    container._cowatchEventsbound = true;

    // Event delegation for dynamically rendered elements (only added once)
    container.addEventListener('click', (e) => {
      const target = e.target;
      
      // Back button (for panels) - use event delegation
      if (target.closest('#cowatchBack') || target.closest('.cowatch-back-btn')) {
        e.preventDefault();
        e.stopPropagation();
        this.currentPanel = 'chat';
        this.updateUI(true);
        return;
      }
      
      // Save profile button - use event delegation
      if (target.closest('#cowatchSaveProfile')) {
        e.preventDefault();
        e.stopPropagation();
        const usernameInput = container.querySelector('#cowatchEditUsername');
        const newUsername = usernameInput?.value.trim();
        
        if (newUsername || this.username) {
          if (newUsername) {
            this.username = newUsername;
          }
          this.saveState();
          this.currentPanel = 'chat';
          this.updateUI(true);
          
          // Notify server of profile update
          if (this.socket && this.socket.readyState === WebSocket.OPEN) {
            this.socket.send(JSON.stringify({
              type: 'user:update',
              userId: this.userId,
              username: this.username,
              avatar: this.userAvatar
            }));
          }
        }
        return;
      }
      
      // Upload image button
      if (target.closest('#cowatchUploadImage')) {
        container.querySelector('#cowatchCustomImage')?.click();
        return;
      }
      
      // Remove custom image
      if (target.closest('#cowatchRemoveImage')) {
        if (this.userAvatar) {
          delete this.userAvatar.customImage;
          delete this.userAvatar.imageSize;
        }
        this.updateUI(true);
        return;
      }
      
      // Profile edit toggle
      if (target.closest('#cowatchProfileEdit')) {
        this.currentPanel = 'profile';
        this.updateUI(true);
        return;
      }
      
      // Participants panel toggle
      if (target.closest('#cowatchParticipants')) {
        this.currentPanel = this.currentPanel === 'participants' ? 'chat' : 'participants';
        this.updateUI(true);
        return;
      }
      
      // Invite friends modal toggle
      if (target.closest('#cowatchInviteFriends')) {
        this.inviteModalOpen = true;
        this.inviteActiveTab = 'link';
        this.loadFriendsList();
        this.updateUI(true);
        return;
      }
      
      // Close invite modal
      if (target.closest('#cowatchCloseInviteModal') || target.classList.contains('cowatch-invite-modal-overlay')) {
        this.inviteModalOpen = false;
        this.updateUI(true);
        return;
      }
      
      // Invite modal tab switching
      if (target.closest('.cowatch-invite-tab')) {
        const tab = target.closest('.cowatch-invite-tab').dataset.tab;
        if (tab) {
          this.inviteActiveTab = tab;
          this.updateUI(true);
        }
        return;
      }
      
      // Copy invite link button
      if (target.closest('#cowatchCopyInviteLink')) {
        const inviteLink = `${window.location.origin}/join/${this.roomCode}`;
        navigator.clipboard.writeText(inviteLink).then(() => {
          const btn = document.querySelector('#cowatchCopyInviteLink');
          if (btn) {
            btn.textContent = 'Copied!';
            btn.classList.add('copied');
            setTimeout(() => {
              btn.textContent = 'Copy';
              btn.classList.remove('copied');
            }, 2000);
          }
        });
        return;
      }
      
      // Share buttons
      if (target.closest('.cowatch-share-btn')) {
        const btn = target.closest('.cowatch-share-btn');
        const inviteLink = `${window.location.origin}/join/${this.roomCode}`;
        const text = `Join my watch party on CoWatch! ${inviteLink}`;
        
        if (btn.classList.contains('whatsapp')) {
          window.open(`https://wa.me/?text=${encodeURIComponent(text)}`, '_blank');
        } else if (btn.classList.contains('telegram')) {
          window.open(`https://t.me/share/url?url=${encodeURIComponent(inviteLink)}&text=${encodeURIComponent('Join my watch party on CoWatch!')}`, '_blank');
        } else if (btn.classList.contains('twitter')) {
          window.open(`https://twitter.com/intent/tweet?text=${encodeURIComponent(text)}`, '_blank');
        }
        return;
      }
      
      // Invite friend button
      if (target.closest('.cowatch-friend-invite-btn')) {
        const btn = target.closest('.cowatch-friend-invite-btn');
        const friendId = btn.dataset.friendId;
        if (friendId && !this.invitedFriends.has(friendId)) {
          this.inviteFriend(friendId);
        }
        return;
      }
      
      // Voice message play button
      if (target.closest('.cowatch-voice-play')) {
        this.playVoiceMessage(target.closest('.cowatch-voice-play'));
        return;
      }
      
      // Microphone button - start recording
      if (target.closest('#cowatchMic')) {
        e.preventDefault();
        e.stopPropagation();
        this.startRecording();
        return;
      }
      
      // Stop recording button
      if (target.closest('#cowatchStopRecording')) {
        e.preventDefault();
        e.stopPropagation();
        this.stopRecording();
        return;
      }
      
      // Send recorded audio button
      if (target.closest('#cowatchSendRecording')) {
        e.preventDefault();
        e.stopPropagation();
        this.sendRecordedAudio();
        return;
      }
      
      // Cancel recorded audio button
      if (target.closest('#cowatchCancelRecording')) {
        e.preventDefault();
        e.stopPropagation();
        this.cancelRecording();
        return;
      }
      
      // Reply button
      if (target.closest('.cowatch-message-action-btn[data-action="reply"]')) {
        const btn = target.closest('.cowatch-message-action-btn');
        this.replyingTo = {
          messageId: btn.dataset.msgId,
          userId: btn.dataset.userId,
          username: btn.dataset.username,
          text: btn.dataset.text
        };
        this.updateUI(true);
        container.querySelector('#cowatchChatInput')?.focus();
        return;
      }
      
      // Cancel reply
      if (target.closest('.cowatch-reply-cancel')) {
        this.replyingTo = null;
        const replyBar = container.querySelector('.cowatch-reply-bar');
        replyBar?.remove();
        return;
      }
    });

    // Bind non-delegated events
    this.bindNonDelegatedEvents(container);
  }

  bindNonDelegatedEvents(container) {
    // Close button
    container.querySelector('#cowatchClose')?.addEventListener('click', () => {
      this.togglePanel(false);
    });

    // Create room
    container.querySelector('#cowatchCreate')?.addEventListener('click', () => {
      const nameInput = container.querySelector('#cowatchUsername');
      if (nameInput?.value.trim()) {
        this.username = nameInput.value.trim();
      }
      this.createRoom();
    });

    // Join room
    container.querySelector('#cowatchJoin')?.addEventListener('click', () => {
      const nameInput = container.querySelector('#cowatchUsername');
      const codeInput = container.querySelector('#cowatchRoomCode');
      
      if (nameInput?.value.trim()) {
        this.username = nameInput.value.trim();
      }
      
      const code = codeInput?.value.toUpperCase().trim();
      if (code && code.length >= 4) {
        this.connectToRoom(code);
      } else {
        codeInput.style.borderColor = '#ef4444';
        codeInput.focus();
      }
    });

    // Enter key on room code
    container.querySelector('#cowatchRoomCode')?.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        container.querySelector('#cowatchJoin')?.click();
      }
    });

    // Copy link
    container.querySelector('#cowatchCopy')?.addEventListener('click', () => {
      const link = `${COWATCH_URL}/room/${this.roomCode}`;
      navigator.clipboard.writeText(link);
      const btn = container.querySelector('#cowatchCopy');
      const originalHTML = btn.innerHTML;
      btn.innerHTML = `<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="#4ade80" stroke-width="2"><polyline points="20 6 9 17 4 12"/></svg>`;
      setTimeout(() => {
        btn.innerHTML = originalHTML;
      }, 1500);
    });

    // Reset to video - scroll video into view and focus
    container.querySelector('#cowatchResetVideo')?.addEventListener('click', () => {
      if (this.video) {
        this.video.scrollIntoView({ behavior: 'smooth', block: 'center' });
        // Also try to focus the video player area
        setTimeout(() => {
          const videoContainer = this.video.closest('.watch-video, .NFPlayer, .btm-media-client-element, [data-testid="video-player"], .player-container, .html5-video-player');
          if (videoContainer) {
            videoContainer.focus?.();
          }
        }, 500);
        // Show a brief visual feedback
        const btn = container.querySelector('#cowatchResetVideo');
        if (btn) {
          btn.style.background = 'rgba(74, 222, 128, 0.2)';
          btn.style.borderColor = '#4ade80';
          setTimeout(() => {
            btn.style.background = '';
            btn.style.borderColor = '';
          }, 500);
        }
      }
    });

    // Leave room
    container.querySelector('#cowatchLeave')?.addEventListener('click', () => {
      this.leaveRoom();
    });

    // Minimize button
    container.querySelector('#cowatchMinimize')?.addEventListener('click', () => {
      this.togglePanel(false);
    });

    // Emoji picker toggle
    container.querySelector('#cowatchEmojiToggle')?.addEventListener('click', () => {
      this.emojiPickerOpen = !this.emojiPickerOpen;
      this.gifPickerOpen = false;
      
      const emojiPicker = container.querySelector('#cowatchEmojiPicker');
      const gifPicker = container.querySelector('#cowatchGifPicker');
      
      emojiPicker?.classList.toggle('open', this.emojiPickerOpen);
      gifPicker?.classList.remove('open');
    });

    // Emoji buttons - send floating emoji
    container.querySelectorAll('.cowatch-emoji-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        const emoji = btn.dataset.emoji;
        this.sendFloatingEmoji(emoji);
        this.emojiPickerOpen = false;
        container.querySelector('#cowatchEmojiPicker')?.classList.remove('open');
      });
    });

    // Quick reaction bar buttons - Teleparty style instant reactions
    container.querySelectorAll('.cowatch-quick-reaction').forEach(btn => {
      btn.addEventListener('click', () => {
        const emoji = btn.dataset.quickEmoji;
        if (emoji) {
          // Send on-screen reaction to everyone in the room
          this.sendOnScreenReaction(emoji);
          
          // Visual feedback - scale animation
          btn.style.transform = 'scale(1.3)';
          setTimeout(() => {
            btn.style.transform = '';
          }, 150);
        }
      });
    });

    // Animated reaction buttons - send floating GIF reaction
    container.querySelectorAll('.cowatch-animated-reaction').forEach(btn => {
      btn.addEventListener('click', () => {
        const reactionId = btn.dataset.reactionId;
        const reactionGif = btn.dataset.reactionGif;
        this.sendFloatingReaction(reactionId, reactionGif);
        this.emojiPickerOpen = false;
        container.querySelector('#cowatchEmojiPicker')?.classList.remove('open');
      });
    });

    // Send chat message
    const sendMessage = () => {
      const input = container.querySelector('#cowatchChatInput');
      const text = input?.value.trim();
      if (text && this.socket) {
        this.socket.send(JSON.stringify({
          type: 'chat:message',
          text,
          userId: this.userId,
          username: this.username,
          avatar: this.userAvatar,
          replyTo: this.replyingTo
        }));
        input.value = '';
        
        // Clear reply bar without full refresh
        if (this.replyingTo) {
          this.replyingTo = null;
          const replyBar = container.querySelector('.cowatch-reply-bar');
          replyBar?.remove();
        }
      }
    };

    container.querySelector('#cowatchSendBtn')?.addEventListener('click', sendMessage);
    container.querySelector('#cowatchChatInput')?.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') sendMessage();
    });
    
    // === TELEPARTY-STYLE TYPING INDICATOR ===
    // Send typing status when user types
    const chatInput = container.querySelector('#cowatchChatInput');
    if (chatInput) {
      chatInput.addEventListener('input', () => {
        // Send typing: true
        this.sendTypingStatus(true);
        
        // Clear previous timeout
        if (this.typingTimeout) {
          clearTimeout(this.typingTimeout);
        }
        
        // Set timeout to send typing: false after 2 seconds of no input
        this.typingTimeout = setTimeout(() => {
          this.sendTypingStatus(false);
        }, 2000);
      });
      
      // Also clear typing on blur
      chatInput.addEventListener('blur', () => {
        if (this.typingTimeout) {
          clearTimeout(this.typingTimeout);
        }
        this.sendTypingStatus(false);
      });
    }
    
    // Close emoji/gif picker when clicking outside (only add once)
    if (!this._outsideClickBound) {
      this._outsideClickBound = true;
      document.addEventListener('click', (e) => {
        const container = document.getElementById('cowatch-container');
        if (!container) return;
        
        const emojiPicker = container.querySelector('#cowatchEmojiPicker');
        const emojiToggle = container.querySelector('#cowatchEmojiToggle');
        const gifPicker = container.querySelector('#cowatchGifPicker');
        const gifToggle = container.querySelector('#cowatchGifToggle');
        
        // Close emoji picker if clicking outside
        if (emojiPicker?.classList.contains('open') && 
            !emojiPicker.contains(e.target) && 
            !emojiToggle?.contains(e.target)) {
          emojiPicker.classList.remove('open');
          this.emojiPickerOpen = false;
        }
        
        // Close gif picker if clicking outside
        if (gifPicker?.classList.contains('open') && 
            !gifPicker.contains(e.target) && 
            !gifToggle?.contains(e.target)) {
          gifPicker.classList.remove('open');
          this.gifPickerOpen = false;
        }
      });
    }

    // Profile setup - Continue button
    const setupUsername = container.querySelector('#cowatchSetupUsername');
    const continueBtn = container.querySelector('#cowatchContinue');
    
    setupUsername?.addEventListener('input', () => {
      const value = setupUsername.value.trim();
      continueBtn.disabled = value.length < 2;
    });
    
    continueBtn?.addEventListener('click', () => {
      const name = setupUsername?.value.trim();
      if (name && name.length >= 2) {
        this.username = name;
        this.showProfileSetup = false; // Clear setup flag
        this.saveState();
        // After profile setup, create room automatically
        this.createRoom();
      }
    });

    // Avatar selector
    container.querySelectorAll('.cowatch-avatar-option').forEach(btn => {
      btn.addEventListener('click', () => {
        const index = parseInt(btn.dataset.avatarIndex);
        this.userAvatar = AVATARS[index];
        
        // Update preview (both small and large)
        const preview = container.querySelector('#avatarPreview');
        if (preview) {
          preview.style.background = this.userAvatar.color;
          preview.textContent = this.userAvatar.emoji;
        }
        
        const previewLarge = container.querySelector('.cowatch-avatar-preview-large');
        if (previewLarge) {
          previewLarge.style.background = this.userAvatar.color;
          previewLarge.textContent = this.userAvatar.emoji;
        }
        
        // Update selection state
        container.querySelectorAll('.cowatch-avatar-option').forEach(b => b.classList.remove('selected'));
        btn.classList.add('selected');
        
        this.saveState();
      });
    });

    // GIF picker toggle
    container.querySelector('#cowatchGifToggle')?.addEventListener('click', () => {
      this.gifPickerOpen = !this.gifPickerOpen;
      this.emojiPickerOpen = false;
      
      const gifPicker = container.querySelector('#cowatchGifPicker');
      const emojiPicker = container.querySelector('#cowatchEmojiPicker');
      
      gifPicker?.classList.toggle('open', this.gifPickerOpen);
      emojiPicker?.classList.remove('open');
      
      // Load trending gifs if opening
      if (this.gifPickerOpen) {
        this.loadGifs('trending');
      }
    });

    // GIF category buttons
    container.querySelectorAll('.cowatch-gif-category').forEach(btn => {
      btn.addEventListener('click', () => {
        container.querySelectorAll('.cowatch-gif-category').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        this.loadGifs(btn.dataset.category);
      });
    });

    // Cancel reply - handle dynamically since reply bar is added/removed
    container.addEventListener('click', (e) => {
      if (e.target.id === 'cowatchCancelReply' || e.target.closest('#cowatchCancelReply')) {
        this.replyingTo = null;
        container.querySelector('.cowatch-reply-bar')?.remove();
      }
    });

    // Bind reply buttons immediately
    this.bindReplyButtons();

    // Voice message play buttons
    container.querySelectorAll('.cowatch-voice-play').forEach(btn => {
      btn.addEventListener('click', () => {
        this.playVoiceMessage(btn);
      });
    });

    // Participants panel toggle
    // Theme toggle - use dedicated method to avoid full re-render
    container.querySelector('#cowatchTheme')?.addEventListener('click', () => {
      this.toggleTheme();
    });

    // Language toggle
    container.querySelector('#cowatchLanguage')?.addEventListener('click', () => {
      this.languagePickerOpen = !this.languagePickerOpen;
      this.emojiPickerOpen = false;
      this.gifPickerOpen = false;
      const langPicker = container.querySelector('#cowatchLanguagePicker');
      const emojiPicker = container.querySelector('#cowatchEmojiPicker');
      const gifPicker = container.querySelector('#cowatchGifPicker');
      langPicker?.classList.toggle('open', this.languagePickerOpen);
      emojiPicker?.classList.remove('open');
      gifPicker?.classList.remove('open');
    });

    // Language close button
    container.querySelector('#cowatchLanguageClose')?.addEventListener('click', () => {
      this.languagePickerOpen = false;
      container.querySelector('#cowatchLanguagePicker')?.classList.remove('open');
    });

    // Language selection
    container.querySelectorAll('.cowatch-language-item').forEach(btn => {
      btn.addEventListener('click', () => {
        const lang = btn.dataset.lang;
        if (lang && lang !== this.language) {
          this.language = lang;
          this.languagePickerOpen = false;
          this.saveState();
          
          // Save to localStorage for frontend sync
          try {
            localStorage.setItem('cowatch-language', lang);
          } catch (e) {}
          
          this.updateUI(true);
        }
      });
    });

    container.querySelector('#cowatchCustomImage')?.addEventListener('change', (e) => {
      const file = e.target.files?.[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = (event) => {
          const imageData = event.target?.result;
          if (imageData) {
            this.userAvatar = {
              ...this.userAvatar,
              customImage: imageData,
              imageSize: 100
            };
            this.updateUI(true);
          }
        };
        reader.readAsDataURL(file);
      }
    });

    // Image size slider
    container.querySelector('#cowatchImageSize')?.addEventListener('input', (e) => {
      const size = parseInt(e.target.value);
      if (this.userAvatar) {
        this.userAvatar.imageSize = size;
        const sizeValue = container.querySelector('#cowatchImageSizeValue');
        if (sizeValue) sizeValue.textContent = `${size}%`;
        
        // Update preview
        const preview = container.querySelector('.cowatch-avatar-preview-large img');
        if (preview) {
          preview.style.transform = `scale(${size / 100})`;
        }
      }
    });

    // Remove custom image
    container.querySelector('#cowatchRemoveImage')?.addEventListener('click', () => {
      if (this.userAvatar) {
        delete this.userAvatar.customImage;
        delete this.userAvatar.imageSize;
      }
      this.updateUI(true);
    });

    // GIF search
    container.querySelector('#cowatchGifSearch')?.addEventListener('input', (e) => {
      const query = e.target.value.trim();
      if (query.length >= 2) {
        clearTimeout(this.gifSearchTimeout);
        this.gifSearchTimeout = setTimeout(() => {
          this.loadGifs(query);
        }, 500);
      }
    });
  }

  async startRecording() {
    try {
      console.log('startRecording: Getting user media...');
      this.recordingStream = await navigator.mediaDevices.getUserMedia({ audio: true });
      console.log('startRecording: Got stream, creating MediaRecorder...');
      this.mediaRecorder = new MediaRecorder(this.recordingStream);
      this.audioChunks = [];
      this.recordingStartTime = Date.now();
      this.recordedAudio = null;
      this.recordedDuration = 0;
      
      this.mediaRecorder.ondataavailable = (e) => {
        if (e.data.size > 0) {
          this.audioChunks.push(e.data);
        }
      };
      
      this.mediaRecorder.onstop = () => {
        const recordingDuration = Math.floor((Date.now() - this.recordingStartTime) / 1000);
        console.log('🎙️ onstop triggered, duration:', recordingDuration, 'chunks:', this.audioChunks.length);
        
        // Stop the stream tracks
        if (this.recordingStream) {
          this.recordingStream.getTracks().forEach(track => track.stop());
          this.recordingStream = null;
        }
        
        if (this.audioChunks.length === 0) {
          console.error('🎙️ No audio chunks recorded');
          this.recordedAudio = null;
          this.recordedDuration = 0;
          this.updateUI(true);
          return;
        }
        
        // Store the recorded audio for later sending
        this.recordedAudio = new Blob(this.audioChunks, { type: 'audio/webm' });
        this.recordedDuration = recordingDuration;
        console.log('🎙️ Audio blob created, size:', this.recordedAudio.size, 'recordedAudio set:', !!this.recordedAudio);
        
        // Force UI update to show recorded audio bar
        console.log('🎙️ Calling updateUI to show recorded audio bar');
        this.updateUI(true);
        console.log('🎙️ updateUI called, recordedAudio still set:', !!this.recordedAudio);
      };
      
      // Start recording with timeslice
      this.mediaRecorder.start(100);
      this.isRecording = true;
      
      // Update UI to show recording bar
      this.updateUI(true);
      
      // Start recording timer
      this.updateRecordingTimer();
    } catch (error) {
      console.error('Failed to start recording:', error);
    }
  }

  updateRecordingTimer() {
    if (!this.isRecording) return;
    
    const elapsed = Math.floor((Date.now() - this.recordingStartTime) / 1000);
    const mins = Math.floor(elapsed / 60);
    const secs = elapsed % 60;
    const timeStr = `${mins}:${secs.toString().padStart(2, '0')}`;
    
    // Update the recording time display
    const recordingTime = document.querySelector('#cowatchRecordingTime');
    if (recordingTime) {
      recordingTime.textContent = timeStr;
    }
    
    // Continue updating every second
    setTimeout(() => this.updateRecordingTimer(), 1000);
  }

  stopRecording() {
    console.log('stopRecording called, mediaRecorder:', !!this.mediaRecorder, 'isRecording:', this.isRecording);
    if (this.mediaRecorder && this.isRecording) {
      console.log('Stopping MediaRecorder...');
      this.mediaRecorder.stop();
      this.isRecording = false;
      // onstop callback will handle the rest and update UI
    }
  }

  async sendRecordedAudio() {
    if (!this.recordedAudio) {
      console.error('No recorded audio to send');
      return;
    }
    
    console.log('Sending recorded audio, size:', this.recordedAudio.size, 'duration:', this.recordedDuration);
    await this.sendVoiceMessage(this.recordedAudio, this.recordedDuration);
    
    // Clear recorded audio state
    this.recordedAudio = null;
    this.recordedDuration = 0;
    this.updateUI(true);
  }

  cancelRecording() {
    console.log('Canceling recording');
    
    // Stop recording if still in progress
    if (this.mediaRecorder && this.isRecording) {
      this.mediaRecorder.stop();
      this.isRecording = false;
    }
    
    // Stop the stream tracks
    if (this.recordingStream) {
      this.recordingStream.getTracks().forEach(track => track.stop());
      this.recordingStream = null;
    }
    
    this.recordedAudio = null;
    this.recordedDuration = 0;
    this.audioChunks = [];
    this.updateUI(true);
  }

  playVoiceMessage(btn) {
    const voiceMsg = btn.closest('.cowatch-voice-message');
    if (!voiceMsg) return;
    
    const audioUrl = voiceMsg.dataset.audio;
    if (!audioUrl) return;
    
    const playIcon = btn.querySelector('.play-icon');
    const pauseIcon = btn.querySelector('.pause-icon');
    const progressBar = voiceMsg.querySelector('.cowatch-voice-progress-bar');
    const currentTimeEl = voiceMsg.querySelector('.cowatch-voice-current');
    const durationEl = voiceMsg.querySelector('.cowatch-voice-duration');
    
    // If already has audio and playing, pause it
    if (voiceMsg._audio && !voiceMsg._audio.paused) {
      voiceMsg._audio.pause();
      btn.classList.remove('playing');
      voiceMsg.classList.remove('playing');
      if (playIcon) playIcon.style.display = 'block';
      if (pauseIcon) pauseIcon.style.display = 'none';
      return;
    }
    
    // Stop any other playing audio
    document.querySelectorAll('.cowatch-voice-message').forEach(vm => {
      if (vm._audio && vm !== voiceMsg) {
        vm._audio.pause();
        vm._audio.currentTime = 0;
        vm.classList.remove('playing');
        const pb = vm.querySelector('.cowatch-voice-play');
        if (pb) {
          pb.classList.remove('playing');
          const pi = pb.querySelector('.play-icon');
          const pai = pb.querySelector('.pause-icon');
          if (pi) pi.style.display = 'block';
          if (pai) pai.style.display = 'none';
        }
        const bar = vm.querySelector('.cowatch-voice-progress-bar');
        if (bar) bar.style.width = '0%';
        const curTime = vm.querySelector('.cowatch-voice-current');
        if (curTime) curTime.textContent = '0:00';
      }
    });
    
    // Create audio if not exists
    if (!voiceMsg._audio) {
      voiceMsg._audio = new Audio(audioUrl);
      
      voiceMsg._audio.addEventListener('loadedmetadata', () => {
        const duration = voiceMsg._audio.duration;
        if (durationEl && !isNaN(duration) && isFinite(duration)) {
          const mins = Math.floor(duration / 60);
          const secs = Math.floor(duration % 60);
          durationEl.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
        }
      });
      
      voiceMsg._audio.addEventListener('timeupdate', () => {
        const current = voiceMsg._audio.currentTime;
        const duration = voiceMsg._audio.duration;
        if (progressBar && duration && isFinite(duration)) {
          progressBar.style.width = `${(current / duration) * 100}%`;
        }
        if (currentTimeEl && !isNaN(current)) {
          const mins = Math.floor(current / 60);
          const secs = Math.floor(current % 60);
          currentTimeEl.textContent = `${mins}:${secs.toString().padStart(2, '0')}`;
        }
      });
      
      voiceMsg._audio.addEventListener('ended', () => {
        btn.classList.remove('playing');
        voiceMsg.classList.remove('playing');
        if (playIcon) playIcon.style.display = 'block';
        if (pauseIcon) pauseIcon.style.display = 'none';
        if (progressBar) progressBar.style.width = '0%';
        if (currentTimeEl) currentTimeEl.textContent = '0:00';
      });
      
      // Allow seeking by clicking on progress bar
      const progressContainer = voiceMsg.querySelector('.cowatch-voice-progress');
      if (progressContainer) {
        progressContainer.addEventListener('click', (e) => {
          if (voiceMsg._audio && voiceMsg._audio.duration) {
            const rect = progressContainer.getBoundingClientRect();
            const clickPos = (e.clientX - rect.left) / rect.width;
            voiceMsg._audio.currentTime = clickPos * voiceMsg._audio.duration;
          }
        });
      }
    }
    
    // Play
    btn.classList.add('playing');
    voiceMsg.classList.add('playing');
    if (playIcon) playIcon.style.display = 'none';
    if (pauseIcon) pauseIcon.style.display = 'block';
    
    voiceMsg._audio.play().catch(err => {
      console.error('Error playing audio:', err);
      btn.classList.remove('playing');
      voiceMsg.classList.remove('playing');
      if (playIcon) playIcon.style.display = 'block';
      if (pauseIcon) pauseIcon.style.display = 'none';
    });
  }

  async sendVoiceMessage(audioBlob, duration = 0) {
    // Convert to base64 for sending
    const reader = new FileReader();
    reader.onloadend = () => {
      const base64Audio = reader.result;
      console.log('Sending voice message, socket status:', this.socket?.readyState);
      if (this.socket && this.socket.readyState === WebSocket.OPEN) {
        this.socket.send(JSON.stringify({
          type: 'chat:message',
          text: `AUDIO:${base64Audio}`,
          audioDuration: duration,
          userId: this.userId,
          username: this.username,
          avatar: this.userAvatar
        }));
        console.log('Voice message sent successfully');
      } else {
        console.error('Socket not connected, cannot send voice message');
        // Show error to user
        const chatInput = document.querySelector('#cowatchChatInput');
        if (chatInput) {
          chatInput.placeholder = 'Connection error, try again...';
          setTimeout(() => {
            chatInput.placeholder = 'Type a message...';
          }, 2000);
        }
      }
    };
    reader.readAsDataURL(audioBlob);
  }

  async loadGifs(searchTerm) {
    const grid = document.querySelector('#cowatchGifGrid');
    if (!grid) return;
    
    grid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 20px; color: rgba(255,255,255,0.5);">Loading GIFs...</div>';
    
    try {
      // Using Tenor API - you would need an API key for production
      const TENOR_API_KEY = 'AIzaSyAyimkuYQYF_FXVALexPuGQctUWRURdCYQ'; // Public demo key
      const response = await fetch(`https://tenor.googleapis.com/v2/search?q=${encodeURIComponent(searchTerm)}&key=${TENOR_API_KEY}&limit=16&media_filter=tinygif,gif`);
      const data = await response.json();
      
      if (data.results && data.results.length > 0) {
        grid.innerHTML = data.results.map(gif => {
          const gifUrl = gif.media_formats.tinygif?.url || gif.media_formats.gif?.url;
          return `
            <div class="cowatch-gif-item" data-gif-url="${gifUrl}">
              <img src="${gifUrl}" alt="GIF" loading="lazy">
            </div>
          `;
        }).join('');
        
        // Add click handlers for GIF items
        grid.querySelectorAll('.cowatch-gif-item').forEach(item => {
          item.addEventListener('click', () => {
            const gifUrl = item.dataset.gifUrl;
            this.sendGif(gifUrl);
          });
        });
      } else {
        grid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 20px; color: rgba(255,255,255,0.5);">No GIFs found</div>';
      }
    } catch (error) {
      console.error('Error loading GIFs:', error);
      grid.innerHTML = '<div style="grid-column: 1/-1; text-align: center; padding: 20px; color: rgba(255,255,255,0.5);">Failed to load GIFs</div>';
    }
  }

  sendGif(gifUrl) {
    if (this.socket) {
      const msg = {
        type: 'chat:message',
        text: `GIF:${gifUrl}`,
        userId: this.userId,
        username: this.username,
        avatar: this.userAvatar,
        replyTo: this.replyingTo
      };
      this.socket.send(JSON.stringify(msg));
      this.replyingTo = null;
      this.gifPickerOpen = false;
      
      // Just close the picker without full refresh
      const gifPicker = document.querySelector('#cowatchGifPicker');
      gifPicker?.classList.remove('open');
      
      // Clear reply bar if exists
      const replyBar = document.querySelector('.cowatch-reply-bar');
      replyBar?.remove();
    }
  }

  // Send on-screen animated reaction (Teleparty style)
  sendOnScreenReaction(emoji) {
    // Check if reactions are disabled
    if (this.reactionsDisabled) {
      console.log('🎬 Reactions are disabled for this party');
      return;
    }
    
    // Send to server for broadcast to all participants
    if (this.socket) {
      this.socket.send(JSON.stringify({
        type: 'reaction',
        emoji,
        username: this.username,
        userId: this.userId
      }));
    }
    
    // Show locally immediately for responsive feel
    this.showOnScreenReaction(emoji);
  }

  sendFloatingEmoji(emoji) {
    // Check if reactions are disabled
    if (this.reactionsDisabled) {
      console.log('🎬 Reactions are disabled for this party');
      return;
    }
    // Send to server
    if (this.socket) {
      this.socket.send(JSON.stringify({
        type: 'emoji:reaction',
        emoji,
        username: this.username
      }));
    }
    // Show locally too
    this.showFloatingEmoji(emoji);
  }

  sendFloatingReaction(reactionId, gifUrl) {
    // Check if reactions are disabled
    if (this.reactionsDisabled) {
      console.log('🎬 Reactions are disabled for this party');
      return;
    }
    // Send to server
    if (this.socket) {
      this.socket.send(JSON.stringify({
        type: 'gif:reaction',
        reactionId,
        gifUrl,
        username: this.username
      }));
    }
    // Show locally too
    this.showFloatingReaction(gifUrl);
  }

  showFloatingReaction(gifUrl) {
    // Find the video element to position reaction over it
    const video = this.video || document.querySelector('video');
    if (!video) return;
    
    const videoRect = video.getBoundingClientRect();
    
    // Random X position within video area (with padding)
    const padding = 100;
    const randomX = videoRect.left + padding + Math.random() * (videoRect.width - padding * 2);
    
    // Start from bottom of video
    const startY = videoRect.bottom - 80;
    
    const floatingReaction = document.createElement('div');
    floatingReaction.className = 'cowatch-floating-reaction';
    
    const img = document.createElement('img');
    img.src = gifUrl;
    img.alt = 'reaction';
    floatingReaction.appendChild(img);
    
    floatingReaction.style.left = `${randomX}px`;
    floatingReaction.style.top = `${startY}px`;
    
    // Add random horizontal movement
    floatingReaction.style.setProperty('--drift', `${(Math.random() - 0.5) * 150}px`);
    
    document.body.appendChild(floatingReaction);
    
    // Remove after animation (4 seconds)
    setTimeout(() => {
      floatingReaction.remove();
    }, 4000);
  }

  showFloatingEmoji(emoji) {
    // Find the video element to position emoji over it
    const video = this.video || document.querySelector('video');
    if (!video) return;
    
    const videoRect = video.getBoundingClientRect();
    
    // Random X position within video area (with padding)
    const padding = 80;
    const randomX = videoRect.left + padding + Math.random() * (videoRect.width - padding * 2);
    
    // Start from bottom of video
    const startY = videoRect.bottom - 60;
    
    const floatingEmoji = document.createElement('div');
    floatingEmoji.className = 'cowatch-floating-emoji';
    
    // Add wiggle animation for laughing/happy emojis
    const wiggleEmojis = ['😂', '🤣', '😆', '😅', '🥳', '🤪', '😜', '😝', '😀', '😁', '😄', '🎉'];
    if (wiggleEmojis.includes(emoji)) {
      floatingEmoji.classList.add('wiggle');
    }
    
    floatingEmoji.textContent = emoji;
    floatingEmoji.style.left = `${randomX}px`;
    floatingEmoji.style.top = `${startY}px`;
    
    // Add more random horizontal movement - smaller drift like Teleparty
    floatingEmoji.style.setProperty('--drift', `${(Math.random() - 0.5) * 60}px`);
    
    document.body.appendChild(floatingEmoji);
    
    // Remove after animation (3 seconds - faster like Teleparty)
    setTimeout(() => {
      floatingEmoji.remove();
    }, 3000);
  }

  // ========== SYNC STATUS OVERLAY ==========
  showSyncStatus(message, isSuccess = false) {
    // Remove existing sync overlay
    this.hideSyncStatus();
    
    const overlay = document.createElement('div');
    overlay.className = 'cowatch-sync-overlay' + (isSuccess ? ' cowatch-sync-success' : '');
    overlay.id = 'cowatch-sync-overlay';
    overlay.innerHTML = `
      <div class="cowatch-sync-spinner"></div>
      <span class="cowatch-sync-text">${message}</span>
    `;
    
    document.body.appendChild(overlay);
    
    // Auto-hide after delay
    if (this.syncStatusTimeout) {
      clearTimeout(this.syncStatusTimeout);
    }
    
    this.syncStatusTimeout = setTimeout(() => {
      this.hideSyncStatus();
    }, isSuccess ? 2000 : 4000);
  }
  
  hideSyncStatus() {
    const overlay = document.getElementById('cowatch-sync-overlay');
    if (overlay) {
      overlay.classList.add('hiding');
      setTimeout(() => overlay.remove(), 300);
    }
    if (this.syncStatusTimeout) {
      clearTimeout(this.syncStatusTimeout);
      this.syncStatusTimeout = null;
    }
  }

  // ========== IN-PAGE MESSAGE DIALOG ==========
  showMessageDialog(options) {
    const { type = 'info', title, message, avatar, duration = 4000 } = options;
    
    // Remove existing dialog
    this.hideMessageDialog();
    
    const dialog = document.createElement('div');
    dialog.className = `cowatch-message-dialog ${type}`;
    dialog.id = 'cowatch-message-dialog';
    
    const avatarHtml = avatar 
      ? `<div class="cowatch-message-dialog-avatar" style="background: ${avatar.color || '#ff3b30'}">${avatar.emoji || '👤'}</div>`
      : '';
    
    dialog.innerHTML = `
      ${avatarHtml}
      <div class="cowatch-message-dialog-content">
        <div class="cowatch-message-dialog-title">${title}</div>
        ${message ? `<div class="cowatch-message-dialog-text">${message}</div>` : ''}
      </div>
      <button class="cowatch-message-dialog-close" onclick="this.parentElement.remove()">
        <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <line x1="18" y1="6" x2="6" y2="18"></line>
          <line x1="6" y1="6" x2="18" y2="18"></line>
        </svg>
      </button>
    `;
    
    document.body.appendChild(dialog);
    
    // Auto-hide after delay
    if (this.messageDialogTimeout) {
      clearTimeout(this.messageDialogTimeout);
    }
    
    this.messageDialogTimeout = setTimeout(() => {
      this.hideMessageDialog();
    }, duration);
  }
  
  hideMessageDialog() {
    const dialog = document.getElementById('cowatch-message-dialog');
    if (dialog) {
      dialog.classList.add('hiding');
      setTimeout(() => dialog.remove(), 300);
    }
    if (this.messageDialogTimeout) {
      clearTimeout(this.messageDialogTimeout);
      this.messageDialogTimeout = null;
    }
  }

  // ========== DRAGGABLE/RESIZABLE PANEL ==========
  initDraggablePanel() {
    const container = document.getElementById('cowatch-container');
    if (!container) return;
    
    // Add drag handle
    let dragHandle = container.querySelector('.cowatch-drag-handle');
    if (!dragHandle) {
      dragHandle = document.createElement('div');
      dragHandle.className = 'cowatch-drag-handle';
      container.insertBefore(dragHandle, container.firstChild);
    }
    
    // Add resize handle
    let resizeHandle = container.querySelector('.cowatch-resize-handle-left');
    if (!resizeHandle) {
      resizeHandle = document.createElement('div');
      resizeHandle.className = 'cowatch-resize-handle cowatch-resize-handle-left';
      container.appendChild(resizeHandle);
    }
    
    let startX, startY, startLeft, startTop, startWidth;
    
    // Drag functionality
    const startDrag = (e) => {
      if (e.target !== dragHandle) return;
      
      this.isDragging = true;
      container.classList.add('dragging');
      
      const rect = container.getBoundingClientRect();
      startX = e.clientX;
      startY = e.clientY;
      startLeft = rect.left;
      startTop = rect.top;
      
      document.addEventListener('mousemove', onDrag);
      document.addEventListener('mouseup', stopDrag);
      e.preventDefault();
    };
    
    const onDrag = (e) => {
      if (!this.isDragging) return;
      
      const deltaX = e.clientX - startX;
      const deltaY = e.clientY - startY;
      
      let newLeft = startLeft + deltaX;
      let newTop = startTop + deltaY;
      
      // Constrain to viewport
      const maxLeft = window.innerWidth - container.offsetWidth;
      const maxTop = window.innerHeight - container.offsetHeight;
      
      newLeft = Math.max(0, Math.min(newLeft, maxLeft));
      newTop = Math.max(0, Math.min(newTop, maxTop));
      
      this.panelPosition = { x: newLeft, y: newTop };
      
      container.style.left = `${newLeft}px`;
      container.style.top = `${newTop}px`;
      container.style.right = 'auto';
    };
    
    const stopDrag = () => {
      this.isDragging = false;
      container.classList.remove('dragging');
      document.removeEventListener('mousemove', onDrag);
      document.removeEventListener('mouseup', stopDrag);
    };
    
    dragHandle.addEventListener('mousedown', startDrag);
    
    // Resize functionality
    const startResize = (e) => {
      this.isResizing = true;
      container.classList.add('resizing');
      
      startX = e.clientX;
      startWidth = container.offsetWidth;
      
      document.addEventListener('mousemove', onResize);
      document.addEventListener('mouseup', stopResize);
      e.preventDefault();
    };
    
    const onResize = (e) => {
      if (!this.isResizing) return;
      
      const deltaX = startX - e.clientX;
      let newWidth = startWidth + deltaX;
      
      // Constrain width
      newWidth = Math.max(300, Math.min(newWidth, 600));
      
      this.panelSize.width = newWidth;
      container.style.width = `${newWidth}px`;
      
      // Adjust position if needed
      if (this.panelPosition.x !== null) {
        const maxLeft = window.innerWidth - newWidth;
        if (this.panelPosition.x > maxLeft) {
          this.panelPosition.x = maxLeft;
          container.style.left = `${maxLeft}px`;
        }
      }
    };
    
    const stopResize = () => {
      this.isResizing = false;
      container.classList.remove('resizing');
      document.removeEventListener('mousemove', onResize);
      document.removeEventListener('mouseup', stopResize);
    };
    
    resizeHandle.addEventListener('mousedown', startResize);
    
    // Reset position on double-click
    dragHandle.addEventListener('dblclick', () => {
      this.panelPosition = { x: null, y: null };
      this.panelSize = { width: 380, height: null };
      container.style.left = '';
      container.style.top = '';
      container.style.right = '0';
      container.style.width = '340px';
    });
  }

  // ========== DOM MONITORING (Smart Video Detection) ==========
  initVideoObserver() {
    // Disconnect existing observer
    if (this.videoObserver) {
      this.videoObserver.disconnect();
    }
    
    // Create observer for video elements
    this.videoObserver = new MutationObserver((mutations) => {
      for (const mutation of mutations) {
        // Check added nodes for videos
        for (const node of mutation.addedNodes) {
          if (node.nodeType === Node.ELEMENT_NODE) {
            if (node.tagName === 'VIDEO') {
              this.onVideoFound(node);
            } else if (node.querySelector) {
              const video = node.querySelector('video');
              if (video) {
                this.onVideoFound(video);
              }
            }
          }
        }
        
        // Check removed nodes
        for (const node of mutation.removedNodes) {
          if (node.nodeType === Node.ELEMENT_NODE) {
            if (node.tagName === 'VIDEO' && node === this.video) {
              this.onVideoLost();
            } else if (node.querySelector && node.contains(this.video)) {
              this.onVideoLost();
            }
          }
        }
      }
    });
    
    // Start observing
    this.videoObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
    
    console.log('🎬 Video observer initialized');
  }
  
  onVideoFound(video) {
    if (this.video === video) return;
    
    console.log('🎬 Video element found:', video);
    this.video = video;
    
    // Re-setup video listeners
    this.setupVideoListeners();
    
    // Show toggle button if in room
    if (this.roomCode && !this.chatOpen) {
      this.createToggleButton();
    }
  }
  
  onVideoLost() {
    console.log('🎬 Video element lost');
    this.video = null;
    
    // Hide toggle button when video is gone
    if (!this.chatOpen) {
      this.removeToggleButton();
    }
  }

  togglePanel(open) {
    console.log('🎬 togglePanel called with:', open);
    const container = document.getElementById('cowatch-container');
    
    console.log('🎬 Container exists:', !!container);
    
    if (open === undefined) {
      open = !container?.classList.contains('open');
    }

    console.log('🎬 Setting panel open:', open);
    this.chatOpen = open;

    if (open) {
      container?.classList.add('open');
      // Apply transform for iframe container
      if (container) {
        container.style.transform = 'translateX(0)';
      }
      document.body.classList.add('cowatch-panel-open');
      
      // Remove toggle button when panel is open
      this.removeToggleButton();
      
      // Reset unread count when opening
      this.unreadCount = 0;
    } else {
      container?.classList.remove('open');
      // Apply transform for iframe container
      if (container) {
        container.style.transform = 'translateX(100%)';
      }
      document.body.classList.remove('cowatch-panel-open');
      
      // Create toggle button only if user is in a room
      if (this.roomCode) {
        this.createToggleButton();
      }
    }
    
    // Resize video for chat panel (Teleparty style - video shrinks, not overlay)
    this.resizeVideoForChat(open);
    
    // Notify platform-specific scripts about chat visibility (for video resize in fullscreen)
    this.notifyChatVisibility(open);
    
    // Save chat open state
    this.saveState();
  }
  
  // Notify injected scripts about chat panel visibility
  // Used by YouTube to resize video in fullscreen mode (Teleparty approach)
  notifyChatVisibility(visible) {
    const host = window.location.hostname;
    
    if (host.includes('youtube.com')) {
      // Send message to YouTube injected script
      window.postMessage({
        source: 'cowatch-content',
        action: 'setChatVisible',
        data: { visible }
      }, '*');
      console.log('🎬 Notified YouTube about chat visibility:', visible);
    }
    // Add other platforms here as needed
  }

  updateToggleBadge() {
    const toggle = document.getElementById('cowatch-toggle');
    if (!toggle) return;
    
    // Remove existing unread badge
    const existingBadge = toggle.querySelector('.unread-badge');
    existingBadge?.remove();
    
    // Add new badge if there are unread messages and chat is closed
    if (this.unreadCount > 0 && !this.chatOpen) {
      const badge = document.createElement('span');
      badge.className = 'unread-badge';
      badge.textContent = this.unreadCount > 9 ? '9+' : this.unreadCount;
      toggle.appendChild(badge);
    }
  }

  updateUI(forceFullRender = false) {
    // In iframe mode, sidebar handles its own UI - skip embedded UI updates
    if (this.useIframeChat) {
      return;
    }
    
    const container = document.getElementById('cowatch-container');
    
    if (container) {
      // Force full render when recording state changes or when explicitly requested
      const recordingStateChanged = this._lastRenderState && (
        this._lastRenderState.isRecording !== this.isRecording ||
        this._lastRenderState.hasRecordedAudio !== !!this.recordedAudio
      );
      
      // Only do full render when necessary (room state changes, recording state changes)
      if (forceFullRender || recordingStateChanged || !this._lastRenderState || 
          this._lastRenderState.roomCode !== this.roomCode ||
          this._lastRenderState.isConnected !== this.isConnected ||
          this._lastRenderState.currentPanel !== this.currentPanel) {
        
        console.log('🔄 Full UI render, isRecording:', this.isRecording, 'hasRecordedAudio:', !!this.recordedAudio);
        
        const themeClass = this.theme === 'light' ? 'light' : '';
        const wasOpen = container.classList.contains('open');
        
        container.innerHTML = this.renderPanel();
        container.className = wasOpen ? `open ${themeClass}`.trim() : themeClass;
        this.bindEvents();
        
        this._lastRenderState = {
          roomCode: this.roomCode,
          isConnected: this.isConnected,
          currentPanel: this.currentPanel,
          isRecording: this.isRecording,
          hasRecordedAudio: !!this.recordedAudio
        };
      } else {
        // Partial updates - just update what changed
        this.updateChat();
        this.updateParticipantCount();
      }
    }
  }

  updateParticipantCount() {
    const btn = document.querySelector('#cowatchParticipants');
    if (btn) {
      const count = this.participants.length || 1;
      btn.innerHTML = `
        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
          <circle cx="9" cy="7" r="4"/>
          <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
          <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
        </svg>
        <span class="cowatch-participant-count">${count}</span>
      `;
    }
  }

  toggleTheme() {
    this.theme = this.theme === 'dark' ? 'light' : 'dark';
    const container = document.getElementById('cowatch-container');
    if (container) {
      container.classList.toggle('light', this.theme === 'light');
    }
    // Update theme button icon
    const themeBtn = document.querySelector('#cowatchTheme');
    if (themeBtn) {
      themeBtn.innerHTML = this.theme === 'dark' ? `
        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <circle cx="12" cy="12" r="5"/>
          <line x1="12" y1="1" x2="12" y2="3"/>
          <line x1="12" y1="21" x2="12" y2="23"/>
          <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/>
          <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/>
          <line x1="1" y1="12" x2="3" y2="12"/>
          <line x1="21" y1="12" x2="23" y2="12"/>
          <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/>
          <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/>
        </svg>
      ` : `
        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"/>
        </svg>
      `;
      themeBtn.classList.toggle('active', this.theme === 'light');
    }
    this.saveState();
  }

  generateRoomCode() {
    const chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    let code = '';
    for (let i = 0; i < 6; i++) {
      code += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return code;
  }

  // Teleparty approach: Video must be ready before creating room
  async waitVideoReady(timeout = 10000) {
    const startTime = Date.now();
    
    return new Promise((resolve, reject) => {
      const checkVideo = () => {
        // Check if we have a valid video element
        if (this.video && !isNaN(this.video.duration) && this.video.duration > 0) {
          console.log('🎬 Video is ready:', this.video.duration);
          resolve(true);
          return;
        }
        
        // Check timeout
        if (Date.now() - startTime > timeout) {
          console.log('🎬 Video ready timeout');
          resolve(false); // Don't reject, just indicate video not ready
          return;
        }
        
        // Retry
        setTimeout(checkVideo, 200);
      };
      
      checkVideo();
    });
  }

  // Get video metadata (like Teleparty)
  getVideoMetadata() {
    const video = this.video;
    const url = window.location.href;
    
    let videoId = null;
    let videoTitle = document.title;
    let videoDuration = video?.duration || 0;
    
    // Extract video ID based on platform
    const host = window.location.hostname;
    
    if (host.includes('netflix.com')) {
      const match = url.match(/watch\/(\d+)/);
      videoId = match ? match[1] : null;
      videoTitle = videoTitle.replace(' | Netflix', '').replace(' - Netflix', '');
    } else if (host.includes('youtube.com')) {
      const urlParams = new URLSearchParams(window.location.search);
      videoId = urlParams.get('v');
      videoTitle = document.querySelector('h1.ytd-video-primary-info-renderer')?.textContent || 
                   document.querySelector('h1.title')?.textContent ||
                   videoTitle.replace(' - YouTube', '');
    } else if (host.includes('disneyplus.com')) {
      const match = url.match(/video\/([a-f0-9-]+)/i);
      videoId = match ? match[1] : null;
    } else if (host.includes('primevideo.com') || host.includes('amazon.com')) {
      const match = url.match(/detail\/([A-Z0-9]+)/i) || url.match(/gp\/video\/detail\/([A-Z0-9]+)/i);
      videoId = match ? match[1] : null;
    } else if (host.includes('max.com') || host.includes('hbomax.com')) {
      const match = url.match(/play\/([a-f0-9-]+)/i) || url.match(/video\/([a-f0-9-]+)/i);
      videoId = match ? match[1] : null;
    } else if (host.includes('hulu.com')) {
      const match = url.match(/watch\/([a-f0-9-]+)/i);
      videoId = match ? match[1] : null;
    }
    
    return {
      videoId,
      videoTitle,
      videoDuration,
      videoUrl: url,
      platform: this.platform.name,
      serviceDomain: host
    };
  }

  async createRoom() {
    // Check if profile is set up - if not, show setup panel first
    if (!this.username || this.username === 'Guest') {
      this.showProfileSetup = true;
      this.updateUI(true);
      return;
    }
    
    // Teleparty approach: Wait for video to be ready before creating room
    console.log('🎬 Checking if video is ready...');
    
    const videoReady = await this.waitVideoReady(5000);
    if (!videoReady) {
      console.log('🎬 Warning: Video not fully ready, proceeding anyway');
    }
    
    // Get video metadata
    const metadata = this.getVideoMetadata();
    console.log('🎬 Creating room with video metadata:', metadata);
    
    const code = this.generateRoomCode();
    this.isHost = true;
    
    // Store metadata for room creation
    this._roomMetadata = metadata;
    
    await this.connectToRoom(code);
  }

  async connectToRoom(roomCode) {
    // In iframe mode, sidebar handles WebSocket connection
    if (this.useIframeChat) {
      console.log('🎬 Iframe mode: Skipping embedded WebSocket connection');
      return;
    }
    
    this.roomCode = roomCode;
    
    if (this.socket) {
      this.socket.close();
    }

    try {
      this.socket = new WebSocket(SOCKET_URL);

      this.socket.onopen = () => {
        console.log('🎬 CoWatch: Connected to server');
        this.isConnected = true;
        
        // Notify background that we have an active room
        const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
        browserAPI.runtime.sendMessage({ type: 'ROOM_ACTIVE' }).catch(() => {});
        
        // Get current video metadata
        const metadata = this._roomMetadata || this.getVideoMetadata();
        
        // Send create or join based on isHost (Teleparty approach with full metadata)
        this.socket.send(JSON.stringify({
          type: this.isHost ? 'room:create' : 'room:join',
          roomCode: this.roomCode,
          username: this.username,
          avatar: this.userAvatar,
          isHost: this.isHost,
          // Teleparty-style video metadata
          videoUrl: metadata.videoUrl || window.location.href,
          videoId: metadata.videoId,
          videoTitle: metadata.videoTitle,
          videoDuration: metadata.videoDuration,
          platform: this.platform.name,
          serviceDomain: metadata.serviceDomain,
          // Room settings (Teleparty style)
          createSettings: {
            controlLock: this.hostOnlyControl,
            reactionsDisabled: this.reactionsDisabled
          }
        }));

        // Clear stored metadata
        this._roomMetadata = null;
        
        this.saveState();
        this.updateUI();
      };

      this.socket.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data);
          this.handleMessage(data);
        } catch (e) {
          console.error('CoWatch: Parse error', e);
        }
      };

      this.socket.onclose = (event) => {
        console.log('🎬 CoWatch: Disconnected', event.code, event.reason);
        this.isConnected = false;
        
        // Clear server sync interval
        if (this.serverSyncInterval) {
          clearInterval(this.serverSyncInterval);
          this.serverSyncInterval = null;
        }
        
        // Check if we should attempt reconnection
        if (this.roomCode && !this.connectionLost && event.code !== 1000) {
          this.connectionLost = true;
          this.attemptReconnection();
        }
        
        this.updateUI();
      };

      this.socket.onerror = (error) => {
        console.error('🎬 CoWatch: Socket error', error);
      };
    } catch (error) {
      console.error('🎬 CoWatch: Connection failed', error);
    }
  }

  // === WEBSOCKET RECONNECTION (Teleparty-style) ===
  attemptReconnection() {
    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      console.log('🎬 CoWatch: Max reconnection attempts reached');
      this.showReconnectionFailed();
      return;
    }
    
    this.reconnectAttempts++;
    const delay = Math.min(this.reconnectBackoff * Math.pow(2, this.reconnectAttempts - 1), 30000);
    
    console.log(`🎬 CoWatch: Attempting reconnection ${this.reconnectAttempts}/${this.maxReconnectAttempts} in ${delay}ms`);
    
    // Show reconnecting status
    this.showReconnectingStatus();
    
    this.reconnectTimeout = setTimeout(() => {
      if (this.roomCode) {
        this.connectToRoom(this.roomCode);
      }
    }, delay);
  }
  
  showReconnectingStatus() {
    const container = document.getElementById('cowatch-container');
    if (!container) return;
    
    let statusEl = container.querySelector('.cowatch-reconnect-status');
    if (!statusEl) {
      statusEl = document.createElement('div');
      statusEl.className = 'cowatch-reconnect-status';
      statusEl.innerHTML = `
        <div class="cowatch-reconnect-content">
          <div class="cowatch-reconnect-spinner"></div>
          <span>Reconnecting...</span>
        </div>
      `;
      container.appendChild(statusEl);
    }
  }
  
  hideReconnectingStatus() {
    const statusEl = document.querySelector('.cowatch-reconnect-status');
    if (statusEl) statusEl.remove();
  }
  
  showReconnectionFailed() {
    const container = document.getElementById('cowatch-container');
    if (!container) return;
    
    this.hideReconnectingStatus();
    
    const failedEl = document.createElement('div');
    failedEl.className = 'cowatch-reconnect-failed';
    failedEl.innerHTML = `
      <div class="cowatch-reconnect-content">
        <span>Connection lost</span>
        <button id="cowatchManualReconnect">Retry</button>
      </div>
    `;
    container.appendChild(failedEl);
    
    failedEl.querySelector('#cowatchManualReconnect')?.addEventListener('click', () => {
      failedEl.remove();
      this.reconnectAttempts = 0;
      this.attemptReconnection();
    });
  }

  // === SERVER TIME SYNCHRONIZATION (Teleparty-style) ===
  async syncServerTime() {
    if (!this.socket || this.socket.readyState !== WebSocket.OPEN) return;
    
    const startTime = Date.now();
    
    // Send ping request
    this.socket.send(JSON.stringify({
      type: 'ping',
      clientTime: startTime
    }));
  }
  
  handlePongResponse(serverTime, clientTime) {
    const now = Date.now();
    const rtt = now - clientTime; // Round trip time
    const oneWayDelay = rtt / 2;
    
    // Calculate server time offset
    this.serverTimeOffset = serverTime - (clientTime + oneWayDelay);
    
    // Track ping history (keep last 5)
    this.pingHistory.push(rtt);
    if (this.pingHistory.length > 5) {
      this.pingHistory.shift();
    }
    
    // Calculate average ping
    this.avgPing = this.pingHistory.reduce((a, b) => a + b, 0) / this.pingHistory.length;
    
    this.lastServerSync = now;
    console.log(`🎬 CoWatch: Server time synced - offset: ${this.serverTimeOffset}ms, avg ping: ${Math.round(this.avgPing)}ms`);
  }
  
  getServerTime() {
    return Date.now() + this.serverTimeOffset;
  }
  
  startServerTimeSync() {
    // Sync immediately
    this.syncServerTime();
    
    // Then sync every 12.5 seconds (like Teleparty)
    this.serverSyncInterval = setInterval(() => {
      this.syncServerTime();
    }, 12500);
  }

  // === TASK QUEUE SYSTEM (Teleparty-style) ===
  pushTask(fn, name) {
    // Don't add duplicate tasks
    if (name && this.taskNames.has(name)) {
      return;
    }
    
    if (name) this.taskNames.add(name);
    this.taskQueue.push({ fn, name });
    this.processTaskQueue();
  }
  
  async processTaskQueue() {
    if (this.taskInProgress || this.taskQueue.length === 0) return;
    
    this.taskInProgress = true;
    const task = this.taskQueue.shift();
    
    try {
      await task.fn();
    } catch (error) {
      console.error('🎬 CoWatch: Task error:', error);
    }
    
    if (task.name) this.taskNames.delete(task.name);
    this.taskInProgress = false;
    
    // Process next task
    if (this.taskQueue.length > 0) {
      this.processTaskQueue();
    }
  }
  
  hasTaskInQueue(name) {
    return this.taskNames.has(name);
  }
  
  clearTaskQueue() {
    this.taskQueue = [];
    this.taskNames.clear();
    this.taskInProgress = false;
  }

  // === ROOM HISTORY FOR POPUP "LAST ROOMS" ===
  async saveRoomToHistory() {
    if (!this.roomCode) return;
    
    try {
      // Get video title from page
      let videoTitle = '';
      const titleSelectors = [
        'h1', // YouTube
        '.video-title', // Generic
        '[data-testid="video-title"]', // Various
        '.title', // Generic
        document.title // Fallback
      ];
      
      for (const selector of titleSelectors) {
        if (selector === document.title) {
          videoTitle = selector;
          break;
        }
        const el = document.querySelector(selector);
        if (el?.textContent?.trim()) {
          videoTitle = el.textContent.trim();
          break;
        }
      }
      
      // Clean up title
      if (videoTitle) {
        videoTitle = videoTitle.replace(/- YouTube$/, '').replace(/\| Netflix$/, '').trim();
      }
      
      // Get thumbnail (video poster if available)
      let thumbnail = '';
      const video = this.video || document.querySelector('video');
      if (video?.poster) {
        thumbnail = video.poster;
      }
      
      const roomData = {
        code: this.roomCode,
        name: `Watch Party ${this.roomCode}`,
        platform: this.platform?.name || 'video',
        videoTitle: videoTitle || document.title,
        videoUrl: window.location.href,
        thumbnail: thumbnail,
        isHost: this.isHost,
        participantCount: this.participants?.length || 1
      };
      
      // Send to popup via storage
      const data = await new Promise((resolve) => {
        browserAPI.storage.local.get(['lastRooms'], (result) => {
          resolve(result);
        });
      });
      
      let lastRooms = data.lastRooms || [];
      
      // Remove if exists (we'll add to top)
      lastRooms = lastRooms.filter(r => r.code !== roomData.code);
      
      // Add to top
      lastRooms.unshift(roomData);
      
      // Keep only last 10
      lastRooms = lastRooms.slice(0, 10);
      
      await browserAPI.storage.local.set({ lastRooms });
      
      console.log('🎬 CoWatch: Room saved to history:', roomData.code);
    } catch (e) {
      console.warn('🎬 CoWatch: Could not save room to history:', e);
    }
  }

  // === AUTO-REJOIN PARTY ===
  saveAutoRejoinData() {
    if (!this.roomCode) return;
    
    const data = {
      roomCode: this.roomCode,
      username: this.username,
      userAvatar: this.userAvatar,
      isHost: this.isHost,
      timestamp: Date.now(),
      retries: 0
    };
    
    localStorage.setItem('cowatch_autoRejoin', JSON.stringify(data));
  }
  
  clearAutoRejoinData() {
    localStorage.removeItem('cowatch_autoRejoin');
  }
  
  async checkAutoRejoin() {
    const saved = localStorage.getItem('cowatch_autoRejoin');
    if (!saved) return false;
    
    try {
      const data = JSON.parse(saved);
      
      // Check if data is too old (3 hours)
      if (Date.now() - data.timestamp > 3 * 60 * 60 * 1000) {
        this.clearAutoRejoinData();
        return false;
      }
      
      // Check retry limit
      if (data.retries >= this.maxAutoRejoinRetries) {
        this.clearAutoRejoinData();
        return false;
      }
      
      // Update retry count
      data.retries++;
      localStorage.setItem('cowatch_autoRejoin', JSON.stringify(data));
      
      // Restore user data
      this.username = data.username;
      this.userAvatar = data.userAvatar;
      
      console.log('🎬 CoWatch: Auto-rejoining party', data.roomCode);
      
      // Wait a bit for page to stabilize
      await new Promise(r => setTimeout(r, 2000));
      
      // Rejoin
      this.isHost = false; // Always join as non-host on rejoin
      await this.connectToRoom(data.roomCode);
      
      return true;
    } catch (e) {
      console.error('🎬 CoWatch: Auto-rejoin failed', e);
      this.clearAutoRejoinData();
      return false;
    }
  }

  // === EXPECTED STATE CALCULATION (for sync validation) ===
  calculateExpectedTime() {
    if (!this.expectedState.lastUpdate) return this.expectedState.time;
    
    const elapsed = this.expectedState.playing 
      ? (this.getServerTime() - this.expectedState.lastUpdate) / 1000
      : 0;
    
    return this.expectedState.time + elapsed;
  }
  
  updateExpectedState(time, playing) {
    this.expectedState = {
      time,
      playing,
      lastUpdate: this.getServerTime()
    };
  }

  handleMessage(data) {
    console.log('🎬 CoWatch:', data.type);

    switch (data.type) {
      case 'pong':
        // Handle server time sync response
        this.handlePongResponse(data.serverTime, data.clientTime);
        break;
        
      case 'room:joined':
        // Prevent duplicate join handling
        if (this._hasJoinedRoom) {
          console.log('🎬 Already joined room, ignoring duplicate room:joined');
          this.participants = data.participants || this.participants;
          this.updateUI();
          break;
        }
        this._hasJoinedRoom = true;
        
        this.participants = data.participants || [{ username: this.username, isHost: this.isHost }];
        
        // CRITICAL: Get our odaId from participants list (server assigns this)
        const myParticipant = this.participants.find(p => p.username === this.username);
        if (myParticipant && myParticipant.odaId) {
          this.odaId = myParticipant.odaId;
          this.visitorId = myParticipant.odaId; // Sync visitorId too
          console.log('🎬 Got server-assigned odaId:', this.odaId);
        }
        
        this.togglePanel(true); // Open chat sidebar when connected
        this.addSystemMessage('You joined the party!');
        
        // Reset reconnection state
        this.reconnectAttempts = 0;
        this.connectionLost = false;
        this.hideReconnectingStatus();
        
        // Start server time sync
        this.startServerTimeSync();
        
        // Save auto-rejoin data
        this.saveAutoRejoinData();
        
        // Save to room history for popup "Last Rooms" feature
        this.saveRoomToHistory();
        
        this.updateUI();
        
        // Call join callback if it exists (popup waiting for response)
        if (this._joinCallback) {
          this._joinCallback({ success: true, roomName: 'Watch Party' });

          this._joinCallback = null;
          this._joinRoomCode = null;
        }
        break;

      case 'room:error':
        console.error('🎬 CoWatch: Room error:', data.error, data.message);
        
        // Call join callback with error if it exists (popup waiting for response)
        if (this._joinCallback) {
          this._joinCallback({ success: false, error: data.message || 'Room not found' });
          this._joinCallback = null;
          this._joinRoomCode = null;
        }
        
        // Clear room state
        this.roomCode = null;
        this.isHost = false;
        this.isConnected = false;
        this.chatOpen = false;
        
        // Remove toggle button since we're not in a room
        this.removeToggleButton();
        
        this.clearState();
        
        // Show error to user
        alert(data.message || 'Failed to join room');
        
        // Close socket
        if (this.socket) {
          this.socket.close();
          this.socket = null;
        }
        
        // Hide panel
        const container = document.getElementById('cowatch-container');
        if (container) {
          container.classList.remove('open');
        }
        document.body.classList.remove('cowatch-panel-open');
        
        this.updateUI();
        break;

      case 'room:user-joined':
        this.participants = data.participants || [];
        this.addSystemMessage(`${data.username} joined the party`);
        // Show in-page notification when chat is closed
        if (!this.chatOpen) {
          const joiner = this.participants.find(p => p.username === data.username);
          this.showMessageDialog({
            type: 'join',
            title: `${data.username} joined the party`,
            message: `${this.participants.length} ${this.participants.length === 1 ? 'viewer' : 'viewers'} watching`,
            avatar: joiner?.avatar ? { emoji: joiner.avatar.emoji, color: joiner.avatar.color } : null,
            duration: 4000
          });
        }
        this.updateUI();
        break;

      case 'room:user-left':
        this.participants = data.participants || [];
        this.addSystemMessage(`${data.username} left the party`);
        // Show in-page notification when chat is closed
        if (!this.chatOpen) {
          this.showMessageDialog({
            type: 'leave',
            title: `${data.username} left the party`,
            message: `${this.participants.length} ${this.participants.length === 1 ? 'viewer' : 'viewers'} remaining`,
            duration: 3000
          });
        }
        this.updateUI();
        break;

      case 'user:updated':
        // Update participants list with new user info
        this.participants = data.participants || this.participants;
        // Only update UI if we're on participants panel
        if (this.currentPanel === 'participants') {
          this.updateUI(true);
        }
        break;

      case 'video:play':
        if (this.video && this.video.paused) {
          this.ignoreVideoEvents = true;
          // Show sync overlay
          if (data.username && data.username !== this.username) {
            this.showSyncStatus(this.t('syncing') || 'Syncing...');
          }
          if (data.time !== undefined) {
            // Use platform-specific method for seeking
            if (this.platform.name === 'netflix') {
              window.postMessage({ type: 'SEEK', time: data.time * 1000 }, '*');
            } else {
              this.video.currentTime = data.time;
            }
          }
          // Use platform-specific method for play
          if (this.platform.name === 'netflix') {
            window.postMessage({ type: 'PLAY' }, '*');
            setTimeout(() => { 
              this.ignoreVideoEvents = false;
              if (data.username && data.username !== this.username) {
                this.showSyncStatus(this.t('synced') || 'Synced!', true);
              }
            }, 500);
          } else {
            this.video.play().finally(() => {
              setTimeout(() => { 
                this.ignoreVideoEvents = false;
                if (data.username && data.username !== this.username) {
                  this.showSyncStatus(this.t('synced') || 'Synced!', true);
                }
              }, 500);
            });
          }
        }
        // Add system message for video play (show for everyone including self)
        if (data.username) {
          const displayName = data.username === this.username ? 'You' : data.username;
          this.addSystemMessage(`${displayName} resumed playback`);
          // Show in-page notification for other users
          if (data.username !== this.username && !this.chatOpen) {
            const participant = this.participants.find(p => p.username === data.username);
            this.showMessageDialog({
              type: 'sync',
              title: `${data.username} resumed playback`,
              avatar: participant?.avatar ? { emoji: participant.avatar.emoji, color: participant.avatar.color } : null,
              duration: 3000
            });
          }
        }
        break;

      case 'video:pause':
        if (this.video && !this.video.paused) {
          this.ignoreVideoEvents = true;
          // Show sync overlay
          if (data.username && data.username !== this.username) {
            this.showSyncStatus(this.t('syncing') || 'Syncing...');
          }
          // Use platform-specific method for pause
          if (this.platform.name === 'netflix') {
            window.postMessage({ type: 'PAUSE' }, '*');
          } else {
            this.video.pause();
          }
          if (data.time !== undefined) {
            // Use platform-specific method for seeking
            if (this.platform.name === 'netflix') {
              window.postMessage({ type: 'SEEK', time: data.time * 1000 }, '*');
            } else {
              this.video.currentTime = data.time;
            }
          }
          setTimeout(() => { 
            this.ignoreVideoEvents = false;
            if (data.username && data.username !== this.username) {
              this.showSyncStatus(this.t('synced') || 'Synced!', true);
            }
          }, 500);
        }
        // Add system message for video pause (show for everyone including self)
        if (data.username) {
          const displayName = data.username === this.username ? 'You' : data.username;
          this.addSystemMessage(`${displayName} paused playback`);
          // Show in-page notification for other users
          if (data.username !== this.username && !this.chatOpen) {
            const participant = this.participants.find(p => p.username === data.username);
            this.showMessageDialog({
              type: 'sync',
              title: `${data.username} paused playback`,
              avatar: participant?.avatar ? { emoji: participant.avatar.emoji, color: participant.avatar.color } : null,
              duration: 3000
            });
          }
        }
        break;

      case 'video:seek':
        if (this.video && data.time !== undefined) {
          this.ignoreVideoEvents = true;
          // Show sync overlay
          if (data.username && data.username !== this.username) {
            this.showSyncStatus(this.t('syncing') || 'Syncing...');
          }
          // Use platform-specific method for seeking
          if (this.platform.name === 'netflix') {
            window.postMessage({ type: 'SEEK', time: data.time * 1000 }, '*');
          } else {
            this.video.currentTime = data.time;
          }
          setTimeout(() => { 
            this.ignoreVideoEvents = false;
            if (data.username && data.username !== this.username) {
              this.showSyncStatus(this.t('synced') || 'Synced!', true);
            }
          }, 500);
        }
        // Add system message for video seek (show for everyone including self)
        if (data.username) {
          const displayName = data.username === this.username ? 'You' : data.username;
          const formattedTime = this.formatVideoTime(data.time);
          this.addSystemMessage(`${displayName} jumped to ${formattedTime}`);
          // Show in-page notification for other users
          if (data.username !== this.username && !this.chatOpen) {
            const participant = this.participants.find(p => p.username === data.username);
            this.showMessageDialog({
              type: 'sync',
              title: `${data.username} jumped to ${formattedTime}`,
              avatar: participant?.avatar ? { emoji: participant.avatar.emoji, color: participant.avatar.color } : null,
              duration: 3000
            });
          }
        }
        break;

      case 'video:sync':
        // Use task queue to prevent conflicts (Teleparty-style)
        if (this.video && data.time !== undefined) {
          this.pushTask(() => {
            const diff = Math.abs(this.video.currentTime - data.time);
            if (diff > 2) {
              this.ignoreVideoEvents = true;
              this.showSyncStatus(this.t('syncing') || 'Syncing...');
              
              // Update expected state
              this.updateExpectedState(data.time, !this.video.paused);
              
              // Use platform-specific method for seeking
              if (this.platform.name === 'netflix') {
                window.postMessage({ type: 'SEEK', time: data.time * 1000 }, '*');
              } else {
                this.video.currentTime = data.time;
              }
              setTimeout(() => { 
                this.ignoreVideoEvents = false;
                this.showSyncStatus(this.t('synced') || 'Synced!', true);
              }, 500);
            }
          }, 'SYNC');
        }
        break;

      case 'chat:message':
        const newMsg = {
          id: data.id || Date.now(),
          userId: data.userId,
          username: data.username,
          text: data.text,
          avatar: data.avatar,
          replyTo: data.replyTo,
          timestamp: Date.now(),
          type: 'message'
        };
        this.messages.push(newMsg);
        this.saveState(); // Save messages to persist
        
        // Increment unread count only if chat is closed AND message is from someone else
        if (!this.chatOpen && data.userId !== this.userId) {
          this.unreadCount++;
          this.updateToggleBadge();
        }
        
        // Append only new message instead of re-rendering all
        const messagesEl = document.getElementById('cowatchMessages');
        if (messagesEl) {
          const wasNearBottom = messagesEl.scrollHeight - messagesEl.scrollTop <= messagesEl.clientHeight + 100;
          messagesEl.insertAdjacentHTML('beforeend', this.renderMessage(newMsg));
          
          // Bind reply button for new message only
          const newMsgEl = messagesEl.lastElementChild;
          const replyBtn = newMsgEl?.querySelector('.cowatch-message-action-btn[data-action="reply"]');
          if (replyBtn) {
            replyBtn.addEventListener('click', (e) => {
              e.stopPropagation();
              this.handleReplyClick(replyBtn);
            });
          }
          
          if (wasNearBottom) {
            messagesEl.scrollTop = messagesEl.scrollHeight;
          }
        }
        break;

      case 'emoji:reaction':
        // Show floating emoji from other users (Teleparty-style on-screen reaction)
        if (data.username !== this.username && !this.reactionsDisabled) {
          this.showOnScreenReaction(data.emoji);
        }
        break;

      case 'gif:reaction':
        // Show floating GIF reaction from other users
        if (data.username !== this.username && data.gifUrl && !this.reactionsDisabled) {
          this.showOnScreenReaction(null, data.gifUrl);
        }
        break;
        
      // === TELEPARTY-STYLE NEW MESSAGE TYPES ===
      case 'user:typing':
        // Handle typing indicator (Teleparty style)
        // Server sends odaId which is the server-assigned guest ID
        const typingUserId = data.odaId || data.userId;
        
        // Check if this is our own typing (compare with server-assigned odaId)
        const isOwnTyping = !typingUserId || 
                           typingUserId === this.odaId || 
                           typingUserId === this.visitorId ||
                           data.username === this.username;
        
        console.log('🎬 Typing event:', { typingUserId, myOdaId: this.odaId, isOwnTyping, username: data.username });
        
        if (!isOwnTyping && typingUserId) {
          // Clear any existing timeout for this user
          if (this.typingTimeouts && this.typingTimeouts[typingUserId]) {
            clearTimeout(this.typingTimeouts[typingUserId]);
          }
          if (!this.typingTimeouts) this.typingTimeouts = {};
          
          if (data.isTyping) {
            if (!this.usersTyping.includes(typingUserId)) {
              this.usersTyping.push(typingUserId);
            }
            // Auto-clear typing after 3 seconds (shorter timeout)
            this.typingTimeouts[typingUserId] = setTimeout(() => {
              this.usersTyping = this.usersTyping.filter(id => id !== typingUserId);
              this.updateTypingIndicator();
            }, 3000);
          } else {
            this.usersTyping = this.usersTyping.filter(id => id !== typingUserId);
          }
          this.updateTypingIndicator();
        }
        break;
        
      case 'user:buffering':
        // Handle buffering indicator (Teleparty style)
        const bufferingUserId = data.odaId || data.userId;
        const isOwnBuffering = !bufferingUserId ||
                              bufferingUserId === this.odaId ||
                              bufferingUserId === this.visitorId ||
                              data.username === this.username;
        
        if (!isOwnBuffering && bufferingUserId) {
          if (data.isBuffering) {
            if (!this.usersBuffering.includes(bufferingUserId)) {
              this.usersBuffering.push(bufferingUserId);
            }
          } else {
            this.usersBuffering = this.usersBuffering.filter(id => id !== bufferingUserId);
          }
          this.updateBufferingIndicator();
        }
        break;
        
      case 'reaction:sent':
        // On-screen reaction from another user (Teleparty style)
        if (data.userId !== this.userId && !this.reactionsDisabled) {
          this.showOnScreenReaction(data.reactionType);
        }
        break;
        
      case 'gif:sent':
        // GIF message from another user
        const gifMsg = {
          id: data.id || Date.now(),
          userId: data.userId,
          username: data.username,
          avatar: data.avatar,
          gifUrl: data.gifUrl,
          gifTitle: data.gifTitle,
          timestamp: Date.now(),
          type: 'gif'
        };
        this.messages.push(gifMsg);
        this.saveState();
        
        if (!this.chatOpen && data.userId !== this.userId) {
          this.unreadCount++;
          this.updateToggleBadge();
        }
        
        // Append GIF message
        const gifMessagesEl = document.getElementById('cowatchMessages');
        if (gifMessagesEl) {
          gifMessagesEl.insertAdjacentHTML('beforeend', this.renderGifMessage(gifMsg));
          gifMessagesEl.scrollTop = gifMessagesEl.scrollHeight;
        }
        break;
        
      case 'room:settings':
        // Room settings update (controlLock, reactionsDisabled)
        if (data.controlLock !== undefined) {
          this.hostOnlyControl = data.controlLock;
        }
        if (data.reactionsDisabled !== undefined) {
          this.reactionsDisabled = data.reactionsDisabled;
        }
        this.updateUI();
        break;
    }
  }

  // === TELEPARTY-STYLE ON-SCREEN REACTIONS ===
  // Show animated emoji floating up from bottom of video (like Teleparty)
  showOnScreenReaction(emoji, gifUrl = null) {
    // Get or create reaction container
    let container = this.getReactionContainer();
    if (!container) return;
    
    // Calculate random position
    const containerWidth = container.offsetWidth;
    const chatWidth = this.chatOpen ? 340 : 0; // Don't show reactions behind chat
    const maxLeft = containerWidth - chatWidth - 80;
    const randomLeft = Math.floor(Math.random() * maxLeft) + chatWidth;
    
    // Random size (40-80px)
    const size = Math.floor(Math.random() * 40) + 40;
    
    // Random animation variant (1-3)
    const animVariant = Math.ceil(Math.random() * 3);
    
    // Create reaction element
    const reaction = document.createElement('div');
    reaction.className = `cowatch-on-screen-reaction cowatch-reaction-anim-${animVariant}`;
    reaction.style.cssText = `
      position: absolute;
      bottom: 0;
      right: ${randomLeft}px;
      font-size: ${size}px;
      z-index: 9999999999;
      pointer-events: none;
    `;
    
    if (gifUrl) {
      // GIF reaction
      reaction.innerHTML = `<img src="${gifUrl}" style="width: ${size}px; height: ${size}px; border-radius: 8px;">`;
    } else {
      // Emoji reaction
      reaction.textContent = emoji;
    }
    
    container.appendChild(reaction);
    
    // Remove after animation completes (5 seconds)
    setTimeout(() => {
      reaction.remove();
    }, 5000);
  }
  
  // Get or create the reaction container (video overlay area)
  getReactionContainer() {
    let container = document.getElementById('cowatch-reaction-container');
    if (!container) {
      container = document.createElement('div');
      container.id = 'cowatch-reaction-container';
      container.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100vw;
        height: 100vh;
        pointer-events: none;
        z-index: 9999999998;
        overflow: hidden;
      `;
      document.body.appendChild(container);
      
      // Add animation styles
      this.injectReactionAnimationStyles();
    }
    return container;
  }
  
  // Inject CSS animations for on-screen reactions (Teleparty style)
  injectReactionAnimationStyles() {
    if (document.getElementById('cowatch-reaction-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'cowatch-reaction-styles';
    style.textContent = `
      /* Teleparty-style on-screen reaction animations */
      .cowatch-on-screen-reaction {
        animation: cowatch-reaction-float 5s cubic-bezier(0.5, 1, 0.89, 1) forwards;
      }
      
      .cowatch-reaction-anim-1 {
        animation: cowatch-reaction-float 5s cubic-bezier(0.5, 1, 0.89, 1) forwards,
                   cowatch-reaction-wiggle-1 12s cubic-bezier(0.5, 1, 0.89, 1) forwards;
      }
      
      .cowatch-reaction-anim-2 {
        animation: cowatch-reaction-float 6s cubic-bezier(0.5, 1, 0.89, 1) forwards,
                   cowatch-reaction-wiggle-2 12s cubic-bezier(0.5, 1, 0.89, 1) forwards;
      }
      
      .cowatch-reaction-anim-3 {
        animation: cowatch-reaction-float 7s cubic-bezier(0.5, 1, 0.89, 1) forwards,
                   cowatch-reaction-wiggle-3 12s cubic-bezier(0.5, 1, 0.89, 1) forwards;
      }
      
      @keyframes cowatch-reaction-float {
        0% { opacity: 0; transform: translateY(0); }
        20% { opacity: 0.9; }
        30% { opacity: 0.9; }
        90% { opacity: 0; }
        100% { transform: translateY(-100vh); opacity: 0; }
      }
      
      @keyframes cowatch-reaction-wiggle-1 {
        10% { margin-left: -6px; }
        25% { margin-left: 4px; }
        30% { margin-left: -5px; }
        45% { margin-left: 5px; }
        55% { margin-left: -3px; }
        60% { margin-left: 5px; }
        70% { margin-left: -5px; }
        85% { margin-left: 5px; }
        90% { margin-left: -7px; }
        100% { margin-left: 5px; }
      }
      
      @keyframes cowatch-reaction-wiggle-2 {
        15% { margin-left: -2px; }
        20% { margin-left: 5px; }
        35% { margin-left: -6px; }
        40% { margin-left: 5px; }
        50% { margin-left: -5px; }
        65% { margin-left: 5px; }
        70% { margin-left: -5px; }
        80% { margin-left: 4px; }
        95% { margin-left: -5px; }
        100% { margin-left: 5px; }
      }
      
      @keyframes cowatch-reaction-wiggle-3 {
        15% { margin-left: -4px; }
        20% { margin-left: 5px; }
        35% { margin-left: -2px; }
        40% { margin-left: 5px; }
        50% { margin-left: -3px; }
        65% { margin-left: 5px; }
        70% { margin-left: -5px; }
        80% { margin-left: 5px; }
        95% { margin-left: -4px; }
        100% { margin-left: 5px; }
      }
    `;
    document.head.appendChild(style);
  }
  
  // === TYPING INDICATOR (Teleparty style) ===
  sendTypingStatus(isTyping) {
    if (this.socket && this.socket.readyState === WebSocket.OPEN) {
      this.socket.send(JSON.stringify({
        type: 'user:typing',
        roomCode: this.roomCode,
        userId: this.userId,
        username: this.username,
        isTyping: isTyping
      }));
    }
  }
  
  updateTypingIndicator() {
    const indicator = document.getElementById('cowatchTypingIndicator');
    if (!indicator) return;
    
    if (this.usersTyping.length === 0) {
      indicator.style.display = 'none';
      return;
    }
    
    // Get usernames of people typing - match by odaId
    const typingNames = this.usersTyping.map(odaId => {
      const participant = this.participants.find(p => 
        p.odaId === odaId || p.id === odaId || p.visitorId === odaId || p.userId === odaId
      );
      return participant?.username || 'Someone';
    });
    
    let text = '';
    if (typingNames.length === 1) {
      text = `${typingNames[0]} is typing...`;
    } else if (typingNames.length === 2) {
      text = `${typingNames[0]} and ${typingNames[1]} are typing...`;
    } else {
      text = `${typingNames.length} people are typing...`;
    }
    
    const textSpan = indicator.querySelector('.cowatch-typing-text');
    if (textSpan) {
      textSpan.textContent = text;
    }
    indicator.style.display = 'flex';
  }
  
  // === BUFFERING INDICATOR (Teleparty style) ===
  sendBufferingStatus(isBuffering) {
    if (this.socket && this.socket.readyState === WebSocket.OPEN) {
      this.socket.send(JSON.stringify({
        type: 'user:buffering',
        roomCode: this.roomCode,
        userId: this.userId,
        username: this.username,
        isBuffering: isBuffering
      }));
    }
  }
  
  updateBufferingIndicator() {
    const indicator = document.getElementById('cowatchBufferingIndicator');
    if (!indicator) return;
    
    if (this.usersBuffering.length === 0) {
      indicator.style.display = 'none';
      return;
    }
    
    // Get usernames of people buffering (exclude self)
    const bufferingNames = this.usersBuffering
      .filter(userId => userId !== this.userId)
      .map(userId => {
        const participant = this.participants.find(p => 
          p.id === userId || p.visitorId === userId || p.odaId === userId || p.userId === userId
        );
        return participant?.username || 'Someone';
      });
    
    if (bufferingNames.length === 0) {
      indicator.style.display = 'none';
      return;
    }
    
    const textSpan = indicator.querySelector('.cowatch-buffering-text');
    if (textSpan) {
      if (bufferingNames.length === 1) {
        textSpan.textContent = `Waiting for ${bufferingNames[0]} to buffer...`;
      } else {
        textSpan.textContent = `Waiting for ${bufferingNames.length} people to buffer...`;
      }
    }
    indicator.style.display = 'flex';
  }
  
  // === SEND REACTION (Teleparty style) ===
  sendReaction(reactionType) {
    if (this.reactionsDisabled) return;
    
    // Show locally immediately
    this.showOnScreenReaction(reactionType);
    
    // Send to others
    if (this.socket && this.socket.readyState === WebSocket.OPEN) {
      this.socket.send(JSON.stringify({
        type: 'reaction:send',
        roomCode: this.roomCode,
        userId: this.userId,
        username: this.username,
        reactionType: reactionType
      }));
    }
  }
  
  // === GIF SEARCH (Tenor API) ===
  async searchGifs(query) {
    if (!query || query.trim() === '') {
      return this.getTrendingGifs();
    }
    
    try {
      const response = await fetch(
        `${TENOR_API_URL}/search?q=${encodeURIComponent(query)}&key=${TENOR_API_KEY}&limit=20&media_filter=gif,tinygif`
      );
      const data = await response.json();
      this.gifResults = data.results || [];
      this.renderGifResults();
    } catch (error) {
      console.error('🎬 GIF search error:', error);
      this.gifResults = [];
    }
  }
  
  async getTrendingGifs() {
    try {
      const response = await fetch(
        `${TENOR_API_URL}/featured?key=${TENOR_API_KEY}&limit=20&media_filter=gif,tinygif`
      );
      const data = await response.json();
      this.gifResults = data.results || [];
      this.renderGifResults();
    } catch (error) {
      console.error('🎬 Trending GIFs error:', error);
      this.gifResults = [];
    }
  }
  
  renderGifResults() {
    const container = document.getElementById('cowatch-gif-results');
    if (!container) return;
    
    if (this.gifResults.length === 0) {
      container.innerHTML = '<div class="cowatch-gif-empty">No GIFs found</div>';
      return;
    }
    
    container.innerHTML = this.gifResults.map(gif => {
      const gifUrl = gif.media_formats?.gif?.url || gif.media_formats?.tinygif?.url || '';
      const previewUrl = gif.media_formats?.tinygif?.url || gif.media_formats?.nanogif?.url || gifUrl;
      return `
        <div class="cowatch-gif-item" data-gif-url="${gifUrl}" data-gif-title="${gif.title || ''}">
          <img src="${previewUrl}" alt="${gif.title || 'GIF'}" loading="lazy">
        </div>
      `;
    }).join('');
    
    // Bind click events
    container.querySelectorAll('.cowatch-gif-item').forEach(item => {
      item.addEventListener('click', () => {
        const gifUrl = item.dataset.gifUrl;
        const gifTitle = item.dataset.gifTitle;
        this.sendGifMessage(gifUrl, gifTitle);
        this.toggleGifPicker(); // Close picker
      });
    });
  }
  
  sendGifMessage(gifUrl, gifTitle = '') {
    if (this.socket && this.socket.readyState === WebSocket.OPEN) {
      this.socket.send(JSON.stringify({
        type: 'gif:send',
        roomCode: this.roomCode,
        userId: this.userId,
        username: this.username,
        avatar: this.userAvatar,
        gifUrl: gifUrl,
        gifTitle: gifTitle
      }));
      
      // Add to local messages immediately
      const gifMsg = {
        id: Date.now(),
        userId: this.userId,
        username: this.username,
        avatar: this.userAvatar,
        gifUrl: gifUrl,
        gifTitle: gifTitle,
        timestamp: Date.now(),
        type: 'gif'
      };
      this.messages.push(gifMsg);
      
      const messagesEl = document.getElementById('cowatchMessages');
      if (messagesEl) {
        messagesEl.insertAdjacentHTML('beforeend', this.renderGifMessage(gifMsg));
        messagesEl.scrollTop = messagesEl.scrollHeight;
      }
    }
  }
  
  renderGifMessage(msg) {
    const isMe = msg.userId === this.userId;
    const avatarEmoji = msg.avatar?.emoji || '👤';
    const avatarColor = msg.avatar?.color || '#6366f1';
    
    return `
      <div class="cowatch-message cowatch-message-gif ${isMe ? 'cowatch-message-me' : ''}">
        <div class="cowatch-message-avatar" style="background: ${avatarColor}">
          ${avatarEmoji}
        </div>
        <div class="cowatch-message-content">
          <div class="cowatch-message-header">
            <span class="cowatch-message-name" ${this.nameColor ? `style="color: ${this.nameColor}"` : ''}>${msg.username}</span>
            <span class="cowatch-message-time">${this.formatTime(msg.timestamp)}</span>
          </div>
          <div class="cowatch-message-gif-container">
            <img src="${msg.gifUrl}" alt="${msg.gifTitle || 'GIF'}" class="cowatch-gif-image" loading="lazy">
          </div>
        </div>
      </div>
    `;
  }

  addSystemMessage(text) {
    this.messages.push({
      text,
      timestamp: Date.now(),
      type: 'system'
    });
    // Just update the chat area, not full UI
    const messagesEl = document.getElementById('cowatchMessages');
    if (messagesEl) {
      const msgHtml = this.renderMessage(this.messages[this.messages.length - 1]);
      messagesEl.insertAdjacentHTML('beforeend', msgHtml);
      messagesEl.scrollTop = messagesEl.scrollHeight;
    }
  }

  updateChat() {
    const messagesEl = document.getElementById('cowatchMessages');
    if (messagesEl) {
      // Only update if messages count changed
      const currentMessageCount = messagesEl.querySelectorAll('.cowatch-message').length;
      if (currentMessageCount === this.messages.length) {
        // No new messages, don't re-render
        return;
      }
      
      // Check if we're near the bottom before update
      const scrollTop = messagesEl.scrollTop;
      const scrollHeight = messagesEl.scrollHeight;
      const clientHeight = messagesEl.clientHeight;
      const isNearBottom = scrollHeight - scrollTop <= clientHeight + 100;
      
      messagesEl.innerHTML = this.messages.map(m => this.renderMessage(m)).join('');
      
      // Re-bind reply buttons after re-render
      this.bindReplyButtons();
      
      // Only auto-scroll if user was near bottom
      if (isNearBottom) {
        messagesEl.scrollTop = messagesEl.scrollHeight;
      } else {
        // Restore scroll position
        messagesEl.scrollTop = scrollTop;
      }
    }
  }

  bindReplyButtons() {
    const container = document.getElementById('cowatch-container');
    if (!container) return;
    
    container.querySelectorAll('.cowatch-message-action-btn[data-action="reply"]').forEach(btn => {
      // Remove old listener first
      const newBtn = btn.cloneNode(true);
      btn.parentNode?.replaceChild(newBtn, btn);
      
      newBtn.addEventListener('click', (e) => {
        e.stopPropagation();
        this.handleReplyClick(newBtn);
      });
    });
  }

  handleReplyClick(btn) {
    const container = document.getElementById('cowatch-container');
    if (!container) return;
    
    const replyUserId = btn.dataset.userId;
    const replyUsername = replyUserId === this.userId ? this.username : btn.dataset.username;
    
    this.replyingTo = {
      id: btn.dataset.msgId,
      userId: replyUserId,
      username: replyUsername,
      text: btn.dataset.text
    };
    
    const chatInput = container.querySelector('.cowatch-chat-input');
    if (chatInput) {
      let replyBar = chatInput.querySelector('.cowatch-reply-bar');
      if (!replyBar) {
        replyBar = document.createElement('div');
        replyBar.className = 'cowatch-reply-bar';
        chatInput.insertBefore(replyBar, chatInput.firstChild);
      }
      replyBar.innerHTML = `
        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
          <path d="M3 10h10a8 8 0 0 1 8 8v4M3 10l6 6M3 10l6-6"/>
        </svg>
        <div class="cowatch-reply-info">
          <span class="cowatch-reply-to">Replying to <strong>${this.replyingTo.username}</strong></span>
          <span class="cowatch-reply-text">${this.replyingTo.text || ''}</span>
        </div>
        <button class="cowatch-cancel-reply">✕</button>
      `;
      
      replyBar.querySelector('.cowatch-cancel-reply')?.addEventListener('click', () => {
        this.replyingTo = null;
        replyBar.remove();
      });
    }
    
    container.querySelector('#cowatchChatInput')?.focus();
  }

  // Get platform-specific video container for resize
  getVideoContainer(video) {
    const host = window.location.hostname;
    
    // HBO Max / Max
    if (host.includes('max.com') || host.includes('hbomax.com')) {
      return video.closest('[class*="Player"]') || 
             video.closest('.default-player') ||
             video.closest('[data-testid="player"]') ||
             video.parentElement;
    }
    
    // Netflix
    if (host.includes('netflix.com')) {
      return video.closest('.watch-video') ||
             video.closest('.player-video-wrapper') ||
             video.parentElement;
    }
    
    // YouTube
    if (host.includes('youtube.com')) {
      return video.closest('#movie_player') ||
             video.closest('.html5-video-player') ||
             video.parentElement;
    }
    
    // Disney+
    if (host.includes('disneyplus.com')) {
      return video.closest('.btm-media-client-element') ||
             video.closest('[data-testid="media-container"]') ||
             video.parentElement;
    }
    
    // Prime Video
    if (host.includes('primevideo.com') || host.includes('amazon.com')) {
      return video.closest('.webPlayerElement') ||
             video.closest('.cascadesContainer') ||
             video.closest('[data-testid="video-player"]') ||
             video.parentElement;
    }
    
    // Hulu
    if (host.includes('hulu.com')) {
      return video.closest('.Player') ||
             video.closest('.VideoPlayer') ||
             video.parentElement;
    }
    
    // Peacock
    if (host.includes('peacocktv.com')) {
      return video.closest('.video-player') ||
             video.closest('[data-testid="video-container"]') ||
             video.parentElement;
    }
    
    // Twitch
    if (host.includes('twitch.tv')) {
      return video.closest('.video-player') ||
             video.closest('.video-ref') ||
             video.parentElement;
    }
    
    // Default - try common patterns
    return video.closest('[class*="player"]') ||
           video.closest('[class*="Player"]') ||
           video.closest('[class*="video"]') ||
           video.parentElement;
  }

  // Resize video when chat panel opens/closes (Teleparty style)
  // Works in BOTH fullscreen and normal mode
  resizeVideoForChat(chatOpen) {
    const video = document.querySelector('video');
    if (!video) return;
    
    const CHAT_WIDTH = 340;
    const host = window.location.hostname;
    
    // Get fullscreen state
    const isFullscreen = !!(document.fullscreenElement || 
                           document.webkitFullscreenElement || 
                           document.mozFullScreenElement);
    
    if (chatOpen) {
      // Add class to body for CSS-based video resize
      document.body.classList.add('cowatch-chat-open');
      document.documentElement.classList.add('cowatch-chat-open');
      
      // Shrink video to make room for chat - works in both normal and fullscreen
      if (host.includes('youtube.com')) {
        // YouTube specific - resize player and video
        const player = document.getElementById('movie_player');
        const videoContainer = document.querySelector('.html5-video-container');
        const ytdWatch = document.querySelector('ytd-watch-flexy');
        const playerTheater = document.querySelector('#player-theater-container');
        const primaryInner = document.querySelector('#primary-inner');
        
        // Normal mode - resize the page layout
        if (!isFullscreen) {
          if (ytdWatch) {
            ytdWatch.style.setProperty('--ytd-watch-flexy-width-ratio', '0.75', 'important');
            ytdWatch.style.marginRight = `${CHAT_WIDTH}px`;
          }
          if (playerTheater) {
            playerTheater.style.marginRight = `${CHAT_WIDTH}px`;
            playerTheater.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          }
          if (primaryInner) {
            primaryInner.style.marginRight = `${CHAT_WIDTH}px`;
          }
          if (player) {
            player.style.transition = 'width 0.3s ease';
          }
        } else {
          // Fullscreen mode - resize video itself
          if (player) {
            player.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
            player.style.maxWidth = `calc(100vw - ${CHAT_WIDTH}px)`;
          }
          if (videoContainer) {
            videoContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          }
          video.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.maxWidth = `calc(100vw - ${CHAT_WIDTH}px)`;
        }
      } else if (host.includes('netflix.com')) {
        // Netflix specific
        const playerContainer = document.querySelector('.watch-video--player-view');
        const watchVideo = document.querySelector('.watch-video');
        
        if (!isFullscreen) {
          if (watchVideo) {
            watchVideo.style.marginRight = `${CHAT_WIDTH}px`;
            watchVideo.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          }
        } else if (playerContainer) {
          playerContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.width = '100%';
        }
      } else if (host.includes('disneyplus.com')) {
        // Disney+ specific
        const playerContainer = document.querySelector('.btm-media-client-element');
        const mainContainer = document.querySelector('[data-testid="web-player"]') || 
                             document.querySelector('.dss-hls-player-container');
        
        if (!isFullscreen && mainContainer) {
          mainContainer.style.marginRight = `${CHAT_WIDTH}px`;
          mainContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
        } else if (playerContainer) {
          playerContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.width = '100%';
        }
      } else if (host.includes('primevideo.com') || host.includes('amazon.com')) {
        // Prime Video specific
        const playerContainer = document.querySelector('.webPlayerElement') || 
                               document.querySelector('.cascadesContainer');
        const mainContainer = document.querySelector('[data-testid="video-player-wrapper"]') ||
                             document.querySelector('.webPlayerUIContainer');
        
        if (!isFullscreen && mainContainer) {
          mainContainer.style.marginRight = `${CHAT_WIDTH}px`;
          mainContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
        } else if (playerContainer) {
          playerContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.width = '100%';
        }
      } else if (host.includes('max.com') || host.includes('hbomax.com')) {
        // HBO Max specific
        const playerContainer = document.querySelector('[data-testid="player"]') ||
                               document.querySelector('.default-player');
        
        if (!isFullscreen && playerContainer) {
          playerContainer.style.marginRight = `${CHAT_WIDTH}px`;
          playerContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
        } else if (playerContainer) {
          playerContainer.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.width = '100%';
        }
      } else {
        // Generic fallback for other platforms
        const videoParent = video.closest('[class*="player"]') || video.parentElement;
        
        if (!isFullscreen && videoParent) {
          videoParent.style.marginRight = `${CHAT_WIDTH}px`;
          videoParent.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          videoParent.style.transition = 'all 0.3s ease';
        } else {
          video.style.width = `calc(100% - ${CHAT_WIDTH}px)`;
          video.style.maxWidth = `calc(100vw - ${CHAT_WIDTH}px)`;
          video.style.objectFit = 'contain';
        }
      }
      
      console.log('🎬 Video resized for chat panel (Teleparty style) - fullscreen:', isFullscreen);
    } else {
      // Remove classes
      document.body.classList.remove('cowatch-chat-open');
      document.documentElement.classList.remove('cowatch-chat-open');
      
      // Reset video to full width
      this.resetVideoContainerStyles();
      video.style.cssText = '';
      
      // Platform specific resets
      if (host.includes('youtube.com')) {
        const player = document.getElementById('movie_player');
        const videoContainer = document.querySelector('.html5-video-container');
        const ytdWatch = document.querySelector('ytd-watch-flexy');
        const playerTheater = document.querySelector('#player-theater-container');
        const primaryInner = document.querySelector('#primary-inner');
        
        if (player) player.style.cssText = '';
        if (videoContainer) videoContainer.style.cssText = '';
        if (ytdWatch) {
          ytdWatch.style.removeProperty('--ytd-watch-flexy-width-ratio');
          ytdWatch.style.marginRight = '';
        }
        if (playerTheater) playerTheater.style.cssText = '';
        if (primaryInner) primaryInner.style.marginRight = '';
      } else if (host.includes('netflix.com')) {
        const watchVideo = document.querySelector('.watch-video');
        if (watchVideo) watchVideo.style.cssText = '';
      } else if (host.includes('disneyplus.com')) {
        const mainContainer = document.querySelector('[data-testid="web-player"]') || 
                             document.querySelector('.dss-hls-player-container');
        if (mainContainer) mainContainer.style.cssText = '';
      } else if (host.includes('primevideo.com') || host.includes('amazon.com')) {
        const mainContainer = document.querySelector('[data-testid="video-player-wrapper"]') ||
                             document.querySelector('.webPlayerUIContainer');
        if (mainContainer) mainContainer.style.cssText = '';
      } else if (host.includes('max.com') || host.includes('hbomax.com')) {
        const playerContainer = document.querySelector('[data-testid="player"]') ||
                               document.querySelector('.default-player');
        if (playerContainer) playerContainer.style.cssText = '';
      } else {
        const videoParent = video.closest('[class*="player"]') || video.parentElement;
        if (videoParent) videoParent.style.cssText = '';
      }
      
      console.log('🎬 Video restored to full width');
    }
  }

  // Reset video container styles when exiting fullscreen
  resetVideoContainerStyles() {
    const video = document.querySelector('video');
    if (!video) return;
    
    const container = this.getVideoContainer(video);
    if (container) {
      container.style.cssText = '';
    }
  }

  leaveRoom() {
    if (this.socket) {
      this.socket.send(JSON.stringify({
        type: 'room:leave',
        roomCode: this.roomCode,
        username: this.username
      }));
      this.socket.close();
    }

    // Clear auto-rejoin data (user intentionally left)
    this.clearAutoRejoinData();
    
    // Clear reconnection state
    this.connectionLost = false;
    this.reconnectAttempts = 0;
    if (this.reconnectTimeout) {
      clearTimeout(this.reconnectTimeout);
      this.reconnectTimeout = null;
    }
    
    // Clear server sync interval
    if (this.serverSyncInterval) {
      clearInterval(this.serverSyncInterval);
      this.serverSyncInterval = null;
    }
    
    // Clear task queue
    this.clearTaskQueue();

    // Clear room state FIRST (before togglePanel which checks roomCode)
    this.roomCode = null;
    this.isHost = false;
    this.socket = null;
    this.isConnected = false;
    this.participants = [];
    this.messages = [];
    this.unreadCount = 0;
    this.hostOnlyControl = false;
    this._hasJoinedRoom = false; // Reset join flag
    this.odaId = null; // Reset server-assigned ID
    this.visitorId = null;
    this.reactionsDisabled = false;
    this.chatOpen = false;
    this.usersTyping = []; // Clear typing users
    this.usersBuffering = []; // Clear buffering users

    // Remove toggle button since we're not in a room anymore
    this.removeToggleButton();
    
    this.clearState(); // Clear all persisted state
    
    // Hide the container completely instead of showing setup
    const container = document.getElementById('cowatch-container');
    if (container) {
      container.classList.remove('open');
      container.style.transform = 'translateX(100%)';
    }
    document.body.classList.remove('cowatch-panel-open');
  }

  // === FRIENDS SYSTEM ===
  async loadFriendsList() {
    this.friendsLoading = true;
    try {
      // Get auth token from storage
      const storage = await browserAPI.storage.local.get(['authToken', 'userId']);
      const authToken = storage.authToken;
      
      if (!authToken) {
        console.log('🎬 No auth token, user not logged in');
        this.friends = [];
        this.friendsLoading = false;
        return;
      }
      
      const response = await fetch('https://cowatch.tv/api/friends', {
        headers: {
          'Authorization': `Bearer ${authToken}`,
          'Content-Type': 'application/json'
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        this.friends = data.friends || [];
        console.log('🎬 Loaded friends:', this.friends.length);
      } else {
        console.warn('🎬 Failed to load friends:', response.status);
        this.friends = [];
      }
    } catch (error) {
      console.error('🎬 Error loading friends:', error);
      this.friends = [];
    }
    this.friendsLoading = false;
    this.updateUI(true);
  }

  async inviteFriend(friendId) {
    try {
      const storage = await browserAPI.storage.local.get(['authToken']);
      const authToken = storage.authToken;
      
      if (!authToken) {
        console.log('🎬 No auth token for invite');
        return;
      }
      
      // Send invite via socket
      if (this.socket && this.socket.readyState === WebSocket.OPEN) {
        this.socket.send(JSON.stringify({
          type: 'invite:friend',
          roomCode: this.roomCode,
          friendId: friendId,
          inviterName: this.username,
          videoUrl: window.location.href,
          videoTitle: document.title
        }));
      }
      
      // Also send via API for push notification
      await fetch('https://cowatch.tv/api/friends/invite', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${authToken}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          friendId: friendId,
          roomCode: this.roomCode,
          videoUrl: window.location.href,
          videoTitle: document.title
        })
      });
      
      // Mark as invited
      this.invitedFriends.add(friendId);
      this.updateUI(true);
      
      // Show success message
      this.showToast('Invite sent!', 'success');
    } catch (error) {
      console.error('🎬 Error inviting friend:', error);
      this.showToast('Failed to send invite', 'error');
    }
  }

  showToast(message, type = 'info') {
    // Create or get toast container
    let toastContainer = document.querySelector('.cowatch-toast-container');
    if (!toastContainer) {
      toastContainer = document.createElement('div');
      toastContainer.className = 'cowatch-toast-container';
      toastContainer.style.cssText = `
        position: fixed !important;
        bottom: 20px !important;
        right: 360px !important;
        z-index: 2147483647 !important;
        display: flex !important;
        flex-direction: column !important;
        gap: 8px !important;
      `;
      document.body.appendChild(toastContainer);
    }
    
    const toast = document.createElement('div');
    toast.className = `cowatch-toast cowatch-toast-${type}`;
    toast.style.cssText = `
      background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'} !important;
      color: white !important;
      padding: 12px 20px !important;
      border-radius: 10px !important;
      font-size: 14px !important;
      font-weight: 500 !important;
      box-shadow: 0 4px 20px rgba(0,0,0,0.3) !important;
      animation: cowatchToastIn 0.3s ease !important;
    `;
    toast.textContent = message;
    
    toastContainer.appendChild(toast);
    
    setTimeout(() => {
      toast.style.animation = 'cowatchToastOut 0.3s ease forwards';
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  }

  renderInviteModal() {
    if (!this.inviteModalOpen) return '';
    
    const inviteLink = `https://cowatch.tv/join/${this.roomCode}`;
    
    return `
      <div class="cowatch-modal-overlay cowatch-invite-modal-overlay">
        <div class="cowatch-invite-modal">
          <div class="cowatch-modal-header">
            <span class="cowatch-modal-title">Invite Friends</span>
            <button class="cowatch-modal-close" id="cowatchCloseInviteModal">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="18" y1="6" x2="6" y2="18"/>
                <line x1="6" y1="6" x2="18" y2="18"/>
              </svg>
            </button>
          </div>
          
          <div class="cowatch-invite-tabs">
            <button class="cowatch-invite-tab ${this.inviteActiveTab === 'link' ? 'active' : ''}" data-tab="link">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/>
                <path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/>
              </svg>
              Share Link
            </button>
            <button class="cowatch-invite-tab ${this.inviteActiveTab === 'friends' ? 'active' : ''}" data-tab="friends">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                <circle cx="9" cy="7" r="4"/>
                <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
              </svg>
              Friends
            </button>
          </div>
          
          <div class="cowatch-invite-content">
            ${this.inviteActiveTab === 'link' ? this.renderLinkTab(inviteLink) : this.renderFriendsTab()}
          </div>
        </div>
      </div>
    `;
  }

  renderLinkTab(inviteLink) {
    return `
      <div class="cowatch-invite-section">
        <div class="cowatch-invite-section-title">Party Link</div>
        <div class="cowatch-invite-link-box">
          <span class="cowatch-invite-link">${inviteLink}</span>
          <button class="cowatch-invite-copy-btn" id="cowatchCopyInviteLink">Copy</button>
        </div>
      </div>
      
      <div class="cowatch-invite-section">
        <div class="cowatch-invite-section-title">Share via</div>
        <div class="cowatch-share-buttons">
          <button class="cowatch-share-btn whatsapp">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
              <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413z"/>
            </svg>
            WhatsApp
          </button>
          <button class="cowatch-share-btn telegram">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
              <path d="M11.944 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0a12 12 0 0 0-.056 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 0 1 .171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.48.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z"/>
            </svg>
            Telegram
          </button>
        </div>
        <div class="cowatch-share-buttons" style="margin-top: 8px;">
          <button class="cowatch-share-btn twitter">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
              <path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/>
            </svg>
            X (Twitter)
          </button>
        </div>
      </div>
    `;
  }

  renderFriendsTab() {
    if (this.friendsLoading) {
      return `
        <div class="cowatch-no-friends">
          <div style="font-size: 24px; margin-bottom: 10px;">⏳</div>
          <p>Loading friends...</p>
        </div>
      `;
    }
    
    if (this.friends.length === 0) {
      return `
        <div class="cowatch-no-friends">
          <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
            <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
            <circle cx="9" cy="7" r="4"/>
            <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
            <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
          </svg>
          <p>No friends yet</p>
          <p style="font-size: 12px; margin-top: 8px;">Add friends from the CoWatch dashboard to invite them directly.</p>
        </div>
      `;
    }
    
    return `
      <input type="text" class="cowatch-friends-search" placeholder="Search friends..." id="cowatchFriendsSearch">
      <div class="cowatch-friends-list" id="cowatchFriendsList">
        ${this.friends.map(friend => {
          const isInvited = this.invitedFriends.has(friend._id || friend.id);
          const avatarBg = friend.avatar?.color || '#6366f1';
          const avatarEmoji = friend.avatar?.emoji || '😊';
          
          return `
            <div class="cowatch-friend-item" data-friend-id="${friend._id || friend.id}">
              <div class="avatar" style="background: ${avatarBg};">${avatarEmoji}</div>
              <div class="info">
                <div class="name">${friend.username || friend.name || 'Friend'}</div>
                <div class="status ${friend.online ? 'online' : 'offline'}">
                  <span class="status-dot"></span>
                  ${friend.online ? 'Online' : 'Offline'}
                </div>
              </div>
              <button class="cowatch-invite-btn cowatch-friend-invite-btn ${isInvited ? 'invited' : ''}" 
                      data-friend-id="${friend._id || friend.id}"
                      ${isInvited ? 'disabled' : ''}>
                ${isInvited ? 'Invited' : 'Invite'}
              </button>
            </div>
          `;
        }).join('')}
      </div>
    `;
  }

  // === SCHEDULED EVENTS SYSTEM ===
  async loadScheduledEvents() {
    try {
      const storage = await browserAPI.storage.local.get(['authToken']);
      const authToken = storage.authToken;
      
      if (!authToken) {
        this.scheduledEvents = [];
        return;
      }
      
      const response = await fetch('https://cowatch.tv/api/schedule', {
        headers: {
          'Authorization': `Bearer ${authToken}`,
          'Content-Type': 'application/json'
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        this.scheduledEvents = data.events || [];
        console.log('🎬 Loaded scheduled events:', this.scheduledEvents.length);
      } else {
        this.scheduledEvents = [];
      }
    } catch (error) {
      console.error('🎬 Error loading scheduled events:', error);
      this.scheduledEvents = [];
    }
  }

  async createScheduledEvent(eventData) {
    try {
      const storage = await browserAPI.storage.local.get(['authToken']);
      const authToken = storage.authToken;
      
      if (!authToken) {
        this.showToast('Please log in to schedule events', 'error');
        return null;
      }
      
      const response = await fetch('https://cowatch.tv/api/schedule', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${authToken}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          title: eventData.title,
          scheduledTime: eventData.scheduledTime,
          platform: eventData.platform || this.platform?.name,
          videoUrl: eventData.videoUrl || window.location.href,
          description: eventData.description,
          invitedFriends: eventData.invitedFriends || []
        })
      });
      
      if (response.ok) {
        const data = await response.json();
        this.scheduledEvents.push(data.event);
        this.showToast('Event scheduled!', 'success');
        this.createEventModalOpen = false;
        this.updateUI(true);
        return data.event;
      } else {
        this.showToast('Failed to schedule event', 'error');
        return null;
      }
    } catch (error) {
      console.error('🎬 Error creating scheduled event:', error);
      this.showToast('Failed to schedule event', 'error');
      return null;
    }
  }

  async deleteScheduledEvent(eventId) {
    try {
      const storage = await browserAPI.storage.local.get(['authToken']);
      const authToken = storage.authToken;
      
      if (!authToken) return;
      
      const response = await fetch(`https://cowatch.tv/api/schedule/${eventId}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${authToken}`
        }
      });
      
      if (response.ok) {
        this.scheduledEvents = this.scheduledEvents.filter(e => e._id !== eventId && e.id !== eventId);
        this.showToast('Event deleted', 'success');
        this.updateUI(true);
      }
    } catch (error) {
      console.error('🎬 Error deleting scheduled event:', error);
    }
  }

  async setEventReminder(eventId) {
    try {
      // Use browser alarms API via background script
      const event = this.scheduledEvents.find(e => e._id === eventId || e.id === eventId);
      if (!event) return;
      
      const reminderTime = new Date(event.scheduledTime).getTime() - 5 * 60 * 1000; // 5 minutes before
      
      await browserAPI.runtime.sendMessage({
        type: 'SET_ALARM',
        alarmName: `cowatch_event_${eventId}`,
        scheduledTime: reminderTime,
        eventData: {
          title: event.title,
          roomCode: event.roomCode
        }
      });
      
      this.showToast('Reminder set for 5 minutes before', 'success');
    } catch (error) {
      console.error('🎬 Error setting reminder:', error);
      this.showToast('Failed to set reminder', 'error');
    }
  }

  renderScheduledEventsModal() {
    if (!this.scheduledEventsModalOpen) return '';
    
    return `
      <div class="cowatch-modal-overlay" id="cowatchScheduleOverlay">
        <div class="cowatch-schedule-modal">
          <div class="cowatch-schedule-header">
            <span class="cowatch-schedule-title">📅 Scheduled Parties</span>
            <button class="cowatch-modal-close" id="cowatchCloseScheduleModal">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="18" y1="6" x2="6" y2="18"/>
                <line x1="6" y1="6" x2="18" y2="18"/>
              </svg>
            </button>
          </div>
          <div class="cowatch-schedule-content">
            ${this.scheduledEvents.length === 0 ? `
              <div class="cowatch-schedule-empty">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                  <rect x="3" y="4" width="18" height="18" rx="2"/>
                  <line x1="16" y1="2" x2="16" y2="6"/>
                  <line x1="8" y1="2" x2="8" y2="6"/>
                  <line x1="3" y1="10" x2="21" y2="10"/>
                </svg>
                <p>No scheduled parties yet</p>
                <button class="cowatch-btn cowatch-btn-primary" id="cowatchCreateEventBtn" style="margin-top: 16px;">
                  + Schedule a Party
                </button>
              </div>
            ` : `
              <div class="cowatch-events-list">
                ${this.scheduledEvents.map(event => this.renderEventItem(event)).join('')}
              </div>
              <button class="cowatch-btn cowatch-btn-primary" id="cowatchCreateEventBtn" style="margin-top: 16px; width: 100%;">
                + Schedule New Party
              </button>
            `}
          </div>
        </div>
      </div>
    `;
  }

  renderEventItem(event) {
    const eventDate = new Date(event.scheduledTime);
    const isUpcoming = eventDate > new Date();
    const timeString = eventDate.toLocaleString(undefined, {
      weekday: 'short',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
    
    return `
      <div class="cowatch-event-item" data-event-id="${event._id || event.id}">
        <div class="cowatch-event-title">${event.title || 'Watch Party'}</div>
        <div class="cowatch-event-time">
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <circle cx="12" cy="12" r="10"/>
            <polyline points="12 6 12 12 16 14"/>
          </svg>
          ${timeString}
        </div>
        <div class="cowatch-event-platform">
          ${event.platform ? `🎬 ${event.platform}` : ''}
          ${event.invitedCount ? `• ${event.invitedCount} invited` : ''}
        </div>
        ${isUpcoming ? `
          <div class="cowatch-event-actions">
            <button class="cowatch-event-btn reminder" data-event-id="${event._id || event.id}">
              🔔 Remind Me
            </button>
            ${event.roomCode ? `
              <button class="cowatch-event-btn join" data-room-code="${event.roomCode}">
                Join Party
              </button>
            ` : ''}
          </div>
        ` : ''}
      </div>
    `;
  }

  renderCreateEventModal() {
    if (!this.createEventModalOpen) return '';
    
    const now = new Date();
    const minDate = now.toISOString().slice(0, 16);
    
    return `
      <div class="cowatch-modal-overlay" id="cowatchCreateEventOverlay">
        <div class="cowatch-schedule-modal">
          <div class="cowatch-schedule-header">
            <span class="cowatch-schedule-title">📅 Schedule Party</span>
            <button class="cowatch-modal-close" id="cowatchCloseCreateEventModal">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="18" y1="6" x2="6" y2="18"/>
                <line x1="6" y1="6" x2="18" y2="18"/>
              </svg>
            </button>
          </div>
          <div class="cowatch-schedule-content">
            <form class="cowatch-create-event-form" id="cowatchEventForm">
              <div class="cowatch-form-group">
                <label class="cowatch-form-label">Party Title</label>
                <input type="text" class="cowatch-form-input" id="cowatchEventTitle" 
                       placeholder="Movie Night with Friends" required>
              </div>
              
              <div class="cowatch-form-group">
                <label class="cowatch-form-label">Date & Time</label>
                <input type="datetime-local" class="cowatch-form-input" id="cowatchEventTime" 
                       min="${minDate}" required>
              </div>
              
              <div class="cowatch-form-group">
                <label class="cowatch-form-label">Video URL (optional)</label>
                <input type="url" class="cowatch-form-input" id="cowatchEventUrl" 
                       placeholder="https://netflix.com/watch/..." 
                       value="${window.location.href}">
              </div>
              
              <div class="cowatch-form-group">
                <label class="cowatch-form-label">Description (optional)</label>
                <textarea class="cowatch-form-input" id="cowatchEventDesc" 
                          placeholder="Let's watch this together!" 
                          rows="2" style="resize: none;"></textarea>
              </div>
              
              <button type="submit" class="cowatch-btn cowatch-btn-primary" style="width: 100%; margin-top: 8px;">
                Schedule Party
              </button>
            </form>
          </div>
        </div>
      </div>
    `;
  }

  setupMessageListener() {
    console.log('🎬 Setting up message listener...');
    const handler = (message, sender, sendResponse) => {
      console.log('🎬 Received message:', message.type, message);
      
      // If init not complete for CREATE_ROOM, wait a bit then process
      if (!this._initComplete && message.type === 'CREATE_ROOM') {
        console.log('🎬 Init not complete, waiting before CREATE_ROOM...');
        // Wait for init to complete (max 3 seconds)
        const waitForInit = () => {
          let waited = 0;
          const checkInterval = setInterval(() => {
            waited += 100;
            if (this._initComplete || waited >= 3000) {
              clearInterval(checkInterval);
              console.log('🎬 Processing CREATE_ROOM after wait, initComplete:', this._initComplete);
              this.handleExtensionMessage(message, sendResponse);
            }
          }, 100);
        };
        waitForInit();
        return true; // Keep channel open for async response
      }
      
      return this.handleExtensionMessage(message, sendResponse);
    };
    
    // Use browser API or chrome API
    const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
    console.log('🎬 Using browserAPI for message listener');
    browserAPI.runtime.onMessage.addListener(handler);
  }
  
  handleExtensionMessage(message, sendResponse) {
      switch (message.type) {
        case 'GET_STATUS':
          // Try to find video if not already found
          if (!this.video) {
            this.findVideo();
          }
          // Also do a direct video check as fallback
          const directVideoCheck = document.querySelector('video');
          const hasVideo = !!(this.video || (directVideoCheck && (directVideoCheck.src || directVideoCheck.currentSrc || directVideoCheck.readyState > 0)));
          
          console.log('🎬 GET_STATUS response - hasVideo:', hasVideo, 'this.video:', !!this.video, 'directCheck:', !!directVideoCheck);
          
          sendResponse({
            connected: this.isConnected,
            roomCode: this.roomCode,
            isHost: this.isHost,
            platform: this.platform,
            hasVideo: hasVideo,
            participants: this.participants
          });
          break;

        case 'JOIN_ROOM':
          // If user info is passed from popup, use it
          if (message.user) {
            if (message.user.name) this.username = message.user.name;
            if (message.user.image) {
              this.userAvatar = { customImage: message.user.image, color: '#6366f1' };
            }
            this.saveState();
          }
          
          // In iframe mode, just open the sidebar with room code
          // Sidebar will handle WebSocket connection
          if (this.useIframeChat) {
            this.roomCode = message.roomCode;
            this.isHost = false;
            this.saveState();
            this.createUI();
            setTimeout(() => this.togglePanel(true), 100);
            sendResponse({ success: true, roomName: 'Watch Party' });
            return true;
          }
          
          // Store sendResponse for async callback (embedded mode)
          this._joinCallback = sendResponse;
          this._joinRoomCode = message.roomCode;
          
          this.connectToRoom(message.roomCode);
          
          // Return true to keep channel open for async response
          return true;

        case 'CREATE_ROOM':
          console.log('🎬 CREATE_ROOM received, generating room code...');
          
          // If user info is passed from popup, use it
          if (message.user) {
            if (message.user.name) this.username = message.user.name;
            if (message.user.image) {
              this.userAvatar = { customImage: message.user.image, color: '#6366f1' };
            }
          }
          
          // Generate room code and create room
          const roomCode = this.generateRoomCode();
          console.log('🎬 Generated room code:', roomCode);
          this.roomCode = roomCode;
          this.isHost = true;
          this.roomName = message.roomName || 'Watch Party';
          
          // Store party settings
          this.hostOnlyControl = message.hostOnlyControl || false;
          this.reactionsDisabled = message.disableReactions || false;
          
          this.saveState();
          
          // Recreate UI to ensure it exists (will create iframe in iframe mode)
          this.createUI();
          console.log('🎬 UI recreated, container:', !!document.getElementById('cowatch-container'));
          
          // Open sidebar immediately
          setTimeout(() => {
            console.log('🎬 Opening panel...');
            this.togglePanel(true);
          }, 100);
          
          // In iframe mode, sidebar handles WebSocket connection
          // In embedded mode, connect here
          if (!this.useIframeChat) {
            this.updateUI();
            this.connectToRoom(roomCode);
          }
          
          console.log('🎬 Sending response with roomCode:', roomCode);
          sendResponse({ success: true, roomCode: roomCode, roomName: this.roomName });
          break;

        case 'LEAVE_ROOM':
          this.leaveRoom();
          sendResponse({ success: true });
          break;

        case 'TOGGLE_PANEL':
          this.togglePanel(true); // Always open when called from popup
          sendResponse({ success: true });
          break;
      }
      return true;
  }

  // Utility functions
  getInitials(name) {
    return (name || 'G').substring(0, 2).toUpperCase();
  }

  formatTime(timestamp) {
    const date = new Date(timestamp);
    return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  }

  formatDuration(seconds) {
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  }

  formatVideoTime(seconds) {
    const hrs = Math.floor(seconds / 3600);
    const mins = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    if (hrs > 0) {
      return `${hrs}:${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  }

  escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
}

// Initialize
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => new CoWatch());
} else {
  new CoWatch();
}

})(); // End IIFE
