// CoWatch Extension Detector
// Lightweight script that only runs on cowatch.app to detect extension presence

(function() {
  'use strict';
  
  console.log('🎬 CoWatch Extension Detector loaded');
  
  // Add marker element immediately (document_start)
  const addMarker = () => {
    if (document.getElementById('cowatch-extension-installed')) return;
    
    const marker = document.createElement('div');
    marker.id = 'cowatch-extension-installed';
    marker.style.display = 'none';
    marker.dataset.version = '1.4.2';
    
    // Append to documentElement if body not ready, otherwise body
    (document.body || document.documentElement).appendChild(marker);
    console.log('🎬 Extension marker added');
  };
  
  // Add marker as soon as possible
  if (document.documentElement) {
    addMarker();
  }
  
  // Also add on DOMContentLoaded in case it was too early
  document.addEventListener('DOMContentLoaded', addMarker);
  
  // Listen for extension check events from the page
  window.addEventListener('cowatch-extension-check', () => {
    console.log('🎬 Extension check received, responding...');
    window.dispatchEvent(new CustomEvent('cowatch-extension-response', {
      detail: { installed: true, version: '1.4.2' }
    }));
  });
  
  // Also respond to message-based checks
  window.addEventListener('message', (event) => {
    if (event.source !== window) return;
    
    if (event.data?.type === 'COWATCH_EXTENSION_CHECK') {
      window.postMessage({ type: 'COWATCH_EXTENSION_RESPONSE', installed: true, version: '1.4.2' }, '*');
    }
    
    // Handle login success - save user to extension storage
    if (event.data?.type === 'COWATCH_USER_LOGGED_IN' && event.data.user) {
      console.log('🎬 User logged in, saving to extension storage:', event.data.user.email);
      browserAPI.storage.local.set({ 
        user: event.data.user, 
        guestMode: false 
      }, () => {
        console.log('🎬 User saved to extension storage');
        // Ask background script to close this tab since window.close() won't work
        browserAPI.runtime.sendMessage({ type: 'CLOSE_AUTH_TAB' });
      });
    }
  });
  
  // Listen for messages from popup/background script
  const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
  if (browserAPI?.runtime?.onMessage) {
    browserAPI.runtime.onMessage.addListener((message, sender, sendResponse) => {
      console.log('🎬 Extension detector received message:', message.type);
      
      if (message.type === 'GET_SESSION') {
        // Fetch session from the page context
        fetch('/api/auth/session', { credentials: 'include' })
          .then(res => res.json())
          .then(data => {
            console.log('🎬 Session fetched:', data.user?.email || 'no user');
            sendResponse(data);
          })
          .catch(error => {
            console.error('🎬 Session fetch error:', error);
            sendResponse({ user: null });
          });
        return true; // Keep message channel open for async response
      }
      
      if (message.type === 'CHECK_EXTENSION') {
        sendResponse({ installed: true, version: '1.4.2' });
      }
      
      if (message.type === 'LOGOUT') {
        // Logout from cowatch.app - this runs in the page context so cookies work
        console.log('🎬 Logging out from cowatch.app...');
        fetch('/api/auth/logout', {
          method: 'POST',
          credentials: 'include',
          headers: { 'Content-Type': 'application/json' }
        })
          .then(res => {
            console.log('🎬 Logout response:', res.status);
            // Just clear the session, don't call signout which causes redirects
            sendResponse({ success: true });
          })
          .catch(error => {
            console.error('🎬 Logout error:', error);
            sendResponse({ success: false, error: error.message });
          });
        return true; // Keep message channel open for async response
      }
    });
  }
})();
