// CoWatch Apple TV+ Injected Script
// This script runs in the PAGE CONTEXT to access Apple's MusicKit API
// Based on Teleparty's proven approach
(function() {
  'use strict';

  // Prevent double initialization
  if (window.cowatchAppleTvLoaded) return;
  window.cowatchAppleTvLoaded = true;

  console.log('🎬 CoWatch Apple TV+: Injected script loaded');

  // ============== SYNC CONSTANTS ==============
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  const VIDEO_ID_POLL_INTERVAL = 100;

  // ============== MUSICKIT API ==============
  function getMusicKitInstance() {
    try {
      return window.MusicKit?.getInstance();
    } catch (e) {
      return null;
    }
  }

  function getCurrentPlayable() {
    try {
      const mk = getMusicKitInstance();
      if (!mk) return null;
      
      return mk.playbackControllers?.serial?.services
        ?.mediaItemPlayback?._currentPlayer?._nowPlayingItem?.playables?.[0];
    } catch (e) {
      return null;
    }
  }

  function getVideoId() {
    const playable = getCurrentPlayable();
    if (!playable) return null;

    return {
      canonicalId: playable.canonicalId,
      contentId: playable.contentId
    };
  }

  function getSkipMap() {
    try {
      const mk = getMusicKitInstance();
      return mk?._playbackController?._skipIntro?.skipMap;
    } catch (e) {
      return null;
    }
  }

  // ============== STORE VIDEO ID IN DOM ==============
  function storeVideoId() {
    const videoId = getVideoId();
    if (!videoId) return;

    let container = document.querySelector('#cowatch-appletv-id');
    if (!container) {
      container = document.createElement('div');
      container.id = 'cowatch-appletv-id';
      container.style.display = 'none';
      document.body.appendChild(container);
    }
    
    container.setAttribute('data-canonical-id', videoId.canonicalId || '');
    container.setAttribute('data-content-id', videoId.contentId || '');
  }

  // Poll for video ID changes
  setInterval(storeVideoId, VIDEO_ID_POLL_INTERVAL);

  // ============== DISPATCH TO CONTENT ==============
  function dispatchToContent(detail) {
    const evt = new CustomEvent('FromNode', { detail });
    window.dispatchEvent(evt);
  }

  // ============== MESSAGE HANDLER ==============
  window.addEventListener('message', function(e) {
    if (e.source !== window) return;
    
    const type = e.data?.type;
    if (!type) return;

    switch (type) {
      case 'cowatch:getVideoId': {
        const videoId = getVideoId();
        dispatchToContent({
          type: 'cowatch:videoId',
          data: videoId
        });
        break;
      }

      case 'cowatch:getSkipMap': {
        const skipMap = getSkipMap();
        dispatchToContent({
          type: 'cowatch:skipMap',
          data: skipMap
        });
        break;
      }

      case 'cowatch:getState': {
        const video = document.querySelector('video');
        if (video) {
          dispatchToContent({
            type: 'cowatch:state',
            time: video.currentTime * 1000,
            paused: video.paused,
            duration: video.duration * 1000,
            videoId: getVideoId()
          });
        }
        break;
      }

      case 'cowatch:play': {
        const video = document.querySelector('video');
        if (video) video.play();
        break;
      }

      case 'cowatch:pause': {
        const video = document.querySelector('video');
        if (video) video.pause();
        break;
      }

      case 'cowatch:seek': {
        const video = document.querySelector('video');
        if (video) {
          const offsetTime = Math.max(0, e.data.time - SEEK_OFFSET_MS);
          video.currentTime = offsetTime / 1000;
        }
        break;
      }
    }
  });

  // ============== VIDEO EVENT LISTENERS ==============
  function setupVideoListeners() {
    const video = document.querySelector('video');
    if (!video) {
      setTimeout(setupVideoListeners, 500);
      return;
    }

    video.addEventListener('play', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'play' });
    });

    video.addEventListener('pause', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'pause' });
    });

    video.addEventListener('seeked', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'seek' });
    });

    video.addEventListener('waiting', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: true });
    });

    video.addEventListener('playing', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: false });
    });
  }

  // ============== INITIALIZATION ==============
  function init() {
    console.log('🎬 CoWatch Apple TV+: Initializing...');
    setupVideoListeners();
    storeVideoId();
    console.log('🎬 CoWatch Apple TV+: Initialization complete');
  }

  init();
})();
