// CoWatch Canal+ Injected Script
// Based on Teleparty's approach - React Fiber access for video info
(function() {
  'use strict';

  if (window.cowatchCanalplusLoaded) return;
  window.cowatchCanalplusLoaded = true;

  console.log('📺 CoWatch Canal+: Injected script loaded');

  // ============== REACT FIBER ACCESS ==============
  
  function getReactFiber(element) {
    if (!element) return null;
    const keys = Object.keys(element);
    for (const key of keys) {
      if (key.startsWith('__reactFiber$')) {
        return element[key];
      }
    }
    return null;
  }

  // ============== CONTENT ID ==============

  function getContentId() {
    try {
      const videoCore = document.querySelector("[data-testid='videoCore']");
      if (!videoCore) return null;

      const fiber = getReactFiber(videoCore);
      if (!fiber) return null;

      let current = fiber;
      while (current) {
        if (current.memoizedProps?.contentId) {
          return current.memoizedProps.contentId;
        }
        current = current.return;
      }
      return null;
    } catch (e) {
      return null;
    }
  }

  // ============== VIDEO ELEMENT ==============

  function getVideo() {
    return document.querySelector('video');
  }

  // ============== VIDEO INFORMATION ==============

  function getCurrentVideoInformation() {
    try {
      const contentId = getContentId();
      if (!contentId) return null;

      const video = getVideo();
      return {
        videoId: contentId,
        duration: video ? video.duration * 1000 : 0,
        title: document.querySelector('.video-title, [class*="title"]')?.textContent || ''
      };
    } catch (e) {
      return null;
    }
  }

  // ============== MESSAGE HANDLER ==============

  function handleMessage(event) {
    if (event.source !== window) return;
    
    const { type, time, infoSending } = event.data;
    
    // Handle Teleparty-style messages
    if (infoSending) {
      const infoType = infoSending.type;
      if (infoType === 'getVideoData') {
        const videoInfo = getCurrentVideoInformation();
        if (videoInfo) {
          window.postMessage({ type: 'VideoData', videoData: videoInfo }, '*');
        }
      }
      return;
    }

    if (!type) return;

    console.log('📺 CoWatch Canal+: Message received:', type);

    switch (type) {
      case 'COWATCH_GET_STATE': {
        const video = getVideo();
        
        if (video) {
          window.dispatchEvent(new CustomEvent('COWATCH_STATE', {
            detail: {
              currentTime: video.currentTime * 1000,
              duration: video.duration * 1000,
              paused: video.paused,
              playbackRate: video.playbackRate,
              videoId: getContentId(),
              platform: 'canalplus'
            }
          }));
        }
        break;
      }

      case 'COWATCH_SEEK': {
        const video = getVideo();
        if (video && time !== undefined) {
          video.currentTime = time / 1000;
        }
        break;
      }

      case 'COWATCH_PLAY': {
        const video = getVideo();
        if (video) video.play();
        break;
      }

      case 'COWATCH_PAUSE': {
        const video = getVideo();
        if (video) video.pause();
        break;
      }

      case 'COWATCH_GET_VIDEO_DATA': {
        const videoInfo = getCurrentVideoInformation();
        if (videoInfo) {
          window.dispatchEvent(new CustomEvent('COWATCH_VIDEO_DATA', {
            detail: videoInfo
          }));
        }
        break;
      }

      case 'COWATCH_TEARDOWN': {
        window.removeEventListener('message', handleMessage);
        window.cowatchCanalplusLoaded = false;
        break;
      }
    }
  }

  // ============== INITIALIZATION ==============

  window.addEventListener('message', handleMessage);

  // Notify content script that injected script is ready
  window.postMessage({ type: 'COWATCH_INJECTED_READY', platform: 'canalplus' }, '*');
})();
