// CoWatch Crunchyroll Injected Script
// This script runs in the PAGE CONTEXT (inside iframe) to access Crunchyroll's internal API
// Based on Teleparty's proven approach
(function() {
  'use strict';

  // Only run in iframe context
  if (window === top) return;
  
  // Check for player element
  if (!document.querySelector('#player0')) return;

  // Prevent double initialization
  if (window.cowatchCrunchyrollLoaded) return;
  window.cowatchCrunchyrollLoaded = true;

  console.log('🎬 CoWatch Crunchyroll: Injected script loaded');

  // ============== SYNC CONSTANTS ==============
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  const CONTROLS_TIMEOUT = 1500;

  // ============== PLAYBACK STATE ENUM ==============
  const PlaybackState = {
    LOADING: 'loading',
    PLAYING: 'playing',
    IDLE: 'idle',
    AD_PLAYING: 'ad_playing',
    PAUSED: 'paused',
    NOT_READY: 'not_ready'
  };

  // ============== VARIABLES ==============
  let buttonViewTimeout;
  let checkForAdsInterval;
  let isInAd = false;

  // ============== REACT INTERNAL ACCESS ==============
  function getReactInternals(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      // Crunchyroll uses __reactInternal prefix (without "Instance")
      if (keys[i].startsWith('__reactInternal') || keys[i].startsWith('__reactFiber')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  // Get player props through React
  function getPlayerProps() {
    try {
      const player = document.querySelector('#player0');
      if (!player) return null;
      const internals = getReactInternals(player);
      if (internals) {
        return internals.return?.stateNode?.props;
      }
    } catch (e) {
      console.warn('🎬 CoWatch Crunchyroll: getPlayerProps failed:', e);
    }
    return null;
  }

  // ============== PLAYBACK STATE ==============
  function getPlaybackState() {
    const video = document.querySelector('#player0');
    if (!video) {
      return PlaybackState.NOT_READY;
    }
    
    // Check for ads
    if (isInAd || document.querySelector('[data-testid="vilos-ad_label"]')) {
      return PlaybackState.AD_PLAYING;
    }
    
    if (video.readyState < 4) {
      return PlaybackState.LOADING;
    }
    if (video.paused) {
      return PlaybackState.PAUSED;
    }
    return PlaybackState.PLAYING;
  }

  function isAdPlaying() {
    const adLabel = document.querySelector('[data-testid="vilos-ad_label"]');
    const props = getPlayerProps();
    return !!adLabel || props?.isInAdBreak;
  }

  // ============== MESSAGE RELAY ==============
  function relayMessage(messageObj) {
    top.postMessage(messageObj, '*');
  }

  // ============== VIDEO INFORMATION ==============
  function getCurrentVideoInfo() {
    try {
      const props = getPlayerProps();
      if (!props?.configuration?.metadata) return null;
      
      const metadata = props.configuration.metadata;
      return {
        id: metadata.id,
        title: metadata.title,
        type: metadata.type, // 'episode', 'movie'
        episodeNumber: metadata.sequenceNumber,
        seriesId: metadata.seriesId,
        duration: document.querySelector('#player0')?.duration * 1000 || 0
      };
    } catch (e) {
      return null;
    }
  }

  // ============== PLAYER CONTROLS ==============
  function playVideo() {
    const playButton = document.querySelector('[data-test-state="stopped"]');
    if (playButton) {
      playButton.click();
    } else {
      const video = document.querySelector('#player0');
      if (video) video.play();
    }
  }

  function pauseVideo() {
    const video = document.querySelector('#player0');
    if (video) video.pause();
  }

  function seekTo(timeMs) {
    const props = getPlayerProps();
    if (props?.playerActions?.requestSeekToContentTime) {
      // Crunchyroll uses seconds, apply offset
      const offsetTime = Math.max(0, timeMs - SEEK_OFFSET_MS);
      props.playerActions.requestSeekToContentTime(offsetTime / 1000);
    }
  }

  function jumpToNext() {
    const props = getPlayerProps();
    if (props?.playerActions?.ended) {
      props.playerActions.ended();
    }
  }

  // ============== CONTROLS VISIBILITY ==============
  function hideControls() {
    const controlsContainer = document.getElementById('velocity-controls-package');
    if (controlsContainer) {
      controlsContainer.style.transform = 'translateY(50px)';
    }
  }

  function unhideControls() {
    const controlsContainer = document.getElementById('velocity-controls-package');
    if (controlsContainer) {
      controlsContainer.style.transform = '';
    }
  }

  // ============== AD DETECTION ==============
  function checkAdStatus() {
    const currentlyInAd = isAdPlaying();
    if (currentlyInAd && !isInAd) {
      isInAd = true;
      relayMessage({ type: 'cowatch:adStart' });
    } else if (!currentlyInAd && isInAd) {
      isInAd = false;
      relayMessage({ type: 'cowatch:adEnd' });
    }
  }

  // ============== SETTINGS OBSERVER ==============
  const settingsObserver = new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type === 'childList') {
        const settingsMenu = document.getElementById('velocity-settings-menu');
        const middleContainer = document.querySelector('[data-testid="middleBarContainer"]');
        
        // Hide middle container for better watch party experience
        if (middleContainer) {
          middleContainer.style.display = 'none';
        }
        
        // Keep settings menu above CoWatch UI
        if (settingsMenu) {
          settingsMenu.style.zIndex = '9999';
        }
        
        // Detect controls visibility
        const controlsVisible = !!document.querySelector('[data-testid="vilos-play_pause_button"]');
        if (controlsVisible) {
          if (buttonViewTimeout) clearTimeout(buttonViewTimeout);
          relayMessage({ type: 'cowatch:controlsVisible', visible: true });
        } else {
          if (buttonViewTimeout) clearTimeout(buttonViewTimeout);
          buttonViewTimeout = setTimeout(() => {
            relayMessage({ type: 'cowatch:controlsVisible', visible: false });
          }, CONTROLS_TIMEOUT);
        }
      }
    }
  });

  // ============== MESSAGE HANDLER ==============
  window.addEventListener('message', function(evt) {
    const data = evt.data?.infoSending || evt.data;
    if (!data?.type) return;

    switch (data.type) {
      case 'cowatch:getVideoData':
        const videoInfo = getCurrentVideoInfo();
        if (videoInfo) {
          relayMessage({ type: 'cowatch:videoData', data: videoInfo });
        }
        break;

      case 'cowatch:play':
        playVideo();
        break;

      case 'cowatch:pause':
        pauseVideo();
        break;

      case 'cowatch:getState':
        const video = document.querySelector('#player0');
        const state = {
          time: video ? video.currentTime * 1000 : 0,
          playbackState: getPlaybackState(),
          isAdPlaying: isAdPlaying()
        };
        relayMessage({ type: 'cowatch:state', data: state });
        break;

      case 'cowatch:seek':
        seekTo(data.time);
        break;

      case 'cowatch:next':
        jumpToNext();
        break;

      case 'cowatch:hideControls':
        hideControls();
        break;

      case 'cowatch:showControls':
        unhideControls();
        break;
    }
  });

  // ============== INITIALIZATION ==============
  function init() {
    console.log('🎬 CoWatch Crunchyroll: Initializing...');

    // Start ad check interval
    checkForAdsInterval = setInterval(checkAdStatus, 2000);

    // Fix player positioning
    const playerPackage = document.getElementById('velocity-player-package');
    const controlsPackage = document.getElementById('velocity-controls-package');
    
    if (playerPackage) playerPackage.style.position = 'fixed';
    if (controlsPackage) {
      controlsPackage.style.position = 'fixed';
      settingsObserver.observe(controlsPackage, { 
        attributes: true, 
        childList: true, 
        subtree: true 
      });
    }

    // Video event listeners
    const video = document.querySelector('#player0');
    if (video) {
      video.addEventListener('loadstart', () => {
        relayMessage({ type: 'cowatch:videoLoadStart' });
      });

      video.addEventListener('play', () => {
        relayMessage({ type: 'cowatch:userInteraction', action: 'play' });
      });

      video.addEventListener('pause', () => {
        relayMessage({ type: 'cowatch:userInteraction', action: 'pause' });
      });

      video.addEventListener('seeked', () => {
        relayMessage({ type: 'cowatch:userInteraction', action: 'seek' });
      });
    }

    // User interaction listeners
    window.addEventListener('mouseup', () => {
      relayMessage({ type: 'cowatch:userInteraction', action: 'click' });
    });

    window.addEventListener('keyup', (e) => {
      relayMessage({ type: 'cowatch:userInteraction', action: 'keypress', key: e.key });
      if (e.key === 'Escape') {
        relayMessage({ type: 'cowatch:exitFullscreen' });
      }
    });

    // Fullscreen override
    document.addEventListener('fullscreenchange', () => {
      relayMessage({ type: 'cowatch:fullscreenChange', isFullscreen: !!document.fullscreenElement });
    });

    const vilosRoot = document.querySelector('#vilosRoot');
    if (vilosRoot) {
      const originalRequestFullscreen = vilosRoot.requestFullscreen;
      vilosRoot.requestFullscreen = function() {
        relayMessage({ type: 'cowatch:fullscreenRequest' });
        // Don't actually go fullscreen - CoWatch handles it
      };
    }

    console.log('🎬 CoWatch Crunchyroll: Initialization complete');
  }

  // Wait for player to be ready
  if (document.querySelector('#player0')) {
    init();
  } else {
    const observer = new MutationObserver(() => {
      if (document.querySelector('#player0')) {
        observer.disconnect();
        init();
      }
    });
    observer.observe(document.body, { childList: true, subtree: true });
  }
})();
