// CoWatch Disney+ Injected Script
// Based on Teleparty's disney_injected_bundled.js approach
// Disney+ uses Shadow DOM for player controls - we need to handle that
(function() {
  'use strict';
  
  // Mark script as loaded (Teleparty pattern)
  window.seekScriptLoaded = true;
  
  // Sync constants (Teleparty approach)
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  
  /**
   * Find React props from Disney+ internal structure
   * Teleparty exact approach: main#section_index > div → __reactInternalInstance
   */
  function findReactProps() {
    try {
      const elementRoot = document.querySelector("main#section_index > div");
      if (!elementRoot) {
        return null;
      }
      
      const keys = Object.keys(elementRoot);
      let key = null;
      
      for (let i = 0; i < keys.length; i++) {
        if (keys[i].startsWith("__reactInternalInstance")) {
          key = keys[i];
          break;
        }
      }
      
      if (!key) {
        return null;
      }
      
      if (typeof elementRoot[key] === "undefined" ||
          typeof elementRoot[key].memoizedProps?.children?._owner === "undefined") {
        return null;
      }
      
      return elementRoot[key].memoizedProps.children._owner.memoizedProps;
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: findReactProps error:', e);
      return null;
    }
  }
  
  /**
   * Navigate to next episode using React internal navigation
   * Teleparty exact pattern
   */
  function navigateToVideo(videoId) {
    try {
      const reactProps = findReactProps();
      if (reactProps && reactProps.navigate) {
        reactProps.navigate({
          name: "video",
          params: {
            contentId: videoId,
            timerAutoAdvanced: true
          }
        });
        return true;
      }
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: Navigate failed:', e);
    }
    return false;
  }
  
  /**
   * Get video element
   */
  function getVideoElement() {
    return document.querySelector('.btm-media-player video') ||
           document.querySelector('[data-testid="video-player"] video') ||
           document.querySelector('video');
  }
  
  /**
   * Play video using Shadow DOM controls (Teleparty approach)
   * Disney+ uses Shadow DOM for player controls
   */
  function playVideo() {
    try {
      // Method 1: Direct video API
      const video = getVideoElement();
      if (video && video.paused) {
        video.play();
        return true;
      }
      
      // Method 2: Shadow DOM controls (Teleparty approach)
      const playerContainer = document.querySelector(".btm-media-player");
      if (playerContainer) {
        playerContainer.click(); // Show controls first
        
        const toggleButton = document.querySelector(".toggle-play-pause");
        if (toggleButton && toggleButton.shadowRoot) {
          const playButton = toggleButton.shadowRoot.querySelector('.play-button');
          if (playButton) {
            playButton.click();
            return true;
          }
        }
      }
      
      // Method 3: Test ID selector
      const playBtn = document.querySelector('[data-testid="play-button"]');
      if (playBtn) {
        playBtn.click();
        return true;
      }
      
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: Play failed:', e);
      return false;
    }
  }
  
  /**
   * Pause video using Shadow DOM controls (Teleparty approach)
   */
  function pauseVideo() {
    try {
      // Method 1: Direct video API
      const video = getVideoElement();
      if (video && !video.paused) {
        video.pause();
        return true;
      }
      
      // Method 2: Shadow DOM controls (Teleparty approach)
      const playerContainer = document.querySelector(".btm-media-player");
      if (playerContainer) {
        playerContainer.click(); // Show controls first
        
        const toggleButton = document.querySelector(".toggle-play-pause");
        if (toggleButton && toggleButton.shadowRoot) {
          const pauseButton = toggleButton.shadowRoot.querySelector('.pause-button');
          if (pauseButton) {
            pauseButton.click();
            return true;
          }
        }
      }
      
      // Method 3: Test ID selector
      const pauseBtn = document.querySelector('[data-testid="pause-button"]');
      if (pauseBtn) {
        pauseBtn.click();
        return true;
      }
      
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: Pause failed:', e);
      return false;
    }
  }
  
  /**
   * Seek to specific time
   */
  function seekTo(time) {
    try {
      const video = getVideoElement();
      if (video) {
        video.currentTime = time;
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: Seek failed:', e);
      return false;
    }
  }
  
  /**
   * Get content ID from URL
   */
  function getContentId() {
    const match = window.location.pathname.match(/video\/([a-zA-Z0-9-]+)/);
    return match ? match[1] : null;
  }
  
  /**
   * Get current playback state
   */
  function getCurrentState() {
    try {
      const video = getVideoElement();
      if (!video) return null;
      
      return {
        time: video.currentTime || 0,
        duration: video.duration || 0,
        paused: video.paused,
        playbackRate: video.playbackRate || 1,
        videoId: getContentId(),
        title: document.title?.replace(' | Disney+', '').replace(' - Disney+', '') || '',
        buffered: video.buffered?.length > 0 ? video.buffered.end(video.buffered.length - 1) : 0
      };
    } catch (e) {
      console.warn('🎬 CoWatch Disney+: Failed to get state:', e);
      return null;
    }
  }
  
  /**
   * Dispatch response via FromNode CustomEvent (Teleparty pattern)
   */
  function dispatchResponse(type, data) {
    document.dispatchEvent(new CustomEvent("FromNode", {
      detail: {
        type: type,
        ...data
      }
    }));
  }
  
  /**
   * Main message handler (Teleparty pattern)
   */
  function seekInteraction(e) {
    if (e.source !== window) return;
    
    const { type, data } = e.data || {};
    
    // Teleparty-style message types
    if (type === "NEXT_EPISODE" && e.data.videoId) {
      navigateToVideo(e.data.videoId);
      return;
    }
    
    if (type === "teardown") {
      console.log("🎬 CoWatch Disney+: Teardown");
      window.removeEventListener("message", seekInteraction, false);
      window.seekScriptLoaded = false;
      return;
    }
    
    if (type === "play") {
      playVideo();
      dispatchResponse("playResponse", { success: true });
      return;
    }
    
    if (type === "pause") {
      pauseVideo();
      dispatchResponse("pauseResponse", { success: true });
      return;
    }
    
    // CoWatch-style message handling
    if (e.data?.source === 'cowatch-content') {
      const { action, data: actionData } = e.data;
      let result = null;
      
      switch (action) {
        case 'getState':
          result = getCurrentState();
          break;
        case 'play':
          result = playVideo();
          break;
        case 'pause':
          result = pauseVideo();
          break;
        case 'seek':
          result = seekTo(actionData?.time);
          break;
        case 'getDuration':
          const video = getVideoElement();
          result = video?.duration || 0;
          break;
        case 'nextEpisode':
          if (actionData?.videoId) {
            result = navigateToVideo(actionData.videoId);
          }
          break;
      }
      
      // Send response back via both methods
      window.postMessage({
        source: 'cowatch-injected',
        action: action + 'Response',
        data: result
      }, '*');
      
      // Also dispatch via FromNode for content script
      dispatchResponse(action + 'Response', { data: result });
    }
  }
  
  // Initialize
  console.log("🎬 CoWatch Disney+: Script loaded (Teleparty-enhanced)");
  window.addEventListener("message", seekInteraction, false);
  
  // Monitor for video element and setup event forwarding
  const setupVideoMonitor = () => {
    const video = getVideoElement();
    if (video && !video._cowatchMonitored) {
      video._cowatchMonitored = true;
      
      // Forward video events to content script
      ['play', 'pause', 'seeked', 'timeupdate', 'ended'].forEach(eventType => {
        video.addEventListener(eventType, () => {
          dispatchResponse('videoEvent', {
            event: eventType,
            state: getCurrentState()
          });
        });
      });
      
      console.log('🎬 CoWatch Disney+: Video monitor attached');
    }
  };
  
  // Watch for video element
  const observer = new MutationObserver(() => {
    if (getVideoElement()) {
      setupVideoMonitor();
    }
  });
  
  if (document.body) {
    observer.observe(document.body, { childList: true, subtree: true });
    setupVideoMonitor();
  } else {
    document.addEventListener('DOMContentLoaded', () => {
      observer.observe(document.body, { childList: true, subtree: true });
      setupVideoMonitor();
    });
  }
})();
