// CoWatch ESPN+ Injected Script
// This script runs in the PAGE CONTEXT (inside iframe) to access ESPN's internal API
// Based on Teleparty's proven approach
(function() {
  'use strict';

  // Only run in iframe context for ESPN
  if (window === top) return;

  // Prevent double initialization
  if (window.cowatchEspnLoaded) return;
  window.cowatchEspnLoaded = true;

  console.log('🎬 CoWatch ESPN+: Injected script loaded');

  // ============== SYNC CONSTANTS ==============
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  const CONTROLS_TIMEOUT = 1500;

  // ============== VIDEO TYPES ==============
  const VideoType = {
    LIVE: 'live',
    EPISODE: 'episode',
    FEATURE: 'feature',
    EVENT: 'event'
  };

  // ============== PLAYER API ==============
  function getPlayerOptions() {
    try {
      return window.espn?.video?._player?._playerExtension?._savedInitOptions;
    } catch (e) {
      return null;
    }
  }

  function getVideoInfo() {
    const options = getPlayerOptions();
    if (!options) return null;

    try {
      const sectionPath = options.section_path || {};
      const mediaItem = sectionPath.mediaItem || {};
      const customProps = sectionPath.customProps || {};

      let videoType = VideoType.FEATURE;
      if (mediaItem.streamType === 'LIVE') {
        videoType = VideoType.LIVE;
      } else if (customProps.pageApiMeta?.name?.includes('Ep.')) {
        videoType = VideoType.EPISODE;
      }

      return {
        id: sectionPath.id,
        seriesId: sectionPath.seriesId,
        title: customProps.pageApiMeta?.name,
        type: videoType,
        isLive: mediaItem.streamType === 'LIVE'
      };
    } catch (e) {
      return null;
    }
  }

  // ============== MESSAGE RELAY ==============
  function relayMessage(messageObj) {
    top.postMessage(messageObj, '*');
  }

  // ============== DISPATCH TO CONTENT ==============
  function dispatchToContent(detail) {
    const evt = new CustomEvent('FromNode', { detail });
    window.dispatchEvent(evt);
  }

  // ============== CONTROLS OBSERVER ==============
  let controlsTimeout;
  const controlsObserver = new MutationObserver(() => {
    const controlsVisible = !!document.querySelector('.btm-media-overlays-container:not([hidden])');
    
    if (controlsVisible) {
      if (controlsTimeout) clearTimeout(controlsTimeout);
      relayMessage({ type: 'cowatch:controlsVisible', visible: true });
    } else {
      if (controlsTimeout) clearTimeout(controlsTimeout);
      controlsTimeout = setTimeout(() => {
        relayMessage({ type: 'cowatch:controlsVisible', visible: false });
      }, CONTROLS_TIMEOUT);
    }
  });

  // ============== MESSAGE HANDLER ==============
  window.addEventListener('message', function(e) {
    const data = e.data;
    if (!data?.type) return;

    switch (data.type) {
      case 'cowatch:getVideoData': {
        const videoInfo = getVideoInfo();
        if (videoInfo) {
          relayMessage({ type: 'cowatch:videoData', data: videoInfo });
        }
        break;
      }

      case 'cowatch:getState': {
        const video = document.querySelector('video');
        if (video) {
          dispatchToContent({
            type: 'cowatch:state',
            time: video.currentTime * 1000,
            paused: video.paused,
            duration: video.duration * 1000,
            isLive: getVideoInfo()?.isLive
          });
        }
        break;
      }

      case 'cowatch:play': {
        const video = document.querySelector('video');
        if (video) video.play();
        break;
      }

      case 'cowatch:pause': {
        const video = document.querySelector('video');
        if (video) video.pause();
        break;
      }

      case 'cowatch:seek': {
        const video = document.querySelector('video');
        if (video) {
          const offsetTime = Math.max(0, data.time - SEEK_OFFSET_MS);
          video.currentTime = offsetTime / 1000;
        }
        break;
      }
    }
  });

  // ============== VIDEO EVENT LISTENERS ==============
  function setupVideoListeners() {
    const video = document.querySelector('video');
    if (!video) {
      setTimeout(setupVideoListeners, 500);
      return;
    }

    video.addEventListener('play', () => {
      relayMessage({ type: 'cowatch:userInteraction', action: 'play' });
    });

    video.addEventListener('pause', () => {
      relayMessage({ type: 'cowatch:userInteraction', action: 'pause' });
    });

    video.addEventListener('seeked', () => {
      relayMessage({ type: 'cowatch:userInteraction', action: 'seek' });
    });

    video.addEventListener('waiting', () => {
      relayMessage({ type: 'cowatch:buffering', isBuffering: true });
    });

    video.addEventListener('playing', () => {
      relayMessage({ type: 'cowatch:buffering', isBuffering: false });
    });
  }

  // ============== FULLSCREEN HANDLING ==============
  function setupFullscreenOverride() {
    const playerContainer = document.querySelector('.btm-media-player');
    if (playerContainer) {
      playerContainer.requestFullscreen = function() {
        relayMessage({ type: 'cowatch:fullscreenRequest' });
      };
    }
  }

  // ============== INITIALIZATION ==============
  function init() {
    console.log('🎬 CoWatch ESPN+: Initializing...');
    
    setupVideoListeners();
    setupFullscreenOverride();

    // Observe controls
    const controlsContainer = document.querySelector('.btm-media-overlays-container');
    if (controlsContainer) {
      controlsObserver.observe(controlsContainer, { 
        attributes: true, 
        childList: true, 
        subtree: true 
      });
    }

    console.log('🎬 CoWatch ESPN+: Initialization complete');
  }

  // Wait for video to be ready
  if (document.querySelector('video')) {
    init();
  } else {
    const observer = new MutationObserver(() => {
      if (document.querySelector('video')) {
        observer.disconnect();
        init();
      }
    });
    observer.observe(document.body, { childList: true, subtree: true });
  }
})();
