// CoWatch HBO Max / Max Injected Script
// Based on Teleparty's hbo_max_injected_bundled.js approach
// Uses React internals for ad detection and metadata access
(function() {
  'use strict';
  
  // Mark script as loaded (Teleparty pattern)
  if (window.videoIdScriptLoaded) {
    console.log('🎬 CoWatch Max: Script already loaded');
    return;
  }
  window.videoIdScriptLoaded = true;
  
  // Sync constants (Teleparty approach)
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  
  /**
   * Get React Fiber from element (Teleparty exact approach)
   */
  function getReactFiber(root) {
    if (root == null) return null;
    
    const keys = Object.keys(root);
    let key = null;
    
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith("__reactFiber")) {
        key = keys[i];
        break;
      }
    }
    
    return key ? root[key] : null;
  }
  
  /**
   * Get React Props from element (Teleparty exact approach)
   */
  function findReactPropsObject(elementRoot) {
    if (elementRoot == null) return null;
    
    const keys = Object.keys(elementRoot);
    let key = null;
    
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith("__reactProps")) {
        key = keys[i];
        break;
      }
    }
    
    if (!key || typeof elementRoot[key] === "undefined") {
      return null;
    }
    
    return elementRoot[key];
  }
  
  /**
   * Find React props from overlay-root (Teleparty exact approach)
   * Deep path to player state for ad info
   */
  function findReactProps() {
    try {
      const elementRoot = document.querySelector("#overlay-root");
      if (!elementRoot) return null;
      
      const keys = Object.keys(elementRoot);
      let key = null;
      
      for (let i = 0; i < keys.length; i++) {
        if (keys[i].startsWith("__reactFiber")) {
          key = keys[i];
          break;
        }
      }
      
      if (!key || typeof elementRoot[key] === "undefined") {
        return null;
      }
      
      // Teleparty's exact deep path to player state
      return elementRoot[key]?.return?.pendingProps?.playerCallbacks?.mediator
        ?._playerControls?._playbackEngineAdapter?._playbackEngine
        ?._player?._playerStateTracker?._playerState?.variant?.video
        ?._periodMap?._storage;
    } catch (e) {
      console.warn('🎬 CoWatch Max: findReactProps error:', e);
      return null;
    }
  }
  
  /**
   * Get metadata from content-metadata element (Teleparty approach)
   */
  function getMetadata() {
    try {
      const fiber = getReactFiber(document.querySelector("[data-testid='content-metadata']"));
      return fiber?.return?.return?.memoizedProps?.contentMetadata;
    } catch (e) {
      console.warn('🎬 CoWatch Max: getMetadata error:', e);
      return null;
    }
  }
  
  /**
   * Get video lookup data (Teleparty exact approach)
   */
  function getVideoLookupData() {
    const metadata = getMetadata();
    if (!metadata) return null;
    
    return {
      title: metadata.title,
      type: metadata.videoType === "EPISODE" ? "TV" : "MOVIE",
      year: metadata.videoType !== "EPISODE" && metadata.extras?.AIR_DATE?.split("-")[0],
      episodeLookup: metadata.videoType === "EPISODE"
        ? {
            number: metadata.episodeNumber,
            season: metadata.seasonNumber,
            airDate: metadata.extras?.AIR_DATE
          }
        : undefined
    };
  }
  
  /**
   * Get all ads - alternative method (Teleparty approach)
   * Uses scrubber track data attribute
   */
  function getAllAdsAlt() {
    try {
      const element = document.querySelector('[data-testid="player-ux-scrubber-track"]');
      if (!element) return [];
      
      const props = findReactPropsObject(element);
      if (!props) {
        // Fallback: direct attribute access
        const tmpAdList = element.getAttribute("data-ad-slots");
        if (tmpAdList) {
          return JSON.parse(tmpAdList);
        }
        return [];
      }
      
      return JSON.parse(props["data-ad-slots"] || "[]");
    } catch (e) {
      console.warn('🎬 CoWatch Max: getAllAdsAlt error:', e);
      return [];
    }
  }
  
  /**
   * Get all ads from React internals (Teleparty approach)
   */
  function getAllAds() {
    try {
      const adElem = findReactProps();
      if (!adElem) return [];
      
      const positions = [];
      
      adElem.forEach((a, key) => {
        positions.push([a.start * 1000, a.end * 1000]);
      });
      
      // Remove duplicates
      const uniquePositions = Array.from(new Set(positions.map(JSON.stringify)), JSON.parse);
      
      // Convert to array of objects with startTime and duration
      const finalPositions = uniquePositions.map((position) => ({
        adStartTime: position[0],
        adDuration: position[1] - position[0]
      }));
      
      // Filter out long ads (>61 seconds)
      return finalPositions.filter((position) => position.adDuration <= 61000);
    } catch (err) {
      console.error('🎬 CoWatch Max: getAllAds error:', err);
      return [];
    }
  }
  
  /**
   * Network interception to prune ad-related JSON (Teleparty approach)
   * Removes ssaiInfo to prevent ad-related issues
   */
  function interceptNetwork() {
    const pruneJson = (jsonText) => {
      try {
        const json = JSON.parse(jsonText);
        if (typeof json !== "object" || json === null) return jsonText;
        
        // Remove ssaiInfo if present
        if ("ssaiInfo" in json) {
          delete json.ssaiInfo;
        }
        
        // Remove fallback.ssaiInfo if present
        if ("fallback" in json && typeof json.fallback === "object") {
          delete json.fallback.ssaiInfo;
        }
        
        // Remove manifest.url if present
        if ("manifest" in json && typeof json.manifest === "object") {
          delete json.manifest.url;
        }
        
        return JSON.stringify(json);
      } catch (e) {
        console.error("🎬 CoWatch Max: JSON Prune Error:", e);
        return jsonText;
      }
    };
    
    // Intercept fetch API
    const originalFetch = window.fetch;
    window.fetch = async (...args) => {
      const response = await originalFetch(...args);
      
      if (response.ok && 
          response.headers.get("content-type")?.includes("application/json") &&
          response.url.includes("api.hbomax.com")) {
        const modifiedJson = pruneJson(await response.clone().text());
        return new Response(modifiedJson, {
          status: response.status,
          statusText: response.statusText,
          headers: response.headers
        });
      }
      
      return response;
    };
    
    // Intercept XMLHttpRequest
    const originalXHROpen = XMLHttpRequest.prototype.open;
    XMLHttpRequest.prototype.open = function(...args) {
      originalXHROpen.apply(this, args);
      const url = args[1];
      
      if (url?.includes("api.hbomax.com")) {
        this.addEventListener("readystatechange", function() {
          if (this.readyState === 4 && 
              this.getResponseHeader("content-type")?.includes("application/json")) {
            try {
              const modifiedResponse = pruneJson(this.responseText);
              
              Object.defineProperty(this, "_responseText", {
                value: modifiedResponse,
                configurable: true
              });
              
              Object.defineProperty(this, "responseText", {
                get: function() { return this._responseText; },
                configurable: true
              });
              
              Object.defineProperty(this, "response", {
                get: function() { return this._responseText; },
                configurable: true
              });
            } catch (e) {
              console.error("🎬 CoWatch Max: XHR Prune Error:", e);
            }
          }
        });
      }
    };
  }
  
  /**
   * Dispatch response via FromNode CustomEvent (Teleparty pattern)
   */
  function dispatchResponse(type, data) {
    const evt = new CustomEvent("FromNode", {
      detail: {
        type: type,
        ...data,
        updatedAt: Date.now()
      }
    });
    window.dispatchEvent(evt);
  }
  
  /**
   * Get video element
   */
  function getVideoElement() {
    const containers = [
      document.querySelector('[data-testid="player"]'),
      document.querySelector('.default-player'),
      document.querySelector('.default-player-wrapper'),
      document.querySelector('[class*="PlayerContainer"]')
    ];
    
    for (const container of containers) {
      if (container) {
        const video = container.querySelector('video');
        if (video) return video;
      }
    }
    
    return document.querySelector('video');
  }
  
  /**
   * Get content ID from URL
   */
  function getContentId() {
    const match = window.location.pathname.match(/(?:watch|play|video)\/([a-zA-Z0-9-]+)/);
    return match ? match[1] : null;
  }
  
  /**
   * Get current playback state
   */
  function getCurrentState() {
    try {
      const video = getVideoElement();
      if (!video) return null;
      
      const metadata = getMetadata();
      const lookupData = getVideoLookupData();
      const adList = getAllAdsAlt();
      
      // Check if currently in ad
      const currentTimeMs = video.currentTime * 1000;
      const adPlaying = adList.some(ad => 
        currentTimeMs >= ad.adStartTime && 
        currentTimeMs <= (ad.adStartTime + ad.adDuration)
      );
      
      return {
        time: video.currentTime || 0,
        duration: video.duration || 0,
        paused: video.paused,
        playbackRate: video.playbackRate || 1,
        videoId: getContentId(),
        title: metadata?.title || lookupData?.title || document.title?.replace(' | Max', '').replace(' | HBO Max', '') || '',
        contentType: lookupData?.type,
        adPlaying: adPlaying,
        adList: adList,
        episodeInfo: lookupData?.episodeLookup
      };
    } catch (e) {
      console.warn('🎬 CoWatch Max: getCurrentState error:', e);
      return null;
    }
  }
  
  /**
   * Play video
   */
  function playVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.play();
        return true;
      }
      
      const playButton = document.querySelector('[data-testid="play-pause-button"]');
      if (playButton) {
        playButton.click();
        return true;
      }
      
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Max: play error:', e);
      return false;
    }
  }
  
  /**
   * Pause video
   */
  function pauseVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.pause();
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Max: pause error:', e);
      return false;
    }
  }
  
  /**
   * Seek to time
   */
  function seekTo(time) {
    try {
      const video = getVideoElement();
      if (video) {
        video.currentTime = time;
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Max: seek error:', e);
      return false;
    }
  }
  
  /**
   * Max Video Message handler (Teleparty pattern)
   */
  function handleMaxVideoMessage(event) {
    if (!event.detail) return;
    
    const type = event.detail.type;
    
    if (type === "getAdList") {
      let adList = getAllAdsAlt();
      if (adList.length === 0) {
        const tmpAdList = document
          .querySelector('[data-testid="player-ux-scrubber-track"]')
          ?.getAttribute("data-ad-slots");
        if (tmpAdList) {
          adList = JSON.parse(tmpAdList);
        }
      }
      dispatchResponse("getAd", { adList });
    }
    else if (type === "getVideoLookupData") {
      const data = getVideoLookupData();
      dispatchResponse("getData", { data });
    }
  }
  
  /**
   * CoWatch message handler
   */
  function handleCoWatchMessage(event) {
    if (event.data?.source !== 'cowatch-content') return;
    
    const { action, data } = event.data;
    let result = null;
    
    switch (action) {
      case 'getState':
        result = getCurrentState();
        break;
      case 'play':
        result = playVideo();
        break;
      case 'pause':
        result = pauseVideo();
        break;
      case 'seek':
        result = seekTo(data?.time);
        break;
      case 'getDuration':
        const video = getVideoElement();
        result = video?.duration || 0;
        break;
      case 'getAdList':
        result = getAllAdsAlt();
        break;
      case 'getVideoLookupData':
        result = getVideoLookupData();
        break;
    }
    
    // Send response back
    window.postMessage({
      source: 'cowatch-injected',
      action: action + 'Response',
      data: result
    }, '*');
    
    // Also dispatch via FromNode
    dispatchResponse(action + 'Response', { data: result });
  }
  
  // Initialize network interception
  interceptNetwork();
  
  // Setup event listeners
  console.log("🎬 CoWatch Max: Script loaded (Teleparty-enhanced)");
  
  // Teleparty-style Max event
  window.addEventListener("MaxVideoMessage", handleMaxVideoMessage);
  
  // CoWatch message handler
  window.addEventListener("message", handleCoWatchMessage, false);
  
  // Monitor for video element
  const setupVideoMonitor = () => {
    const video = getVideoElement();
    if (video && !video._cowatchMonitored) {
      video._cowatchMonitored = true;
      
      ['play', 'pause', 'seeked', 'timeupdate', 'ended'].forEach(eventType => {
        video.addEventListener(eventType, () => {
          dispatchResponse('videoEvent', {
            event: eventType,
            state: getCurrentState()
          });
        });
      });
      
      console.log('🎬 CoWatch Max: Video monitor attached');
    }
  };
  
  const observer = new MutationObserver(() => {
    if (getVideoElement()) {
      setupVideoMonitor();
    }
  });
  
  if (document.body) {
    observer.observe(document.body, { childList: true, subtree: true });
    setupVideoMonitor();
  } else {
    document.addEventListener('DOMContentLoaded', () => {
      observer.observe(document.body, { childList: true, subtree: true });
      setupVideoMonitor();
    });
  }
})();
