// CoWatch Hulu Injected Script
// Based on Teleparty's hulu_injected_bundled.js approach
// Uses React internals to access Hulu's player state and ad management
(function() {
  'use strict';
  
  // Mark script as loaded (Teleparty pattern)
  if (window.seekScriptLoaded) {
    console.log('🎬 CoWatch Hulu: Script already loaded');
    return;
  }
  window.seekScriptLoaded = true;
  window.offset = 0;
  window.lastPlayedTime = 0;
  
  // Safari detection (more accurate method from Teleparty)
  const IS_SAFARI = /^((?!chrome|android).)*safari/i.test(navigator.userAgent) || typeof safari !== 'undefined';
  
  // Sync constants
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  
  /**
   * Get Hulu player via React internals (Teleparty exact approach)
   */
  function getHuluPlayer() {
    try {
      const elementRoot = document.querySelector(".Timeline__sliderContainer");
      if (!elementRoot) return null;
      
      const keys = Object.keys(elementRoot);
      let key = null;
      
      for (let i = 0; i < keys.length; i++) {
        if (keys[i].startsWith("__reactInternalInstance")) {
          key = keys[i];
          break;
        }
      }
      
      if (!key || typeof elementRoot[key] === "undefined") {
        return null;
      }
      
      // Teleparty's exact path to player
      return elementRoot[key]?.child?.alternate?.sibling?.stateNode?.context?.storeState?.player;
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: getHuluPlayer error:', e);
      return null;
    }
  }
  
  /**
   * Check if we're at an ad break start (Teleparty approach)
   */
  function checkForceAd() {
    try {
      const player = getHuluPlayer();
      if (!player) return false;
      
      const adBreaks = player.adsManager?.getAdBreaks();
      if (!adBreaks || adBreaks.length === 0) return false;
      
      const currentTime = Math.round(player.currentTime);
      
      for (const adBreak of adBreaks) {
        if (currentTime >= adBreak.startTime - 1 && currentTime <= adBreak.startTime + 1) {
          return true;
        }
      }
      
      return false;
    } catch (e) {
      return false;
    }
  }
  
  /**
   * Calculate ad offset (Teleparty approach)
   * Accounts for watched ad time in total playback position
   */
  function getOffset() {
    try {
      const playerEl = getHuluPlayer();
      const videoEl = document.querySelector("#content-video-player");
      const currentTime = videoEl?.currentTime;
      
      let offset = 0;
      
      if (playerEl) {
        const adBreaks = playerEl.adsManager?.getAdBreaks();
        if (adBreaks && adBreaks.length > 0) {
          for (const adBreak of adBreaks) {
            if (currentTime >= adBreak.streamTimeEnd) {
              offset += adBreak.duration;
            } else if (currentTime >= adBreak.streamTimeStart && currentTime <= adBreak.streamTimeEnd) {
              offset += currentTime - adBreak.streamTimeStart;
            }
          }
        }
      }
      
      return offset;
    } catch (e) {
      return 0;
    }
  }
  
  /**
   * Get current time with Safari handling (Teleparty approach)
   */
  function getCurrentTime() {
    try {
      const videoEl = document.querySelector("#content-video-player");
      
      if (IS_SAFARI) {
        const offset = getOffset();
        const updatedTime = Math.round((videoEl?.currentTime || 0) - offset);
        const huluPlayer = getHuluPlayer();
        
        if (huluPlayer && Math.abs(huluPlayer.currentTime - updatedTime) < 30) {
          return Math.round(huluPlayer.currentTime);
        }
        return updatedTime;
      } else {
        const huluPlayer = getHuluPlayer();
        if (huluPlayer) {
          return Math.round(huluPlayer.currentTime);
        }
        return Math.round(videoEl?.currentTime || 0);
      }
    } catch (e) {
      const videoEl = document.querySelector("video");
      return Math.round(videoEl?.currentTime || 0);
    }
  }
  
  /**
   * Get video element
   */
  function getVideoElement() {
    return document.querySelector("#content-video-player") || 
           document.querySelector("video");
  }
  
  /**
   * Get ad breaks info
   */
  function getAdBreaks() {
    try {
      const player = getHuluPlayer();
      return player?.adsManager?.getAdBreaks() || [];
    } catch (e) {
      return [];
    }
  }
  
  /**
   * Check if currently in ad
   */
  function isInAd() {
    try {
      const player = getHuluPlayer();
      if (!player) return false;
      
      const adBreaks = player.adsManager?.getAdBreaks();
      if (!adBreaks || adBreaks.length === 0) return false;
      
      const videoEl = document.querySelector("#content-video-player");
      const currentTime = videoEl?.currentTime || 0;
      
      for (const adBreak of adBreaks) {
        if (currentTime >= adBreak.streamTimeStart && currentTime <= adBreak.streamTimeEnd) {
          return true;
        }
      }
      
      return false;
    } catch (e) {
      return false;
    }
  }
  
  /**
   * Dispatch response via FromNode CustomEvent (Teleparty pattern)
   */
  function dispatchResponse(type, data) {
    const evt = new CustomEvent("FromNode", {
      detail: {
        type: type,
        ...data,
        updatedAt: Date.now()
      }
    });
    window.dispatchEvent(evt);
  }
  
  /**
   * Dispatch FromInjected event (Teleparty pattern for ads)
   */
  function dispatchFromInjected(type, data) {
    const evt = new CustomEvent("FromInjected", {
      detail: {
        type: type,
        ...data,
        updatedAt: Date.now()
      }
    });
    window.dispatchEvent(evt);
  }
  
  /**
   * Get content ID from URL
   */
  function getContentId() {
    const match = window.location.pathname.match(/watch\/([a-zA-Z0-9-]+)/);
    return match ? match[1] : null;
  }
  
  /**
   * Get current playback state
   */
  function getCurrentState() {
    try {
      const player = getHuluPlayer();
      const video = getVideoElement();
      const adBreaks = getAdBreaks();
      
      if (!video && !player) return null;
      
      const currentTime = getCurrentTime();
      
      return {
        time: currentTime,
        duration: player?.duration || video?.duration || 0,
        paused: player?.paused ?? video?.paused ?? true,
        playbackRate: video?.playbackRate || 1,
        videoId: getContentId(),
        title: document.title?.replace(' • Hulu', '').replace(' - Hulu', '') || '',
        adPlaying: isInAd(),
        adBreaks: adBreaks.map(ab => ({
          startTime: ab.startTime,
          duration: ab.duration,
          streamTimeStart: ab.streamTimeStart,
          streamTimeEnd: ab.streamTimeEnd
        })),
        offset: getOffset()
      };
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: getCurrentState error:', e);
      return null;
    }
  }
  
  /**
   * Play video
   */
  function playVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.play();
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: play error:', e);
      return false;
    }
  }
  
  /**
   * Pause video
   */
  function pauseVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.pause();
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: pause error:', e);
      return false;
    }
  }
  
  /**
   * Seek to time (Teleparty approach with fallback)
   */
  function seekTo(timeInSeconds) {
    try {
      const player = getHuluPlayer();
      
      if (player) {
        player.seek(Math.round(timeInSeconds));
        return true;
      }
      
      // Fallback to HuluDashPlayer
      const videoEl = document.querySelector("#content-video-player");
      if (videoEl?.__HuluDashPlayer__) {
        videoEl.__HuluDashPlayer__.currentTime = Math.round(timeInSeconds);
        return true;
      }
      
      // Last resort: direct video element
      const video = getVideoElement();
      if (video) {
        video.currentTime = timeInSeconds;
        return true;
      }
      
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: seek error:', e);
      return false;
    }
  }
  
  /**
   * Navigate to next episode using Next.js router (Teleparty approach)
   */
  function navigateToNextEpisode(nextEpisodeId) {
    try {
      if (window.next?.router) {
        window.next.router.push({
          pathname: "/",
          query: { entity: "watch", id: nextEpisodeId },
          shallow: true
        });
        return true;
      }
      
      // Fallback to direct navigation
      window.location.href = `/watch/${nextEpisodeId}`;
      return true;
    } catch (e) {
      console.warn('🎬 CoWatch Hulu: navigateToNextEpisode error:', e);
      return false;
    }
  }
  
  /**
   * Main message handler (Teleparty pattern)
   */
  function seekInteraction(event) {
    if (event.source !== window) return;
    
    const eventType = event.data?.type;
    if (!eventType) return;
    
    switch (eventType) {
      case "SEEK": {
        // Teleparty sends time in milliseconds
        const timeInSeconds = event.data.time ? Math.round(event.data.time / 1000) : 0;
        seekTo(timeInSeconds);
        break;
      }
      
      case "UpdateState": {
        const player = getHuluPlayer();
        const currentTime = getCurrentTime() * 1000;
        dispatchResponse("UpdateState", {
          paused: player?.paused ?? true,
          currentTime: currentTime
        });
        break;
      }
      
      case "CHECK_FORCE_AD": {
        if (checkForceAd()) {
          dispatchFromInjected("CHECK_FORCE_AD", { forceAd: true });
        }
        break;
      }
      
      case "ON_AD_START": {
        const player = getHuluPlayer();
        const adBreaks = player?.adsManager?.getAdBreaks();
        if (player && adBreaks?.length > 0) {
          window.lastPlayedTime = document.querySelector("video")?.currentTime || 0;
        }
        break;
      }
      
      case "ON_AD_END": {
        const player = getHuluPlayer();
        if (player && window.lastPlayedTime > 0) {
          window.lastPlayedTime = 0;
        }
        break;
      }
      
      case "teardown": {
        window.removeEventListener("message", seekInteraction, false);
        window.seekScriptLoaded = false;
        console.log("🎬 CoWatch Hulu: Teardown");
        break;
      }
      
      case "nextEpisode": {
        if (event.data.nextEpisodeId) {
          navigateToNextEpisode(event.data.nextEpisodeId);
        }
        break;
      }
    }
  }
  
  /**
   * CoWatch message handler
   */
  function handleCoWatchMessage(event) {
    if (event.data?.source !== 'cowatch-content') return;
    
    const { action, data } = event.data;
    let result = null;
    
    switch (action) {
      case 'getState':
        result = getCurrentState();
        break;
      case 'play':
        result = playVideo();
        break;
      case 'pause':
        result = pauseVideo();
        break;
      case 'seek':
        result = seekTo(data?.time);
        break;
      case 'getDuration':
        const player = getHuluPlayer();
        const video = getVideoElement();
        result = player?.duration || video?.duration || 0;
        break;
      case 'getAdBreaks':
        result = getAdBreaks();
        break;
      case 'nextEpisode':
        if (data?.nextEpisodeId) {
          result = navigateToNextEpisode(data.nextEpisodeId);
        }
        break;
    }
    
    // Send response back
    window.postMessage({
      source: 'cowatch-injected',
      action: action + 'Response',
      data: result
    }, '*');
    
    // Also dispatch via FromNode
    dispatchResponse(action + 'Response', { data: result });
  }
  
  // Setup event listeners
  console.log("🎬 CoWatch Hulu: Script loaded (Teleparty-enhanced)");
  
  // Teleparty-style message handler
  window.addEventListener("message", seekInteraction, false);
  
  // CoWatch message handler
  window.addEventListener("message", handleCoWatchMessage, false);
  
  // Monitor for video element
  const setupVideoMonitor = () => {
    const video = getVideoElement();
    if (video && !video._cowatchMonitored) {
      video._cowatchMonitored = true;
      
      ['play', 'pause', 'seeked', 'timeupdate', 'ended'].forEach(eventType => {
        video.addEventListener(eventType, () => {
          dispatchResponse('videoEvent', {
            event: eventType,
            state: getCurrentState()
          });
        });
      });
      
      console.log('🎬 CoWatch Hulu: Video monitor attached');
    }
  };
  
  const observer = new MutationObserver(() => {
    if (getVideoElement()) {
      setupVideoMonitor();
    }
  });
  
  if (document.body) {
    observer.observe(document.body, { childList: true, subtree: true });
    setupVideoMonitor();
  } else {
    document.addEventListener('DOMContentLoaded', () => {
      observer.observe(document.body, { childList: true, subtree: true });
      setupVideoMonitor();
    });
  }
})();
