// CoWatch Max (HBO Max) Injected Script
// This script runs in the PAGE CONTEXT to access Max's internal API
// Based on Teleparty's proven approach - includes ad interception
(function() {
  'use strict';

  // Prevent double initialization
  if (window.cowatchMaxLoaded) return;
  window.cowatchMaxLoaded = true;

  console.log('🎬 CoWatch Max: Injected script loaded');

  // ============== SYNC CONSTANTS ==============
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;

  // ============== REACT INTERNAL ACCESS ==============
  function getReactFiber(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith('__reactFiber')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  function getReactProps(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith('__reactProps')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  // ============== AD DETECTION ==============
  function getAdListFromAttribute() {
    const scrubber = document.querySelector('[data-testid="player-ux-scrubber-track"]');
    if (!scrubber) return [];
    
    const adSlots = scrubber.getAttribute('data-ad-slots');
    if (adSlots) {
      try {
        return JSON.parse(adSlots);
      } catch (e) {
        return [];
      }
    }
    return [];
  }

  function getAdListFromProps() {
    const scrubber = document.querySelector('[data-testid="player-ux-scrubber-track"]');
    if (!scrubber) return [];
    
    const props = getReactProps(scrubber);
    if (props && props['data-ad-slots']) {
      try {
        return JSON.parse(props['data-ad-slots']);
      } catch (e) {
        return [];
      }
    }
    return [];
  }

  function getAllAds() {
    let ads = getAdListFromAttribute();
    if (ads.length === 0) {
      ads = getAdListFromProps();
    }
    return ads;
  }

  // ============== VIDEO METADATA ==============
  function getMetadata() {
    try {
      const metadataEl = document.querySelector('[data-testid="content-metadata"]');
      if (!metadataEl) return null;
      
      const fiber = getReactFiber(metadataEl);
      if (fiber) {
        return fiber.return?.return?.memoizedProps?.contentMetadata;
      }
    } catch (e) {
      console.warn('🎬 CoWatch Max: getMetadata failed:', e);
    }
    return null;
  }

  function getVideoLookupData() {
    const metadata = getMetadata();
    if (!metadata) return null;

    const title = metadata.title;
    const videoType = metadata.videoType; // 'EPISODE' or 'FEATURE'
    const extras = metadata.extras || {};

    const result = {
      title: title,
      type: videoType === 'EPISODE' ? 'TV' : 'MOVIE'
    };

    if (videoType === 'EPISODE') {
      result.episodeLookup = {
        number: extras.EPISODE_NUMBER,
        season: extras.SEASON_NUMBER,
        airDate: extras.AIR_DATE
      };
    } else {
      // For movies, get year from AIR_DATE
      if (extras.AIR_DATE) {
        result.releaseYear = extras.AIR_DATE.split('-')[0];
      }
    }

    return result;
  }

  // ============== PLAYER CONTROLS ==============
  function getPlayerControls() {
    try {
      const overlayRoot = document.querySelector('#overlay-root');
      if (!overlayRoot) return null;
      
      const fiber = getReactFiber(overlayRoot);
      if (!fiber) return null;
      
      return fiber.return?.pendingProps?.playerCallbacks?.mediator?._playerControls;
    } catch (e) {
      return null;
    }
  }

  // ============== SSAI INTERCEPTION (AD BLOCKING) ==============
  function interceptNetworkRequests() {
    // Prune SSAI (Server-Side Ad Insertion) data from API responses
    const pruneJson = (jsonText) => {
      try {
        const json = JSON.parse(jsonText);
        if (typeof json !== 'object' || json === null) return jsonText;

        // Remove ssaiInfo if present
        if ('ssaiInfo' in json) {
          delete json.ssaiInfo;
        }

        // Remove fallback.ssaiInfo if present
        if ('fallback' in json && typeof json.fallback === 'object') {
          delete json.fallback.ssaiInfo;
        }

        // Remove manifest.url if present (ad manifests)
        if ('manifest' in json && typeof json.manifest === 'object') {
          delete json.manifest.url;
        }

        return JSON.stringify(json);
      } catch (e) {
        return jsonText;
      }
    };

    // Intercept fetch API
    const originalFetch = window.fetch;
    window.fetch = async (...args) => {
      const response = await originalFetch(...args);
      
      if (response.ok && 
          response.headers.get('content-type')?.includes('application/json') &&
          response.url.includes('api.max.com')) {
        const modifiedJson = pruneJson(await response.clone().text());
        return new Response(modifiedJson, {
          status: response.status,
          statusText: response.statusText,
          headers: response.headers
        });
      }
      
      return response;
    };

    // Intercept XMLHttpRequest
    const originalXHROpen = XMLHttpRequest.prototype.open;
    XMLHttpRequest.prototype.open = function(...args) {
      originalXHROpen.apply(this, args);
      
      const url = args[1];
      if (url && url.includes('api.max.com')) {
        this.addEventListener('readystatechange', function() {
          if (this.readyState === 4 && 
              this.getResponseHeader('content-type')?.includes('application/json')) {
            try {
              const modifiedResponse = pruneJson(this.responseText);
              Object.defineProperty(this, '_responseText', {
                value: modifiedResponse,
                configurable: true
              });
              Object.defineProperty(this, 'responseText', {
                get: function() { return this._responseText; },
                configurable: true
              });
              Object.defineProperty(this, 'response', {
                get: function() { return this._responseText; },
                configurable: true
              });
            } catch (e) {
              console.warn('🎬 CoWatch Max: XHR prune error:', e);
            }
          }
        });
      }
    };
  }

  // ============== DISPATCH TO CONTENT ==============
  function dispatchToContent(detail) {
    const evt = new CustomEvent('FromNode', { detail });
    window.dispatchEvent(evt);
  }

  // ============== MESSAGE HANDLER ==============
  window.addEventListener('MaxVideoMessage', function(event) {
    if (!event.detail) return;
    
    const type = event.detail.type;
    
    switch (type) {
      case 'getAdList':
        const adList = getAllAds();
        dispatchToContent({ type: 'getAd', adList });
        break;

      case 'getVideoLookupData':
        const data = getVideoLookupData();
        dispatchToContent({ type: 'getData', data });
        break;
    }
  });

  // Generic CoWatch message handler
  window.addEventListener('message', function(e) {
    if (e.source !== window) return;
    
    const type = e.data?.type;
    if (!type || !type.startsWith('cowatch:')) return;

    switch (type) {
      case 'cowatch:getState': {
        const video = document.querySelector('video');
        if (video) {
          dispatchToContent({
            type: 'cowatch:state',
            time: video.currentTime * 1000,
            paused: video.paused,
            duration: video.duration * 1000,
            ads: getAllAds()
          });
        }
        break;
      }

      case 'cowatch:play': {
        const video = document.querySelector('video');
        if (video) video.play();
        break;
      }

      case 'cowatch:pause': {
        const video = document.querySelector('video');
        if (video) video.pause();
        break;
      }

      case 'cowatch:seek': {
        const video = document.querySelector('video');
        if (video) {
          const offsetTime = Math.max(0, e.data.time - SEEK_OFFSET_MS);
          video.currentTime = offsetTime / 1000;
        }
        break;
      }

      case 'cowatch:getMetadata': {
        dispatchToContent({
          type: 'cowatch:metadata',
          data: getVideoLookupData()
        });
        break;
      }
    }
  });

  // ============== VIDEO EVENT LISTENERS ==============
  function setupVideoListeners() {
    const video = document.querySelector('video');
    if (!video) {
      setTimeout(setupVideoListeners, 500);
      return;
    }

    video.addEventListener('play', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'play' });
    });

    video.addEventListener('pause', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'pause' });
    });

    video.addEventListener('seeked', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'seek' });
    });

    video.addEventListener('waiting', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: true });
    });

    video.addEventListener('playing', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: false });
    });
  }

  // ============== INITIALIZATION ==============
  function init() {
    console.log('🎬 CoWatch Max: Initializing...');
    
    // Intercept network requests for ad blocking
    interceptNetworkRequests();
    
    // Setup video listeners
    setupVideoListeners();
    
    console.log('🎬 CoWatch Max: Initialization complete');
  }

  init();
})();
