// CoWatch Netflix Injected Script
// This script runs in the PAGE CONTEXT to access Netflix's internal API
// Based on Teleparty's proven approach
(function() {
  'use strict';

  // Prevent double initialization
  if (window.cowatchNetflixLoaded) return;
  window.cowatchNetflixLoaded = true;

  console.log('🎬 CoWatch Netflix: Injected script loaded');

  // ============== REACT INTERNAL ACCESS ==============
  
  // Get React Internal Instance (older React versions)
  function getReactInternals(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith('__reactInternalInstance')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  // Get React Fiber (newer React versions)
  function getReactFiber(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith('__reactFiber')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  // Get Watch Video Wrapper State Node
  function getWrapperStateNode() {
    const watchVideoWrapper = document.querySelector('.watch-video');
    if (watchVideoWrapper) {
      const internals = getReactFiber(watchVideoWrapper);
      if (internals) {
        try {
          return internals.return.return.return.return.stateNode;
        } catch (e) {
          return null;
        }
      }
    }
    return null;
  }

  // ============== NETFLIX VIDEO PLAYER API ==============

  // Get Netflix Video Player (CRITICAL - Teleparty's method)
  function getVideoPlayer() {
    try {
      const api = window.netflix?.appContext?.state?.playerApp?.getAPI();
      if (!api?.videoPlayer) return null;
      
      const playerSessionIds = api.videoPlayer.getAllPlayerSessionIds();
      const watchSession = playerSessionIds.find(id => id.includes('watch'));
      
      if (watchSession) {
        return api.videoPlayer.getVideoPlayerBySessionId(watchSession);
      }
      return null;
    } catch (e) {
      console.warn('🎬 CoWatch Netflix: getVideoPlayer failed:', e);
      return null;
    }
  }

  // Get Player API (through React Fiber)
  function getPlayerApi() {
    try {
      const reactFiber = getReactFiber(document.querySelector('.watch-video'));
      if (reactFiber) {
        return reactFiber.return.return.return.return.return.stateNode ||
               reactFiber.return.return.return.return.stateNode;
      }
    } catch (e) {
      return null;
    }
    return null;
  }

  // ============== METADATA FUNCTIONS ==============

  function getCurrentMetaData() {
    try {
      return getWrapperStateNode()?.state?.activeVideoMetadata?._metadata?._metadataObject?.video;
    } catch (e) {
      return undefined;
    }
  }

  function getReleaseYear() {
    try {
      const meta = getCurrentMetaData();
      return meta?.year || meta?.seasons?.[0]?.year;
    } catch (e) {
      return undefined;
    }
  }

  function isMovie() {
    try {
      const wrapperStateNode = getWrapperStateNode();
      if (wrapperStateNode) {
        return wrapperStateNode.state?.playableData?.summary?.type === 'movie';
      }
    } catch (e) {}
    return false;
  }

  function getVideoLookupData() {
    return {
      title: getCurrentMetaData()?.title,
      year: getReleaseYear(),
      type: isMovie() ? 'MOVIE' : 'TV'
    };
  }

  function getEpisodeInformation() {
    try {
      const wrapper = getWrapperStateNode();
      return {
        title: wrapper?.state?.activeVideoMetadata?._video?.title,
        seasonNum: wrapper?.state?.activeVideoMetadata?._season?._season?.seq,
        episodeNum: wrapper?.state?.activeVideoMetadata?._video?.seq
      };
    } catch (e) {
      return undefined;
    }
  }

  // ============== AD STATE ==============

  function getNextAdBreak() {
    try {
      const adBreaks = getPlayerApi()?.getAdBreaks();
      const currentTime = getVideoPlayer()?.getCurrentTime();
      
      if (!adBreaks || adBreaks.length === 0) return undefined;
      
      let closestAdBreak = 0;
      let minDiff = Infinity;
      
      adBreaks.forEach(adBreak => {
        if (adBreak.locationMs) {
          const diff = adBreak.locationMs - currentTime;
          if (diff > 0 && diff < minDiff) {
            minDiff = diff;
            closestAdBreak = adBreak.locationMs;
          }
        }
      });
      
      return closestAdBreak !== 0 ? closestAdBreak : undefined;
    } catch (e) {
      return undefined;
    }
  }

  function getAdState() {
    try {
      const currentAdBreak = getPlayerApi()?.state?.playbackState?.currentAdBreak;
      if (currentAdBreak) {
        return {
          watchingAds: true,
          adDurationLeft: currentAdBreak.progress?.adBreakOffset?.ms || 0,
          nextAdBreak: getNextAdBreak()
        };
      }
    } catch (e) {}
    
    return {
      watchingAds: false,
      adDurationLeft: 0,
      nextAdBreak: getNextAdBreak()
    };
  }

  // ============== PLAYBACK STATE ==============

  function getActionsState() {
    const nextEpisodeButton = document.querySelector("[data-uia='next-episode-seamless-button']");
    return {
      nextEpisodeReady: !!nextEpisodeButton
    };
  }

  // ============== SYNC CONSTANTS (Teleparty approach) ==============
  const SYNC_THRESHOLD_SECONDS = 2; // Only sync if >2 seconds out of sync
  const SEEK_OFFSET_MS = 100; // Subtract from seek target to avoid boundary issues
  const SYNC_INTERVAL_MS = 5000; // Periodic sync check interval

  // ============== UTILITY FUNCTIONS ==============

  function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  async function showControlsAsync() {
    try {
      const wrapper = getPlayerWrapper();
      if (wrapper) {
        const reactInstance = getReactInternals(wrapper);
        if (reactInstance?.memoizedProps?.onPointerMoveCapture) {
          reactInstance.memoizedProps.onPointerMoveCapture({
            stopPropagation: () => {},
            preventDefault: () => {}
          });
          await delay(2);
        }
      }
    } catch (e) {}
  }

  function getPlayerWrapper() {
    const selectors = [
      'div[data-uia="player"]',
      'div[data-videoid]',
      'div.ltr-fntwn3',
      '.active',
      '.inactive',
      '.passive'
    ];
    
    for (const selector of selectors) {
      const element = document.querySelector(selector);
      if (element) return element;
    }
    
    try {
      return document.querySelector('.watch-video--player-view')?.children[0];
    } catch (e) {}
    
    return null;
  }

  // Skip supplemental videos (previews)
  async function checkSkipSupplemental() {
    try {
      await new Promise((resolve, reject) => {
        const startTime = Date.now();
        const check = () => {
          try {
            if (getWrapperStateNode()?.state?.activeVideoMetadata?._video) {
              resolve();
            } else if (Date.now() - startTime > 2500) {
              reject();
            } else {
              setTimeout(check, 250);
            }
          } catch (e) {
            setTimeout(check, 250);
          }
        };
        check();
      });

      if (getWrapperStateNode()?.state?.activeVideoMetadata?._video?.type === 'supplemental') {
        console.log('🎬 CoWatch Netflix: Skipping supplemental video');
        getWrapperStateNode().handleFinishPrePlay?.();
      }
    } catch (e) {}
  }

  // Change episode
  async function changeEpisodeFallback(id) {
    try {
      const fakeEvent = { stopPropagation: () => {} };
      const api = getPlayerApi();
      api?.handleSelectorEpisodePlay?.(fakeEvent, id);
    } catch (e) {
      console.log('🎬 CoWatch Netflix: changeEpisode failed:', e);
    }
  }

  // Disable post-play for movies
  function tryDisablePostPlay() {
    if (isMovie()) {
      const wrapperStateNode = getWrapperStateNode();
      if (wrapperStateNode) {
        window.oldHasPostPlay = wrapperStateNode.hasPostPlay;
        wrapperStateNode.hasPostPlay = () => false;
        console.log('🎬 CoWatch Netflix: Disabled post-play for movie');
      }
    }
  }

  function teardownFixPostPlay() {
    const wrapperStateNode = getWrapperStateNode();
    if (wrapperStateNode && window.oldHasPostPlay) {
      wrapperStateNode.hasPostPlay = window.oldHasPostPlay;
    }
  }

  // ============== MESSAGE HANDLER (Teleparty Protocol Compatible) ==============

  function handleMessage(e) {
    // Check source (both cowatch and window for compatibility)
    if (e.source !== window) return;
    if (!e.data) return;
    
    // Handle both CoWatch and Teleparty-style messages
    const isCoWatch = e.data?.source === 'cowatch-content';
    const type = isCoWatch ? (e.data.action?.toUpperCase() || '') : (e.data.type || '');
    
    if (!type) return;
    
    try {
      const player = getVideoPlayer();
      
      switch (type) {
        case 'SEEK':
          if (player) {
            const time = e.data.time || e.data.data?.time;
            // Teleparty uses -100ms offset to prevent boundary issues
            if (time >= player.getDuration()) {
              player.pause();
              player.seek(player.getDuration() - SEEK_OFFSET_MS);
            } else {
              player.seek(Math.max(0, time - SEEK_OFFSET_MS));
            }
          }
          break;

        case 'PAUSE':
          if (player) player.pause();
          break;

        case 'PLAY':
          if (player) player.play();
          break;

        case 'FIX_POST_PLAY':
          tryDisablePostPlay();
          break;

        case 'ISPAUSED':
        case 'GETSTATE':
        case 'UPDATESTATE':
          if (player) {
            const paused = player.isPaused();
            const time = player.getSegmentTime?.() ?? player.getCurrentTime();
            const loading = player.getBusy() !== null;
            const adState = getAdState();
            const actionsState = getActionsState();
            
            // Send response using CustomEvent (Teleparty style)
            dispatchResponse('UpdateState', {
              time,
              paused,
              loading,
              adState,
              actionsState,
              updatedAt: Date.now()
            });
            
            // Also send CoWatch style response
            if (isCoWatch) {
              sendCowatchResponse('getStateResponse', {
                time: time / 1000, // Convert to seconds
                duration: player.duration / 1000,
                paused,
                loading,
                videoId: getVideoIdFromUrl(),
                title: getCurrentMetaData()?.title || document.title?.replace(' - Netflix', '')
              });
            }
          }
          break;

        case 'GETCURRENTTIME':
          if (player) {
            const time = player.getSegmentTime?.() ?? player.getCurrentTime();
            dispatchResponse('CurrentTime', { time, updatedAt: Date.now() });
          }
          break;

        case 'TEARDOWN':
          teardownFixPostPlay();
          window.removeEventListener('message', handleMessage, false);
          window.cowatchNetflixLoaded = false;
          break;

        case 'NEXT_EPISODE':
          const videoId = e.data.videoId || e.data.data?.videoId;
          if (videoId) {
            changeEpisodeFallback(videoId);
          }
          break;

        case 'SHOWCONTROLS':
          showControlsAsync();
          break;

        case 'CHECKSKIPSUPPLEMENTAL':
          checkSkipSupplemental().then(() => {
            dispatchResponse('CheckSkipSupplemental', { updatedAt: Date.now() });
          });
          break;

        case 'GETPAGETITLE':
          dispatchResponse('GetTitle', {
            pageTitle: getCurrentMetaData()?.title,
            updatedAt: Date.now()
          });
          break;

        case 'GETVIDEOTYPE':
          dispatchResponse('GetType', {
            VideoType: isMovie() ? 'Movie' : 'Episode',
            updatedAt: Date.now()
          });
          break;

        case 'GETEPISODEDATA':
          const episodeData = getEpisodeInformation();
          dispatchResponse('GetEpData', {
            episodeData,
            updatedAt: Date.now()
          });
          break;

        case 'GETVIDEOLOOKUPDATA':
          const videoLookupData = getVideoLookupData();
          dispatchResponse('GetVideoData', {
            videoLookupData,
            updatedAt: Date.now()
          });
          break;
      }
    } catch (error) {
      console.log('🎬 CoWatch Netflix: Message handler error:', error);
    }
  }

  function getVideoIdFromUrl() {
    const match = window.location.pathname.match(/watch\/(\d+)/);
    return match ? match[1] : null;
  }

  // Dispatch CustomEvent (Teleparty style)
  function dispatchResponse(type, data) {
    const evt = new CustomEvent('FromNode', {
      detail: { type, ...data }
    });
    window.dispatchEvent(evt);
  }

  // Send CoWatch style response
  function sendCowatchResponse(action, data) {
    window.postMessage({
      source: 'cowatch-injected',
      action,
      data
    }, '*');
  }

  // ============== INITIALIZATION ==============

  function init() {
    console.log('🎬 CoWatch Netflix: Initializing...');
    
    // Check for supplemental videos
    checkSkipSupplemental();
    
    // Listen for messages
    window.addEventListener('message', handleMessage, false);
    
    console.log('🎬 CoWatch Netflix: Ready');
  }

  // Initialize
  init();

})();
