// CoWatch Peacock Injected Script
// This script runs in the PAGE CONTEXT to access Peacock's internal API
// Based on Teleparty's proven approach
(function() {
  'use strict';

  // Prevent double initialization
  if (window.cowatchPeacockLoaded) return;
  window.cowatchPeacockLoaded = true;

  console.log('🎬 CoWatch Peacock: Injected script loaded');

  // ============== SYNC CONSTANTS ==============
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;

  // ============== VARIABLES ==============
  let checkForAdsInterval;
  let isInAd = false;

  // ============== REACT INTERNAL ACCESS ==============
  function getReactFiber(root) {
    if (!root) return null;
    const keys = Object.keys(root);
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith('__reactFiber$')) {
        return root[keys[i]];
      }
    }
    return null;
  }

  // Get scrubber API for seeking
  function getScrubberApi() {
    try {
      const scrubber = document.querySelector('.playback-scrubber-bar');
      if (!scrubber) return null;
      const internals = getReactFiber(scrubber);
      return internals?.return?.memoizedProps;
    } catch (e) {
      return null;
    }
  }

  // Get player API for state
  function getPlayerApi() {
    try {
      const container = document.querySelector('.playback-container');
      if (!container) return null;
      const internals = getReactFiber(container);
      return internals?.return?.memoizedProps;
    } catch (e) {
      return null;
    }
  }

  // Get next episode API
  function getNextEpisodeApi() {
    try {
      const bingeContainer = document.querySelector('.playback-binge__container');
      if (!bingeContainer) return null;
      const internals = getReactFiber(bingeContainer);
      return internals?.return?.memoizedProps;
    } catch (e) {
      return null;
    }
  }

  // ============== STATE FUNCTIONS ==============
  function getPlayerState() {
    const playerState = getPlayerApi();
    if (!playerState) return null;

    return {
      isPaused: playerState.sessionState === 'PAUSED',
      time: playerState.elapsedTime || 0,
      isLoading: playerState.isLoading,
      isSeeking: playerState.sessionState === 'SEEKING',
      isAdPlaying: playerState.isAdPlaying,
      startOfCredits: playerState.startOfCredits,
      duration: playerState.duration
    };
  }

  // ============== AD DETECTION ==============
  function checkAdStatus() {
    const state = getPlayerState();
    if (!state) return;

    const currentlyInAd = state.isAdPlaying;
    
    if (currentlyInAd && !isInAd) {
      isInAd = true;
      // Start frequent checking during ads
      if (!checkForAdsInterval) {
        checkForAdsInterval = setInterval(checkAdStatus, 1000);
      }
      dispatchToContent({ type: 'cowatch:adStart' });
    } else if (!currentlyInAd && isInAd) {
      isInAd = false;
      // Stop frequent checking when ads end
      if (checkForAdsInterval) {
        clearInterval(checkForAdsInterval);
        checkForAdsInterval = null;
      }
      dispatchToContent({ type: 'cowatch:adEnd' });
    }
  }

  // ============== SEEK FUNCTION ==============
  function seekTo(timeMs) {
    const scrubberApi = getScrubberApi();
    if (scrubberApi?.seekAndUpdateElapsedTime) {
      // Apply offset for precision
      const offsetTime = Math.max(0, timeMs - SEEK_OFFSET_MS);
      scrubberApi.seekAndUpdateElapsedTime(offsetTime);
      console.log('🎬 CoWatch Peacock: Seeking to', offsetTime);
    }
  }

  // ============== NEXT EPISODE ==============
  function getNextEpisodeId() {
    const api = getNextEpisodeApi();
    if (api?.bingePopUpAsset) {
      return `${api.bingePopUpAsset.contentId}/${api.bingePopUpAsset.providerVariantId}`;
    }
    return null;
  }

  // ============== DISPATCH TO CONTENT ==============
  function dispatchToContent(detail) {
    const evt = new CustomEvent('FromNode', { detail });
    window.dispatchEvent(evt);
  }

  // ============== MESSAGE HANDLER ==============
  function handleMessage(e) {
    try {
      if (e.source !== window) return;
      
      const messageType = e.data?.type;
      if (!messageType) return;

      switch (messageType) {
        case 'cowatch:getState':
        case 'GetState': {
          const state = getPlayerState();
          if (state) {
            dispatchToContent({
              type: 'UpdateState',
              isPaused: state.isPaused,
              time: state.time,
              isLoading: state.isLoading,
              isSeeking: state.isSeeking,
              isAdPlaying: state.isAdPlaying,
              startOfCredits: state.startOfCredits,
              duration: state.duration,
              updatedAt: Date.now()
            });

            // Check for ad status
            checkAdStatus();
          }
          break;
        }

        case 'cowatch:seek':
        case 'Seek': {
          const time = e.data.time;
          seekTo(time);
          
          const playerState = getPlayerApi();
          dispatchToContent({
            type: 'Seek',
            isPaused: playerState?.sessionState === 'PAUSED',
            updatedAt: Date.now()
          });
          break;
        }

        case 'cowatch:next':
        case 'NextEpisode': {
          const nextId = getNextEpisodeId();
          dispatchToContent({
            type: 'NextEpisode',
            nextEpisodeId: nextId,
            updatedAt: Date.now()
          });
          break;
        }

        case 'cowatch:play': {
          const video = document.querySelector('video');
          if (video) video.play();
          break;
        }

        case 'cowatch:pause': {
          const video = document.querySelector('video');
          if (video) video.pause();
          break;
        }
      }
    } catch (error) {
      console.error('🎬 CoWatch Peacock: Message handler error:', error);
    }
  }

  // ============== VIDEO EVENT LISTENERS ==============
  function setupVideoListeners() {
    const video = document.querySelector('video');
    if (!video) {
      // Retry if video not found
      setTimeout(setupVideoListeners, 500);
      return;
    }

    video.addEventListener('play', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'play' });
    });

    video.addEventListener('pause', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'pause' });
    });

    video.addEventListener('seeked', () => {
      dispatchToContent({ type: 'cowatch:userInteraction', action: 'seek' });
    });

    video.addEventListener('waiting', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: true });
    });

    video.addEventListener('playing', () => {
      dispatchToContent({ type: 'cowatch:buffering', isBuffering: false });
    });
  }

  // ============== INITIALIZATION ==============
  function init() {
    console.log('🎬 CoWatch Peacock: Initializing...');
    
    window.addEventListener('message', handleMessage);
    setupVideoListeners();

    // Initial ad check
    setTimeout(checkAdStatus, 1000);

    console.log('🎬 CoWatch Peacock: Initialization complete');
  }

  init();
})();
