// CoWatch Prime Video Injected Script
// Based on Teleparty's amazon_injected_bundled.js approach
// Uses React internals to access Amazon's player state
(function() {
  'use strict';
  
  // Mark script as loaded (Teleparty pattern)
  if (window.videoIdScriptLoaded) {
    console.log('🎬 CoWatch Prime Video: Script already loaded');
    return;
  }
  window.videoIdScriptLoaded = true;
  
  // Sync constants (Teleparty approach)
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  
  /**
   * Get React internals from element (Teleparty exact approach)
   */
  function getReactInternals(root) {
    if (root == null) return null;
    
    const keys = Object.keys(root);
    let key = null;
    
    for (let i = 0; i < keys.length; i++) {
      if (keys[i].startsWith("__reactInternalInstance")) {
        key = keys[i];
        break;
      }
    }
    
    return key ? root[key] : null;
  }
  
  /**
   * Find title/video ID (Teleparty exact approach)
   * Accesses Amazon's internal React state for video ID
   */
  function findTitle() {
    try {
      const elementRoots = document.querySelectorAll(".atvwebplayersdk-title-text");
      let elementRoot = null;
      let elementRootFallback = null;
      
      // Find the title element with actual text content
      for (let i = 0; i < elementRoots.length; i++) {
        if (elementRoots[i].innerText) {
          elementRoot = elementRoots[i];
        } else if (elementRoots[i].clientHeight > 0) {
          elementRootFallback = elementRoots[i];
        }
      }
      
      if (!elementRoot) {
        if (elementRoots.length === 0) return null;
        elementRoot = elementRootFallback || elementRoots[0];
      }
      
      const keys = Object.keys(elementRoot);
      let key = null;
      
      for (let i = 0; i < keys.length; i++) {
        if (keys[i].startsWith("__reactInternalInstance")) {
          key = keys[i];
          break;
        }
      }
      
      if (!key || typeof elementRoot[key] === "undefined") {
        return null;
      }
      
      // Teleparty's exact path to title ID
      const titlePath = elementRoot[key].return?.return?.stateNode?.context?.stores?.pin?.currentTitleId ||
                        elementRoot[key].return?.return?.stateNode?.context?.stores?.pin?.clickstreamReporter?.asin;
      
      return titlePath || null;
    } catch (err) {
      console.warn('🎬 CoWatch Prime Video: findTitle error:', err);
      return null;
    }
  }
  
  /**
   * Get video lookup data (title, type, year)
   * Teleparty exact approach
   */
  function getVideoLookupData() {
    try {
      const titleEls = document.querySelectorAll(".atvwebplayersdk-title-text");
      let title = null;
      
      [...titleEls].forEach((titleEl) => {
        if (titleEl?.textContent) {
          title = titleEl.textContent;
        }
      });
      
      // Alternative title element
      const altTitleEl = document.querySelector('[data-automation-id="title"]');
      if (!title && altTitleEl) {
        title = altTitleEl.textContent;
      }
      
      // Determine content type by checking for next episode button
      const nextEl = document.querySelector(".atvwebplayersdk-nexttitle-button");
      const type = nextEl ? "TV" : "MOVIE";
      
      // Get year info
      let episodeYear = undefined;
      let releaseYear = undefined;
      const yearEl = document.querySelector('[data-automation-id="release-year-badge"]');
      
      if (yearEl) {
        if (type === "TV") {
          episodeYear = yearEl.textContent;
        } else {
          releaseYear = yearEl.textContent;
        }
      }
      
      return { title, type, episodeYear, releaseYear };
    } catch (err) {
      console.warn('🎬 CoWatch Prime Video: getVideoLookupData error:', err);
      return null;
    }
  }
  
  /**
   * Get playback props from React internals (Teleparty getPropsAlt approach)
   * This is the most complex part - accessing Amazon's internal state
   */
  function getPropsAlt() {
    try {
      const parentEls = document.querySelectorAll(".webPlayerSDKContainer");
      let el = null;
      
      // Find the container with actual video
      [...parentEls].forEach((parentEl) => {
        if (parentEl.querySelector("video")?.src) {
          el = parentEl;
        }
      });
      
      if (!el) {
        el = document;
      }
      
      // Get ad playback props
      let rootAdProps = el.querySelector(".webPlayerUIContainer")?._reactRootContainer?._internalRoot?.current?.memoizedState?.element?.props?.context?.adPlayback;
      
      // Get timeline props
      let rootTimelineProps = el.querySelector(".webPlayerUIContainer")?._reactRootContainer?._internalRoot?.current?.child?.stateNode?.props?.context?.seekBar?.seekBar;
      
      // Get play state props
      let rootPlayProps = el.querySelector(".webPlayerUIContainer")?._reactRootContainer?._internalRoot?.current?.child?.memoizedState;
      
      // Fullscreen fallback
      if (!el.querySelector(".webPlayerUIContainer")) {
        const altEl = document.querySelector(".dv-player-fullscreen");
        if (altEl) {
          rootAdProps = altEl._reactRootContainer?._internalRoot?.current?.memoizedState?.element?.props?.context?.adPlayback;
          rootTimelineProps = altEl._reactRootContainer?._internalRoot?.current?.child?.stateNode?.props?.context?.seekBar?.seekBar;
          rootPlayProps = altEl._reactRootContainer?._internalRoot?.current?.child?.memoizedState;
        }
      }
      
      // Get main props from player container
      let rootMainProps = null;
      try {
        const internals = getReactInternals(el.querySelector(".atvwebplayersdk-player-container"));
        rootMainProps = internals?.child?.child?.child?.child?.sibling?.child?.return?.sibling?.sibling?.sibling?.memoizedState;
      } catch {
        const internals = getReactInternals(el.querySelector(".atvwebplayersdk-player-container"));
        rootMainProps = internals?.child?.child?.child?.child?.sibling?.sibling?.sibling?.memoizedState;
      }
      
      // Combine all props
      let playbackProps = {
        ...rootMainProps,
        currentAdInfo: rootAdProps?.currentAdInfo || null
      };
      
      // Use timeline props as fallback for position data
      if (!playbackProps?.lastPlayablePositionMs && rootTimelineProps?.lastPlayablePositionMs) {
        playbackProps.lastPlayablePositionMs = rootTimelineProps.lastPlayablePositionMs;
        playbackProps.positionMs = rootTimelineProps.positionMs;
      }
      
      // Use play props as fallback for state
      if (rootPlayProps?.playbackState) {
        if (playbackProps?.state === undefined) {
          playbackProps.state = rootPlayProps.playbackState;
        }
      }
      
      return playbackProps;
    } catch (err) {
      console.warn('🎬 CoWatch Prime Video: getPropsAlt error:', err);
      return null;
    }
  }
  
  /**
   * Dispatch response via FromNode CustomEvent (Teleparty pattern)
   */
  function dispatchResponse(type, data) {
    const evt = new CustomEvent("FromNode", {
      detail: {
        type: type,
        ...data,
        updatedAt: Date.now()
      }
    });
    window.dispatchEvent(evt);
  }
  
  /**
   * Get video element
   */
  function getVideoElement() {
    const containers = document.querySelectorAll(".webPlayerSDKContainer");
    for (const container of containers) {
      const video = container.querySelector("video");
      if (video?.src) return video;
    }
    return document.querySelector("video");
  }
  
  /**
   * Get current state (CoWatch format)
   */
  function getCurrentState() {
    try {
      const video = getVideoElement();
      const props = getPropsAlt();
      const lookupData = getVideoLookupData();
      
      if (!video && !props) return null;
      
      return {
        time: props?.positionMs ? props.positionMs / 1000 : (video?.currentTime || 0),
        duration: props?.lastPlayablePositionMs ? props.lastPlayablePositionMs / 1000 : (video?.duration || 0),
        paused: props?.state !== 'PLAYING' && (video?.paused ?? true),
        playbackRate: video?.playbackRate || 1,
        videoId: findTitle(),
        title: lookupData?.title || document.title?.replace(' - Prime Video', '').replace(' | Prime Video', '') || '',
        contentType: lookupData?.type,
        adPlaying: props?.currentAdInfo !== null,
        state: props?.state
      };
    } catch (e) {
      console.warn('🎬 CoWatch Prime Video: getCurrentState error:', e);
      return null;
    }
  }
  
  /**
   * Play video
   */
  function playVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.play();
        return true;
      }
      
      // Try clicking play button
      const playButton = document.querySelector('.atvwebplayersdk-playpause-button');
      if (playButton) {
        playButton.click();
        return true;
      }
      
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Prime Video: play error:', e);
      return false;
    }
  }
  
  /**
   * Pause video
   */
  function pauseVideo() {
    try {
      const video = getVideoElement();
      if (video) {
        video.pause();
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Prime Video: pause error:', e);
      return false;
    }
  }
  
  /**
   * Seek to time (in seconds)
   */
  function seekTo(time) {
    try {
      const video = getVideoElement();
      if (video) {
        video.currentTime = time;
        return true;
      }
      return false;
    } catch (e) {
      console.warn('🎬 CoWatch Prime Video: seek error:', e);
      return false;
    }
  }
  
  /**
   * Amazon Video Message handler (Teleparty pattern)
   */
  function handleAmazonVideoMessage(evt) {
    const type = evt.detail?.type;
    
    if (type === "getVideoId") {
      const videoId = findTitle();
      if (videoId) {
        dispatchResponse("VideoId", { videoId });
      }
    }
    else if (type === "getVideoLookupData") {
      const data = getVideoLookupData();
      dispatchResponse("GetLookupData", { data });
    }
  }
  
  /**
   * Update state handler (Teleparty pattern)
   */
  function handleUpdateState(event) {
    if (event.source === window && event.data?.type === "UpdateState") {
      const props = getPropsAlt();
      if (props) {
        dispatchResponse("UpdateState", {
          duration: props.lastPlayablePositionMs,
          currentTime: props.positionMs,
          state: props.state,
          adPlaying: props.currentAdInfo !== null
        });
      }
    }
  }
  
  /**
   * CoWatch message handler
   */
  function handleCoWatchMessage(event) {
    if (event.data?.source !== 'cowatch-content') return;
    
    const { action, data } = event.data;
    let result = null;
    
    switch (action) {
      case 'getState':
        result = getCurrentState();
        break;
      case 'play':
        result = playVideo();
        break;
      case 'pause':
        result = pauseVideo();
        break;
      case 'seek':
        result = seekTo(data?.time);
        break;
      case 'getDuration':
        const video = getVideoElement();
        const props = getPropsAlt();
        result = props?.lastPlayablePositionMs ? props.lastPlayablePositionMs / 1000 : (video?.duration || 0);
        break;
      case 'getVideoId':
        result = findTitle();
        break;
      case 'getVideoLookupData':
        result = getVideoLookupData();
        break;
    }
    
    // Send response back
    window.postMessage({
      source: 'cowatch-injected',
      action: action + 'Response',
      data: result
    }, '*');
    
    // Also dispatch via FromNode
    dispatchResponse(action + 'Response', { data: result });
  }
  
  // Setup event listeners
  console.log("🎬 CoWatch Prime Video: Script loaded (Teleparty-enhanced)");
  
  // Teleparty-style Amazon event
  window.addEventListener("AmazonVideoMessage", handleAmazonVideoMessage);
  
  // Update state message
  window.addEventListener("message", handleUpdateState, false);
  
  // CoWatch message handler
  window.addEventListener("message", handleCoWatchMessage, false);
  
  // Monitor for video element and setup event forwarding
  const setupVideoMonitor = () => {
    const video = getVideoElement();
    if (video && !video._cowatchMonitored) {
      video._cowatchMonitored = true;
      
      // Forward video events to content script
      ['play', 'pause', 'seeked', 'timeupdate', 'ended'].forEach(eventType => {
        video.addEventListener(eventType, () => {
          dispatchResponse('videoEvent', {
            event: eventType,
            state: getCurrentState()
          });
        });
      });
      
      console.log('🎬 CoWatch Prime Video: Video monitor attached');
    }
  };
  
  // Watch for video element
  const observer = new MutationObserver(() => {
    if (getVideoElement()) {
      setupVideoMonitor();
    }
  });
  
  if (document.body) {
    observer.observe(document.body, { childList: true, subtree: true });
    setupVideoMonitor();
  } else {
    document.addEventListener('DOMContentLoaded', () => {
      observer.observe(document.body, { childList: true, subtree: true });
      setupVideoMonitor();
    });
  }
})();
