// CoWatch Sling TV Injected Script
// Based on Teleparty's approach - React Fiber access for seek and state
(function() {
  'use strict';

  if (window.cowatchSlingLoaded) return;
  window.cowatchSlingLoaded = true;

  console.log('📺 CoWatch Sling: Injected script loaded');

  // ============== REACT FIBER ACCESS ==============
  
  function getReactFiber(element) {
    if (!element) return null;
    const keys = Object.keys(element);
    for (const key of keys) {
      if (key.startsWith('__reactFiber')) {
        return element[key];
      }
    }
    return null;
  }

  // ============== SEEK FUNCTION ==============
  
  function findSeekUtils() {
    try {
      const el = document.querySelector('[data-testid="seekable"]');
      const elementRoot = el?.parentElement;
      if (!elementRoot) return null;

      const fiber = getReactFiber(elementRoot);
      if (!fiber) return null;

      const children = fiber.pendingProps?.children;
      if (children && children.props?.children?.length > 1) {
        const seekFn = children.props.children[1].props.debouncedSeekForwardOrBackWard;
        const duration = children.props.children[1].props.duration;
        const currentPosition = children.props.children[1].props.currentPosition;
        
        if (seekFn) {
          return { seekFn, duration, currentPosition };
        }
      }
      return null;
    } catch (e) {
      console.warn('CoWatch Sling: findSeekUtils error:', e);
      return null;
    }
  }

  // ============== NAVIGATOR ==============

  function findNavigator() {
    try {
      const element = document.querySelector('.base-screen');
      if (!element) return null;

      const fiber = getReactFiber(element);
      if (!fiber) return null;

      // Navigate through React Fiber tree to find router
      let current = fiber?.alternate?.child?.child?.alternate?.child?.dependencies?.firstContext;
      for (let i = 0; i < 7 && current?.next; i++) {
        current = current.next;
      }
      return current?.memoizedValue?.navigator;
    } catch (e) {
      return null;
    }
  }

  // ============== SEEK ==============

  function seek(ms) {
    const seekUtils = findSeekUtils();
    if (seekUtils) {
      // Only seek if significant difference
      if (ms > 6000 && Math.abs(ms - seekUtils.currentPosition) > 2500) {
        seekUtils.seekFn(ms - seekUtils.currentPosition);
      }
    }
  }

  // ============== VIDEO ELEMENT ACCESS ==============

  function getVideo() {
    return document.querySelector('video');
  }

  // ============== MESSAGE HANDLER ==============

  function handleMessage(event) {
    if (event.source !== window) return;
    
    const { type, time, timestamp, path } = event.data;
    if (!type) return;

    console.log('📺 CoWatch Sling: Message received:', type);

    switch (type) {
      case 'COWATCH_GET_STATE': {
        const video = getVideo();
        const seekUtils = findSeekUtils();
        
        if (video) {
          window.dispatchEvent(new CustomEvent('COWATCH_STATE', {
            detail: {
              currentTime: seekUtils?.currentPosition || (video.currentTime * 1000),
              duration: seekUtils?.duration || (video.duration * 1000),
              paused: video.paused,
              playbackRate: video.playbackRate,
              platform: 'sling'
            }
          }));
        }
        break;
      }

      case 'COWATCH_SEEK': {
        seek(time || timestamp);
        break;
      }

      case 'COWATCH_PLAY': {
        const video = getVideo();
        if (video) video.play();
        break;
      }

      case 'COWATCH_PAUSE': {
        const video = getVideo();
        if (video) video.pause();
        break;
      }

      case 'COWATCH_NEXT_EPISODE': {
        const nextButton = document.querySelector('.skipToLiveFocused');
        if (nextButton) {
          nextButton.click();
        } else {
          const navigator = findNavigator();
          if (navigator?.push && path) {
            try {
              navigator.push(path);
            } catch (e) {
              window.location.href = path;
            }
          } else if (path) {
            window.location.href = path;
          }
        }
        break;
      }

      case 'COWATCH_TEARDOWN': {
        window.removeEventListener('message', handleMessage);
        window.cowatchSlingLoaded = false;
        break;
      }
    }
  }

  // ============== INITIALIZATION ==============

  window.addEventListener('message', handleMessage);

  // Auto-reconnect listener every 25 seconds (Teleparty pattern)
  let lastActivity = new Date();
  setInterval(() => {
    if (new Date() - lastActivity <= 60000) {
      window.removeEventListener('message', handleMessage);
      window.addEventListener('message', handleMessage);
    }
  }, 25000);

  // Notify content script that injected script is ready
  window.postMessage({ type: 'COWATCH_INJECTED_READY', platform: 'sling' }, '*');
})();
