// CoWatch Spotify Injected Script
// Based on Teleparty's approach - React Fiber access for music playback sync
(function() {
  'use strict';

  if (window.cowatchSpotifyLoaded) return;
  window.cowatchSpotifyLoaded = true;

  console.log('🎵 CoWatch Spotify: Injected script loaded');

  // ============== REACT FIBER ACCESS ==============
  
  function fetchReactProp(selector) {
    const el = document.querySelector(selector);
    if (!el) return null;
    
    const keys = Object.keys(el);
    for (const key of keys) {
      if (key.startsWith('__reactFiber')) {
        return el[key];
      }
    }
    return null;
  }

  function fetchHistoryProp() {
    try {
      const el = document.querySelector('#main');
      if (!el) return null;

      const keys = Object.keys(el);
      for (const key of keys) {
        if (key.startsWith('__reactContainer')) {
          const container = el[key];
          let current = container?.child;
          for (let i = 0; i < 9 && current?.child; i++) {
            current = current.child;
          }
          return current?.memoizedProps?.history;
        }
      }
      return null;
    } catch (e) {
      return null;
    }
  }

  // ============== SEEK FUNCTION ==============
  
  function getSeekFunction() {
    try {
      const el = fetchReactProp('[data-testid="playback-progressbar"]');
      return el?.return?.memoizedProps?.onDragEnd;
    } catch (e) {
      return null;
    }
  }

  // ============== SONG DETAILS ==============

  function getSongDetails() {
    try {
      const el = fetchReactProp('[data-testid="now-playing-widget"] [data-testid="context-item-info-title"]');
      return el?.memoizedProps?.children?.props?.item;
    } catch (e) {
      return null;
    }
  }

  function getCurrentTrackId() {
    const songData = getSongDetails();
    if (songData?.uri) {
      const parts = songData.uri.split(':');
      return parts[parts.length - 1];
    }
    return null;
  }

  // ============== AUDIO ELEMENT ==============

  function getAudioElement() {
    return document.querySelector('audio') || document.querySelector('video');
  }

  // ============== PLAYBACK CONTROLS ==============

  function clickPlayButton() {
    const playButton = document.querySelector('[data-testid="control-button-playpause"]');
    if (playButton) playButton.click();
  }

  function clickPauseButton() {
    const pauseButton = document.querySelector('[data-testid="control-button-playpause"]');
    if (pauseButton) pauseButton.click();
  }

  function isPaused() {
    const playButton = document.querySelector('[data-testid="control-button-playpause"]');
    if (playButton) {
      // Check if play icon is visible (meaning currently paused)
      return playButton.getAttribute('aria-label')?.toLowerCase().includes('play');
    }
    const audio = getAudioElement();
    return audio?.paused ?? true;
  }

  // ============== MESSAGE HANDLER ==============

  function handleMessage(event) {
    if (event.source !== window) return;
    
    const { type, time, percentage, trackId, url } = event.data;
    if (!type) return;

    console.log('🎵 CoWatch Spotify: Message received:', type);

    switch (type) {
      case 'COWATCH_GET_STATE': {
        const audio = getAudioElement();
        const currentTrackId = getCurrentTrackId();
        
        window.dispatchEvent(new CustomEvent('COWATCH_STATE', {
          detail: {
            currentTime: audio ? audio.currentTime * 1000 : 0,
            duration: audio ? audio.duration * 1000 : 0,
            paused: isPaused(),
            trackId: currentTrackId,
            platform: 'spotify'
          }
        }));
        break;
      }

      case 'COWATCH_SEEK': {
        const seekFn = getSeekFunction();
        if (seekFn && percentage !== undefined) {
          seekFn(percentage, { wasDraggedBeforeReleased: false });
        }
        break;
      }

      case 'COWATCH_PLAY': {
        if (isPaused()) {
          clickPlayButton();
        }
        break;
      }

      case 'COWATCH_PAUSE': {
        if (!isPaused()) {
          clickPauseButton();
        }
        break;
      }

      case 'COWATCH_CHANGE_TRACK': {
        const history = fetchHistoryProp();
        if (history && trackId) {
          history.push(`/track/${trackId}`);
        }
        break;
      }

      case 'COWATCH_NAVIGATE': {
        const history = fetchHistoryProp();
        if (history && url) {
          history.push(url);
        }
        break;
      }

      case 'COWATCH_VALIDATE_TRACK': {
        // Ensure URL matches currently playing track
        const songData = getSongDetails();
        if (songData?.uri) {
          const parts = songData.uri.split(':');
          const history = fetchHistoryProp();
          const pathname = `/track/${parts[parts.length - 1]}`;
          
          if (history && pathname !== window.location.pathname) {
            history.push(pathname);
          }
        }
        break;
      }

      case 'COWATCH_TEARDOWN': {
        window.removeEventListener('message', handleMessage);
        window.cowatchSpotifyLoaded = false;
        break;
      }
    }
  }

  // ============== HISTORY STATE INTERCEPTION ==============

  if (!window.cowatchHistoryPatched) {
    window.cowatchHistoryPatched = true;
    
    const originalPushState = history.pushState;
    const originalReplaceState = history.replaceState;
    
    history.pushState = function(state) {
      window.postMessage({ type: 'COWATCH_URL_CHANGED', url: arguments[2] }, '*');
      return originalPushState.apply(history, arguments);
    };
    
    history.replaceState = function(state) {
      window.postMessage({ type: 'COWATCH_URL_CHANGED', url: arguments[2] }, '*');
      return originalReplaceState.apply(history, arguments);
    };
    
    window.addEventListener('popstate', () => {
      window.postMessage({ type: 'COWATCH_URL_CHANGED', url: window.location.href }, '*');
    });
  }

  // ============== INITIALIZATION ==============

  window.addEventListener('message', handleMessage);

  // Notify content script that injected script is ready
  window.postMessage({ type: 'COWATCH_INJECTED_READY', platform: 'spotify' }, '*');
})();
