// CoWatch YouTube Injected Script
// This script runs in the page context to access YouTube's internal API
// Based on Teleparty's proven approach with video resize functionality
(function() {
  'use strict';
  
  // Prevent double initialization
  if (window.cowatchYoutubeLoaded) return;
  window.cowatchYoutubeLoaded = true;
  
  console.log('🎬 CoWatch YouTube: Injected script loaded');
  
  // Chat panel width for video resize calculations (match CSS)
  const CHAT_WIDTH = 340;
  const RESIZE_ADJUST = 14;
  
  // Sync constants (Teleparty approach)
  const SYNC_THRESHOLD_SECONDS = 2;
  const SEEK_OFFSET_MS = 100;
  
  // Track if chat is visible for resize purposes
  let resizeVisible = false;
  let resizeScriptReady = false;
  
  // ============== YOUTUBE PLAYER ACCESS ==============
  
  // Get video element (supports both /watch and /shorts/)
  function getVideoElement() {
    const url = window.location.href;
    
    if (url.includes('/watch')) {
      return document.getElementById('movie_player');
    } else if (url.includes('/shorts/')) {
      return document.getElementById('shorts-player');
    }
    return null;
  }
  
  // Check if on a player page
  function isPlayerPage() {
    return window.location.href.includes('/watch') || 
           window.location.href.includes('/shorts/');
  }
  
  // Get video ID from URL
  function getVideoIdFromUrl() {
    const url = new URL(window.location.href);
    if (url.pathname.includes('/watch')) {
      return url.searchParams.get('v');
    } else if (url.pathname.includes('/shorts/')) {
      return url.pathname.split('/shorts/')[1]?.split('?')[0];
    }
    return null;
  }
  
  // ============== VIDEO RESIZE (Teleparty approach) ==============
  
  function resizeYoutube(ytPlayer) {
    const video = document.getElementById('movie_player');
    
    Object.getOwnPropertyNames(ytPlayer).forEach((prop) => {
      const obj = ytPlayer[prop];
      
      // Find the size function: function(a,b){this.width=a;this.height=b}
      if (obj && obj.toString && obj.toString() === 'function(a,b){this.width=a;this.height=b}') {
        const oldObj = obj;
        
        window._yt_player[prop] = function(width, height) {
          // Only resize if fullscreen AND chat visible
          if (video && video.isFullscreen && video.isFullscreen() && resizeVisible) {
            const aspectRatio = height / width;
            width = width - (CHAT_WIDTH + RESIZE_ADJUST);
            height = aspectRatio * width;
          }
          return new oldObj(width, height);
        };
        
        window._yt_player[prop].prototype = oldObj.prototype;
        resizeScriptReady = true;
        console.log('🎬 CoWatch YouTube: Video resize hook installed');
      }
    });
  }
  
  // ============== THEATER MODE ==============
  
  function setTheater() {
    const watchFlexy = document.querySelector('ytd-watch-flexy');
    if (watchFlexy && watchFlexy.theater !== null) {
      const theaterButton = document.querySelector('.ytp-size-button');
      if (theaterButton && !watchFlexy.theater) {
        theaterButton.click();
        theaterButton.style.display = 'none';
      }
      if (theaterButton && watchFlexy.theater) {
        theaterButton.style.display = 'none';
      }
    }
  }
  
  function disableTheater() {
    const theaterButton = document.querySelector('.ytp-size-button');
    if (theaterButton) {
      theaterButton.style.display = '';
    }
  }
  
  // ============== SPA NAVIGATION (for next video) ==============
  
  function jumpToNextVideo(nextVideoId) {
    const urlPath = `/watch?v=${nextVideoId}`;
    const navigationData = {
      endpoint: {
        commandMetadata: {
          webCommandMetadata: {
            url: urlPath,
            rootVe: 3832,
            webPageType: 'WEB_PAGE_TYPE_WATCH'
          },
          watchEndpoint: {
            videoId: nextVideoId,
            nofollow: true
          }
        }
      }
    };
    
    const ytNavigator = document.querySelector('ytd-app');
    if (ytNavigator && ytNavigator.fire) {
      ytNavigator.fire('yt-navigate', navigationData);
      dispatchResponse('Navigated', { videoId: nextVideoId });
      return true;
    }
    
    // Fallback: direct navigation
    window.location.href = urlPath;
    return true;
  }
  
  // ============== RESPONSE DISPATCH ==============
  
  // Dispatch CustomEvent (Teleparty style)
  function dispatchResponse(type, data) {
    const evt = new CustomEvent('FromNode', {
      detail: { type, ...data, updatedAt: Date.now() }
    });
    window.dispatchEvent(evt);
  }
  
  // Send CoWatch style response
  function sendCowatchResponse(action, data) {
    window.postMessage({
      source: 'cowatch-injected',
      action,
      data
    }, '*');
  }
  
  // ============== MESSAGE HANDLER ==============
  
  function handleMessage(event) {
    if (!event.detail && event.data?.source !== 'cowatch-content') return;
    
    // Handle both Teleparty-style (CustomEvent) and CoWatch-style (postMessage)
    const isYoutubeEvent = event.type === 'YoutubeVideoMessage';
    const isCowatchEvent = event.data?.source === 'cowatch-content';
    
    if (!isYoutubeEvent && !isCowatchEvent) return;
    
    const type = isYoutubeEvent ? event.detail?.type : event.data?.action?.toUpperCase();
    const data = isYoutubeEvent ? event.detail : event.data?.data;
    
    if (!type) return;
    
    // Hook resize if not ready
    if (!resizeScriptReady && window._yt_player) {
      resizeYoutube(window._yt_player);
    }
    
    const video = getVideoElement();
    
    try {
      switch (type) {
        case 'pauseVideo':
        case 'PAUSE':
          if (video && video.pauseVideo) {
            video.pauseVideo();
          }
          break;
          
        case 'playVideo':
        case 'PLAY':
          if (video && video.playVideo) {
            video.playVideo();
          }
          break;
          
        case 'getVideoTitle':
        case 'GETTITLE':
          if (video && video.getVideoData) {
            const videoData = video.getVideoData();
            dispatchResponse('VideoTitle', { title: videoData.title });
            if (isCowatchEvent) {
              sendCowatchResponse('getTitleResponse', { title: videoData.title });
            }
          }
          break;
          
        case 'setTheater':
        case 'SETTHEATER':
          setTheater();
          break;
          
        case 'disableTheater':
        case 'DISABLETHEATER':
          disableTheater();
          break;
          
        case 'getVideoId':
        case 'GETVIDEOID':
          if (video && video.getVideoData) {
            const videoData = video.getVideoData();
            dispatchResponse('VideoId', { 
              videoId: videoData.video_id, 
              isLive: videoData.isLive 
            });
            if (isCowatchEvent) {
              sendCowatchResponse('getVideoIdResponse', { 
                videoId: videoData.video_id, 
                isLive: videoData.isLive 
              });
            }
          }
          break;
          
        case 'seekTo':
        case 'SEEK':
          if (video && video.seekTo) {
            const time = data?.seekTo || data?.time;
            video.seekTo(time, true);
          }
          break;
          
        case 'getVideoTime':
        case 'GETSTATE':
        case 'GETCURRENTTIME':
          if (video) {
            const currentTime = video.getCurrentTime ? video.getCurrentTime() : 0;
            const duration = video.getDuration ? video.getDuration() : 0;
            const playerState = video.getPlayerState ? video.getPlayerState() : -1;
            const paused = playerState !== 1;
            const videoData = video.getVideoData ? video.getVideoData() : {};
            
            dispatchResponse('VideoTime', { 
              videoTime: currentTime,
              time: currentTime,
              duration: duration,
              paused: paused,
              isLive: videoData.isLive || false
            });
            
            if (isCowatchEvent) {
              sendCowatchResponse('getStateResponse', {
                time: currentTime,
                duration: duration,
                paused: paused,
                videoId: videoData.video_id || getVideoIdFromUrl(),
                title: videoData.title || document.title?.replace(' - YouTube', ''),
                isLive: videoData.isLive || false
              });
            }
          }
          break;
          
        case 'jumpToNextEpisode':
        case 'NEXTVIDEO':
          const nextVideoId = data?.nextVideoId || data?.videoId;
          if (nextVideoId) {
            jumpToNextVideo(nextVideoId);
          }
          break;
          
        case 'SetChatVisible':
        case 'SETCHATVISIBLE':
          resizeVisible = data?.visible ?? data;
          if (video) {
            if (video.setSize) video.setSize();
            if (video.setInternalSize) video.setInternalSize();
          }
          break;
          
        case 'SETPLAYBACKRATE':
          if (video && video.setPlaybackRate) {
            video.setPlaybackRate(data?.rate || 1);
          }
          break;
      }
    } catch (error) {
      console.log('🎬 CoWatch YouTube: Message handler error:', error);
    }
  }
  
  // ============== INITIALIZATION ==============
  
  function init() {
    console.log('🎬 CoWatch YouTube: Initializing...');
    
    // Listen for Teleparty-style events
    window.addEventListener('YoutubeVideoMessage', handleMessage, false);
    
    // Listen for CoWatch-style postMessage
    window.addEventListener('message', handleMessage, false);
    
    // Try to hook resize immediately if _yt_player available
    if (window._yt_player) {
      resizeYoutube(window._yt_player);
    }
    
    console.log('🎬 CoWatch YouTube: Ready');
  }
  
  // Initialize
  init();

})();
