// CoWatch Popup - Teleparty Style Flow
// v1.5.1 - Production Ready with Platform Logos

const COWATCH_URL = 'https://cowatch.app';

// Browser API abstraction
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
const storage = browserAPI.storage?.local || chrome.storage.local;
const tabs = browserAPI.tabs || chrome.tabs;

// Platform information with logos (light/dark theme support)
const PLATFORM_INFO = {
  'netflix.com': {
    name: 'Netflix',
    color: '#E50914',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/0/08/Netflix_2015_logo.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Netflix_Full_Logo_Red.svg',
  },
  'youtube.com': {
    name: 'YouTube',
    color: '#FF0000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/b/b8/YouTube_Logo_2017.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/4/43/YouTube_2024_%28White%29.svg',
  },
  'disneyplus.com': {
    name: 'Disney+',
    color: '#113CCF',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/3/3e/Disney%2B_logo.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Disney_Full_Logo_White.svg',
  },
  'primevideo.com': {
    name: 'Prime Video',
    color: '#00A8E1',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/11/Amazon_Prime_Video_logo.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Amazon_Full_Logo_White.svg',
  },
  'amazon.com': {
    name: 'Prime Video',
    color: '#00A8E1',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/11/Amazon_Prime_Video_logo.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Amazon_Full_Logo_White.svg',
  },
  'max.com': {
    name: 'Max',
    color: '#002BE7',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Max_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Max_logo.svg',
    invertOnDark: true,
  },
  'hbomax.com': {
    name: 'Max',
    color: '#002BE7',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Max_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Max_logo.svg',
    invertOnDark: true,
  },
  'hulu.com': {
    name: 'Hulu',
    color: '#1CE783',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/f/f9/Hulu_logo_%282018%29.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Hulu_Full_Logo_Green.svg',
  },
  'peacocktv.com': {
    name: 'Peacock',
    color: '#000000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/d/d3/NBCUniversal_Peacock_Logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/d/d3/NBCUniversal_Peacock_Logo.svg',
    invertOnDark: true,
  },
  'paramountplus.com': {
    name: 'Paramount+',
    color: '#0064FF',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/a/a5/Paramount_Plus.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/a/a5/Paramount_Plus.svg',
  },
  'twitch.tv': {
    name: 'Twitch',
    color: '#9146FF',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Twitch_logo_2019.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/c/ce/Twitch_logo_2019.svg',
  },
  'crunchyroll.com': {
    name: 'Crunchyroll',
    color: '#F47521',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/0/08/Crunchyroll_Logo.png',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/0/08/Crunchyroll_Logo.png',
  },
  'vimeo.com': {
    name: 'Vimeo',
    color: '#1AB7EA',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/9/9c/Vimeo_Logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/9/9c/Vimeo_Logo.svg',
  },
  'dailymotion.com': {
    name: 'Dailymotion',
    color: '#0066DC',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/d/d5/Dailymotion_logo_%282017%29.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/d/d5/Dailymotion_logo_%282017%29.svg',
    invertOnDark: true,
  },
  'tubitv.com': {
    name: 'Tubi',
    color: '#FA382F',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/d/db/Tubi_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/d/db/Tubi_logo.svg',
    invertOnDark: true,
  },
  'pluto.tv': {
    name: 'Pluto TV',
    color: '#FFCC00',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/5/5d/Pluto_TV_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/5/5d/Pluto_TV_logo.svg',
    invertOnDark: true,
  },
  'tv.apple.com': {
    name: 'Apple TV+',
    color: '#000000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/2/28/Apple_TV_Plus_Logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/2/28/Apple_TV_Plus_Logo.svg',
    invertOnDark: true,
  },
  'mubi.com': {
    name: 'MUBI',
    color: '#001D3D',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/6/68/MUBI_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/6/68/MUBI_logo.svg',
    invertOnDark: true,
  },
  'stan.com.au': {
    name: 'Stan',
    color: '#0060A8',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/0/0a/Stan_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/0/0a/Stan_logo.svg',
    invertOnDark: true,
  },
  'crave.ca': {
    name: 'Crave',
    color: '#00C9C8',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/f/f4/Crave_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/f/f4/Crave_logo.svg',
    invertOnDark: true,
  },
  'hotstar.com': {
    name: 'Hotstar',
    color: '#1F80E0',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/1e/Disney%2B_Hotstar_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/1/1e/Disney%2B_Hotstar_logo.svg',
  },
  'viki.com': {
    name: 'Viki',
    color: '#1A90D9',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/3/36/Viki_logo_2019.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/3/36/Viki_logo_2019.svg',
  },
  'espn.com': {
    name: 'ESPN+',
    color: '#D00000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/e/eb/ESPN_Plus.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/e/eb/ESPN_Plus.svg',
    invertOnDark: true,
  },
  'fubo.tv': {
    name: 'FuboTV',
    color: '#E5362A',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/17/FuboTV_Logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/1/17/FuboTV_Logo.svg',
    invertOnDark: true,
  },
  'zee5.com': {
    name: 'ZEE5',
    color: '#8230C6',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/b/b1/Zee5-official-logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/b/b1/Zee5-official-logo.svg',
  },
  'sonyliv.com': {
    name: 'SonyLIV',
    color: '#001E62',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/7/78/SonyLIV_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/7/78/SonyLIV_logo.svg',
    invertOnDark: true,
  },
  'jiocinema.com': {
    name: 'JioCinema',
    color: '#E72472',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/thumb/c/c5/JioCinema_logo.svg/512px-JioCinema_logo.svg.png',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/thumb/c/c5/JioCinema_logo.svg/512px-JioCinema_logo.svg.png',
  },
  'shahid.mbc.net': {
    name: 'Shahid',
    color: '#7AB800',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/5/5e/Shahid_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/5/5e/Shahid_logo.svg',
  },
  'viu.com': {
    name: 'Viu',
    color: '#FDBD26',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/0/0c/Viu_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/0/0c/Viu_logo.svg',
  },
  'vidio.com': {
    name: 'Vidio',
    color: '#1E88E5',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/f/ff/Vidio_logo_2023.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/f/ff/Vidio_logo_2023.svg',
  },
  // ========== New Platforms (Teleparty Parity) ==========
  'sling.com': {
    name: 'Sling TV',
    color: '#0F9DFB',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/5/53/Sling_TV_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/5/53/Sling_TV_logo.svg',
    invertOnDark: true,
  },
  'spotify.com': {
    name: 'Spotify',
    color: '#1DB954',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/8/84/Spotify_icon.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/8/84/Spotify_icon.svg',
  },
  'canalplus.com': {
    name: 'Canal+',
    color: '#000000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/1a/Canal%2B.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/1/1a/Canal%2B.svg',
    invertOnDark: true,
  },
  'mycanal.fr': {
    name: 'Canal+',
    color: '#000000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/1/1a/Canal%2B.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/1/1a/Canal%2B.svg',
    invertOnDark: true,
  },
  'starplus.com': {
    name: 'Star+',
    color: '#FF0000',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/5/5b/Star%2B_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/5/5b/Star%2B_logo.svg',
    invertOnDark: true,
  },
  'funimation.com': {
    name: 'Funimation',
    color: '#410099',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/b/b0/Funimation_2016.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/b/b0/Funimation_2016.svg',
    invertOnDark: true,
  },
  'hulu.jp': {
    name: 'Hulu Japan',
    color: '#1CE783',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/f/f9/Hulu_logo_%282018%29.svg',
    logoDark: 'https://d3g0wg4d6u8hjn.cloudfront.net/img/refactor/platform_icons/Hulu_Full_Logo_Green.svg',
  },
  'unext.jp': {
    name: 'U-NEXT',
    color: '#1F1F1F',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/5/5b/U-NEXT_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/5/5b/U-NEXT_logo.svg',
    invertOnDark: true,
  },
  'fancode.com': {
    name: 'FanCode',
    color: '#E91E63',
    logo: 'https://upload.wikimedia.org/wikipedia/commons/4/4e/FanCode_Logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/commons/4/4e/FanCode_Logo.svg',
  },
  'willow.tv': {
    name: 'Willow TV',
    color: '#00796B',
    logo: 'https://upload.wikimedia.org/wikipedia/en/c/c5/Willow_TV_logo.svg',
    logoDark: 'https://upload.wikimedia.org/wikipedia/en/c/c5/Willow_TV_logo.svg',
    invertOnDark: true,
  },
};

// Supported video platforms (domain list)
const SUPPORTED_PLATFORMS = Object.keys(PLATFORM_INFO);

class CoWatchPopup {
  constructor() {
    this.currentView = 'platforms';
    this.user = null;
    this.roomCode = null;
    this.roomName = null;
    this.isOnVideoSite = false;
    this.currentTab = null;
    this.activePopupTab = 'platforms';
    this.friends = [];
    this.pendingRequests = [];
    this.scheduledParties = [];
    this.notifications = [];
    
    this.init();
  }
  
  async init() {
    console.log('🎬 CoWatch Popup v1.6.0 initializing...');
    
    // Setup language
    this.setupLanguage();
    
    // Check current tab
    await this.checkCurrentTab();
    
    // Load saved state
    await this.loadState();
    
    // Check session from cowatch.app (fetch fresh session status)
    await this.checkRemoteSession();
    
    // Determine which view to show
    this.determineView();
    
    // Bind events
    this.bindEvents();
    
    // Setup popup tabs (Friends/Schedule)
    this.setupPopupTabs();
    
    // Load notifications if logged in
    if (this.user) {
      this.loadNotifications();
      this.loadFriends();
      this.loadScheduledParties();
    }
    
    // Update translations
    if (typeof updateAllTranslations === 'function') {
      updateAllTranslations();
    }
  }
  
  async checkRemoteSession() {
    try {
      console.log('🎬 Checking session from cowatch.app...');
      
      // Method 1: Try to fetch with credentials (may work if same-site)
      try {
        const response = await fetch(`${COWATCH_URL}/api/auth/session`, {
          method: 'GET',
          credentials: 'include',
          headers: {
            'Accept': 'application/json',
          }
        });
        const data = await response.json();
        
        if (data.user) {
          console.log('🎬 User session found via fetch:', data.user.email);
          this.user = data.user;
          await this.saveState({ user: data.user, guestMode: false });
          return true;
        }
      } catch (fetchError) {
        console.log('🎬 Fetch method failed:', fetchError.message);
      }
      
      // Method 2: Check if we have a saved user in storage (from previous successful login)
      if (this.user) {
        console.log('🎬 Using cached user from storage:', this.user.email);
        return true;
      }
      
      // Method 3: Try to communicate with cowatch.app tab if open
      try {
        const cowatchTabs = await tabs.query({ url: 'https://cowatch.app/*' });
        if (cowatchTabs.length > 0) {
          console.log('🎬 Found cowatch.app tab, checking session via content script...');
          const response = await tabs.sendMessage(cowatchTabs[0].id, { type: 'GET_SESSION' });
          if (response?.user) {
            console.log('🎬 User session found via tab:', response.user.email);
            this.user = response.user;
            await this.saveState({ user: response.user, guestMode: false });
            return true;
          }
        }
      } catch (tabError) {
        console.log('🎬 Tab communication failed:', tabError.message);
      }
      
      console.log('🎬 No user session found');
      return false;
    } catch (error) {
      console.error('🎬 Failed to check remote session:', error);
      return false;
    }
  }
  
  async handleLoginClick() {
    // First, check if user is already logged in on the website
    const isLoggedIn = await this.checkRemoteSession();
    
    if (isLoggedIn && this.user) {
      console.log('🎬 User already logged in, updating view...');
      // User is already logged in - just update the view
      if (this.isOnVideoSite) {
        this.showView('partySettings');
      } else {
        this.showView('platforms');
      }
      // Show a brief notification
      this.showToast('Already logged in as ' + this.user.email);
      return;
    }
    
    // Not logged in - open login page and listen for auth completion
    console.log('🎬 Opening login page...');
    const tab = await chrome.tabs.create({ url: `${COWATCH_URL}/auth/login?from=extension` });
    
    // Start listening for auth completion
    this.listenForAuthCompletion(tab.id);
    window.close();
  }
  
  listenForAuthCompletion(loginTabId) {
    // Store the tab ID to monitor
    chrome.storage.local.set({ pendingAuthTabId: loginTabId });
  }
  
  showToast(message, type = 'success') {
    // Remove any existing toast
    const existingToast = document.querySelector('.toast-notification');
    if (existingToast) existingToast.remove();
    
    // Create a styled toast notification matching frontend design
    const toast = document.createElement('div');
    toast.className = 'toast-notification';
    
    // Icon based on type
    const icon = type === 'success' 
      ? `<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>`
      : type === 'error'
      ? `<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>`
      : `<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="8" x2="12" y2="12"/><line x1="12" y1="16" x2="12.01" y2="16"/></svg>`;
    
    toast.innerHTML = `
      <div class="toast-icon">${icon}</div>
      <span class="toast-message">${message}</span>
    `;
    
    // Colors based on type
    const colors = {
      success: { bg: 'rgba(34, 197, 94, 0.15)', border: 'rgba(34, 197, 94, 0.3)', text: '#22c55e' },
      error: { bg: 'rgba(239, 68, 68, 0.15)', border: 'rgba(239, 68, 68, 0.3)', text: '#ef4444' },
      info: { bg: 'rgba(59, 130, 246, 0.15)', border: 'rgba(59, 130, 246, 0.3)', text: '#3b82f6' }
    };
    const color = colors[type] || colors.success;
    
    toast.style.cssText = `
      position: fixed;
      bottom: 20px;
      left: 50%;
      transform: translateX(-50%);
      display: flex;
      align-items: center;
      gap: 10px;
      background: ${color.bg};
      backdrop-filter: blur(10px);
      border: 1px solid ${color.border};
      color: #fff;
      padding: 12px 18px;
      border-radius: 12px;
      font-size: 13px;
      font-weight: 500;
      z-index: 9999;
      box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
      animation: toastSlideIn 0.3s ease-out forwards;
    `;
    
    // Add animation style
    if (!document.getElementById('toast-styles')) {
      const style = document.createElement('style');
      style.id = 'toast-styles';
      style.textContent = `
        @keyframes toastSlideIn {
          0% { opacity: 0; transform: translateX(-50%) translateY(20px); }
          100% { opacity: 1; transform: translateX(-50%) translateY(0); }
        }
        @keyframes toastSlideOut {
          0% { opacity: 1; transform: translateX(-50%) translateY(0); }
          100% { opacity: 0; transform: translateX(-50%) translateY(20px); }
        }
        .toast-notification .toast-icon {
          display: flex;
          align-items: center;
          justify-content: center;
          color: ${color.text};
        }
        .toast-notification .toast-message {
          color: #fff;
        }
      `;
      document.head.appendChild(style);
    }
    
    document.body.appendChild(toast);
    
    // Animate out and remove after delay
    setTimeout(() => {
      toast.style.animation = 'toastSlideOut 0.3s ease-in forwards';
      setTimeout(() => toast.remove(), 300);
    }, 2500);
  }
  
  async handleLogout() {
    console.log('🎬 Logging out...');
    
    try {
      // Method 1: Try to logout via cowatch.app tab (most reliable)
      const tabs = await chrome.tabs.query({ url: '*://cowatch.app/*' });
      if (tabs.length > 0) {
        console.log('🎬 Found cowatch.app tab, sending logout message...');
        try {
          await chrome.tabs.sendMessage(tabs[0].id, { type: 'LOGOUT' });
          console.log('🎬 Logout message sent to cowatch.app tab');
        } catch (tabError) {
          console.warn('🎬 Could not send to tab:', tabError.message);
        }
      }
      
      // Method 2: Also try direct fetch as fallback
      try {
        const response = await fetch(`${COWATCH_URL}/api/auth/logout`, {
          method: 'POST',
          credentials: 'include',
          headers: {
            'Content-Type': 'application/json'
          }
        });
        
        if (!response.ok) {
          console.warn('🎬 Logout API returned:', response.status);
        }
      } catch (fetchError) {
        console.warn('🎬 Logout fetch error (expected for cross-origin):', fetchError.message);
      }
      
      // Method 3: Clear cookies via background (no tab opening)
      // The fetch and tab message methods above should be sufficient
      // No need to open a visible logout page
    } catch (error) {
      console.error('🎬 Logout error:', error);
    }
    
    // Clear local storage
    this.user = null;
    this.guestMode = false;
    await chrome.storage.local.remove(['user', 'guestMode']);
    
    // Update UI
    this.updateHeaderProfile();
    
    // Close any open dropdown
    const dropdown = document.getElementById('profileDropdown');
    const profileBtn = document.getElementById('profileBtn');
    if (dropdown) dropdown.classList.add('hidden');
    if (profileBtn) profileBtn.classList.remove('active');
    
    // Reset to platforms tab
    this.activePopupTab = 'platforms';
    this.switchPopupTab('platforms');
    
    // Go back to platforms view
    this.showView('platforms');
    
    this.showToast('Logged out successfully');
  }
  
  setupLanguage() {
    const langBtn = document.getElementById('langBtn');
    const currentLangSpan = document.getElementById('currentLang');
    const langDropdown = document.getElementById('langDropdown');
    
    if (!langBtn || !currentLangSpan || !langDropdown) {
      console.error('🎬 Language elements not found');
      return;
    }
    
    // Get saved language from chrome.storage (same as i18n.js)
    chrome.storage.local.get(['language'], (result) => {
      const savedLang = result.language || localStorage.getItem('cowatch_lang') || 'en';
      console.log('🎬 Saved language:', savedLang);
      
      // Also sync to localStorage for backwards compatibility
      localStorage.setItem('cowatch_lang', savedLang);
      
      if (typeof setLanguage === 'function') {
        setLanguage(savedLang);
      }
      
      // Update button text
      currentLangSpan.textContent = savedLang.toUpperCase();
      
      // Apply translations immediately
      if (typeof updateAllTranslations === 'function') {
        updateAllTranslations();
      }
      
      // Update active state in dropdown
      langDropdown.querySelectorAll('.lang-option').forEach(opt => {
        opt.classList.toggle('active', opt.dataset.lang === savedLang);
      });
    });
    
    // Populate dropdown
    const languages = [
      { code: 'en', name: 'English' },
      { code: 'tr', name: 'Türkçe' },
      { code: 'es', name: 'Español' },
      { code: 'de', name: 'Deutsch' },
      { code: 'fr', name: 'Français' },
      { code: 'pt', name: 'Português' },
      { code: 'it', name: 'Italiano' },
      { code: 'nl', name: 'Nederlands' },
      { code: 'pl', name: 'Polski' },
      { code: 'ru', name: 'Русский' },
      { code: 'ja', name: '日本語' },
      { code: 'ko', name: '한국어' },
      { code: 'zh', name: '中文' },
      { code: 'ar', name: 'العربية' },
      { code: 'hi', name: 'हिन्दी' },
      { code: 'id', name: 'Indonesia' }
    ];
    
    langDropdown.innerHTML = languages.map(lang => `
      <button class="lang-option" data-lang="${lang.code}">
        ${lang.name}
      </button>
    `).join('');
    
    // Toggle dropdown
    langBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      langDropdown.classList.toggle('hidden');
    });
    
    // Language selection - use arrow function to capture currentLangSpan
    langDropdown.addEventListener('click', async (e) => {
      const option = e.target.closest('.lang-option');
      if (option) {
        const lang = option.dataset.lang;
        console.log('🎬 Changing language to:', lang);
        
        // Save to both localStorage (legacy) and chrome.storage (for i18n.js)
        localStorage.setItem('cowatch_lang', lang);
        await chrome.storage.local.set({ language: lang });
        
        if (typeof setLanguage === 'function') {
          await setLanguage(lang);
          console.log('🎬 setLanguage called');
        }
        
        if (typeof updateAllTranslations === 'function') {
          updateAllTranslations();
          console.log('🎬 updateAllTranslations called');
        }
        
        // Update displayed language code
        document.getElementById('currentLang').textContent = lang.toUpperCase();
        langDropdown.classList.add('hidden');
        
        // Update active state
        langDropdown.querySelectorAll('.lang-option').forEach(opt => {
          opt.classList.toggle('active', opt.dataset.lang === lang);
        });
        
        // Show toast
        this.showToast(lang === 'tr' ? 'Dil değiştirildi' : 'Language changed', 'success');
      }
    });
    
    // Close dropdown on outside click
    document.addEventListener('click', () => {
      langDropdown.classList.add('hidden');
    });
  }
  
  async checkCurrentTab() {
    try {
      const tabList = await tabs.query({ active: true, currentWindow: true });
      this.currentTab = tabList[0];
      
      if (this.currentTab?.url) {
        const url = this.currentTab.url.toLowerCase();
        this.isOnVideoSite = SUPPORTED_PLATFORMS.some(platform => url.includes(platform));
        console.log('🎬 On video site:', this.isOnVideoSite, 'URL:', this.currentTab.url);
      }
    } catch (error) {
      console.error('Failed to check tab:', error);
    }
  }
  
  async loadState() {
    return new Promise(resolve => {
      storage.get(['roomCode', 'roomName', 'user', 'guestMode'], (result) => {
        this.roomCode = result.roomCode || null;
        this.roomName = result.roomName || null;
        this.user = result.user || null;
        this.guestMode = result.guestMode || false;
        console.log('🎬 Loaded state:', { roomCode: this.roomCode, user: this.user?.email, guestMode: this.guestMode });
        resolve();
      });
    });
  }
  
  async saveState(data = {}) {
    return new Promise(resolve => {
      storage.set({
        roomCode: data.roomCode ?? this.roomCode,
        roomName: data.roomName ?? this.roomName,
        user: data.user ?? this.user,
        guestMode: data.guestMode ?? this.guestMode
      }, resolve);
    });
  }
  
  determineView() {
    // If already in a room, show party active view
    if (this.roomCode) {
      this.showView('partyActive');
      this.updatePartyActiveView();
      return;
    }
    
    // If not on a video site, show platform selection
    if (!this.isOnVideoSite) {
      this.showView('platforms');
      return;
    }
    
    // On video site - check if user/guest mode is set
    if (this.user || this.guestMode) {
      this.showView('partySettings');
    } else {
      this.showView('signup');
    }
  }
  
  showView(viewName) {
    console.log('🎬 Showing view:', viewName);
    this.currentView = viewName;
    
    // Hide all views
    document.querySelectorAll('.view').forEach(view => {
      view.classList.remove('active');
    });
    
    // Show target view
    const viewMap = {
      'platforms': 'viewPlatforms',
      'signup': 'viewSignup',
      'partySettings': 'viewPartySettings',
      'partyActive': 'viewPartyActive',
      'joinRoom': 'viewJoinRoom'
    };
    
    const targetView = document.getElementById(viewMap[viewName]);
    if (targetView) {
      targetView.classList.add('active');
    }
    
    // Update header profile based on login state
    this.updateHeaderProfile();
  }
  
  updateHeaderProfile() {
    const signInBtn = document.getElementById('googleSignIn');
    const userProfile = document.getElementById('userProfile');
    const profileAvatar = document.getElementById('profileAvatar');
    const profileAvatarInitial = document.getElementById('profileAvatarInitial');
    const dropdownAvatar = document.getElementById('dropdownAvatar');
    const dropdownAvatarInitial = document.getElementById('dropdownAvatarInitial');
    const profileName = document.getElementById('profileName');
    const profileEmail = document.getElementById('profileEmail');
    const notificationBtn = document.getElementById('notificationBtn');
    const popupTabs = document.getElementById('popupTabs');
    
    if (this.user) {
      // User is logged in - show profile and notification
      if (signInBtn) signInBtn.classList.add('hidden');
      if (userProfile) userProfile.classList.remove('hidden');
      if (notificationBtn) notificationBtn.classList.remove('hidden');
      if (popupTabs) popupTabs.classList.remove('hidden');
      
      // Get the first character of username for initial (like frontend)
      const initial = (this.user.name || 'U').charAt(0).toUpperCase();
      
      // Check if user has a Google profile image
      if (this.user.image) {
        // Show image, hide initial
        if (profileAvatar) {
          profileAvatar.src = this.user.image;
          profileAvatar.style.display = 'block';
        }
        if (profileAvatarInitial) profileAvatarInitial.style.display = 'none';
        
        if (dropdownAvatar) {
          dropdownAvatar.src = this.user.image;
          dropdownAvatar.style.display = 'block';
        }
        if (dropdownAvatarInitial) dropdownAvatarInitial.style.display = 'none';
      } else {
        // Hide image, show initial (like frontend)
        if (profileAvatar) profileAvatar.style.display = 'none';
        if (profileAvatarInitial) {
          profileAvatarInitial.textContent = initial;
          profileAvatarInitial.style.display = 'flex';
        }
        
        if (dropdownAvatar) dropdownAvatar.style.display = 'none';
        if (dropdownAvatarInitial) {
          dropdownAvatarInitial.textContent = initial;
          dropdownAvatarInitial.style.display = 'flex';
        }
      }
      
      if (profileName) profileName.textContent = this.user.name || 'User';
      if (profileEmail) profileEmail.textContent = this.user.email || '';
    } else {
      // User is not logged in - show sign in button, hide notification and tabs
      if (signInBtn) signInBtn.classList.remove('hidden');
      if (userProfile) userProfile.classList.add('hidden');
      if (notificationBtn) notificationBtn.classList.add('hidden');
      if (popupTabs) popupTabs.classList.add('hidden');
    }
  }
  
  updatePartyActiveView() {
    document.getElementById('displayRoomCode').textContent = this.roomCode || 'N/A';
    document.getElementById('displayRoomName').textContent = this.roomName || 'Watch Party';
    document.getElementById('shareLink').textContent = `${COWATCH_URL}/join/${this.roomCode}`;
  }
  
  bindEvents() {
    // Platform buttons - open in new tab
    document.querySelectorAll('.platform-btn').forEach(btn => {
      btn.addEventListener('click', () => {
        const url = btn.dataset.url;
        if (url) {
          chrome.tabs.create({ url });
          window.close();
        }
      });
    });
    
    // Google Sign In - check session first, then redirect if needed
    document.getElementById('googleSignIn')?.addEventListener('click', async () => {
      await this.handleLoginClick();
    });
    
    // Profile button - toggle dropdown
    document.getElementById('profileBtn')?.addEventListener('click', (e) => {
      e.stopPropagation();
      const dropdown = document.getElementById('profileDropdown');
      const profileBtn = document.getElementById('profileBtn');
      if (dropdown) dropdown.classList.toggle('hidden');
      if (profileBtn) profileBtn.classList.toggle('active');
    });
    
    // Close dropdown when clicking outside
    document.addEventListener('click', (e) => {
      const dropdown = document.getElementById('profileDropdown');
      const profileBtn = document.getElementById('profileBtn');
      const userProfile = document.getElementById('userProfile');
      if (dropdown && !userProfile?.contains(e.target)) {
        dropdown.classList.add('hidden');
        profileBtn?.classList.remove('active');
      }
    });
    
    // Logout button
    document.getElementById('logoutBtn')?.addEventListener('click', async () => {
      await this.handleLogout();
    });
    
    // Sign Up button
    document.getElementById('signUpBtn')?.addEventListener('click', async () => {
      await this.handleLoginClick();
    });
    
    // Continue as Guest
    document.getElementById('continueAsGuest')?.addEventListener('click', async () => {
      this.guestMode = true;
      await this.saveState({ guestMode: true });
      this.showView('partySettings');
    });
    
    // Settings toggle
    document.getElementById('settingsToggle')?.addEventListener('click', () => {
      const content = document.getElementById('settingsContent');
      const toggle = document.querySelector('.settings-toggle');
      if (content) content.classList.toggle('open');
      if (toggle) toggle.classList.toggle('open');
    });
    
    // Start Party button
    document.getElementById('startPartyBtn')?.addEventListener('click', () => {
      this.startParty();
    });
    
    // Join Existing Party button
    document.getElementById('joinExistingBtn')?.addEventListener('click', () => {
      this.showView('joinRoom');
    });
    
    // Join Party button (in join view)
    document.getElementById('joinPartyBtn')?.addEventListener('click', () => {
      const code = document.getElementById('joinRoomCode').value.trim().toUpperCase();
      if (code && code.length >= 4) {
        this.joinParty(code);
      }
    });
    
    // Back button
    document.getElementById('backToMainBtn')?.addEventListener('click', () => {
      this.showView('partySettings');
    });
    
    // Chat toggle
    document.getElementById('chatToggle')?.addEventListener('change', async (e) => {
      const showChat = e.target.checked;
      if (this.currentTab) {
        try {
          await tabs.sendMessage(this.currentTab.id, {
            type: 'TOGGLE_PANEL',
            show: showChat
          });
        } catch (error) {
          console.error('Failed to toggle chat:', error);
        }
      }
    });
    
    // Copy link button
    document.getElementById('copyLinkBtn')?.addEventListener('click', () => {
      this.copyInviteLink();
    });
    
    // Disconnect button
    document.getElementById('disconnectBtn')?.addEventListener('click', () => {
      this.disconnect();
    });
    
    // Room code input - auto uppercase
    document.getElementById('joinRoomCode')?.addEventListener('input', (e) => {
      e.target.value = e.target.value.toUpperCase();
    });
  }
  
  async startParty() {
    if (!this.currentTab) {
      alert('Please open a video streaming site first.');
      return;
    }
    
    console.log('🎬 Starting party on tab:', this.currentTab.id);
    
    const hostOnlyControl = document.getElementById('hostOnlyControl')?.checked || false;
    const disableReactions = document.getElementById('disableReactions')?.checked || false;
    
    try {
      // First check if content script is already loaded and has video
      let contentScriptLoaded = false;
      let hasVideo = false;
      try {
        const statusResponse = await tabs.sendMessage(this.currentTab.id, { type: 'GET_STATUS' });
        console.log('🎬 Content script already loaded, status:', statusResponse);
        contentScriptLoaded = true;
        hasVideo = statusResponse?.hasVideo || false;
      } catch (e) {
        console.log('🎬 Content script not loaded yet, will inject');
      }
      
      // If not loaded, inject it
      if (!contentScriptLoaded) {
        try {
          if (chrome.scripting) {
            console.log('🎬 Injecting via chrome.scripting API...');
            await chrome.scripting.executeScript({
              target: { tabId: this.currentTab.id },
              files: ['browser-polyfill.js', 'content.js']
            });
            console.log('🎬 Content script injected via scripting API');
            await chrome.scripting.insertCSS({
              target: { tabId: this.currentTab.id },
              files: ['content.css']
            });
            console.log('🎬 CSS injected');
          } else {
            console.log('🎬 chrome.scripting not available, trying background...');
            // Fallback to background script
            const injectResponse = await chrome.runtime.sendMessage({
              type: 'INJECT_CONTENT_SCRIPT',
              tabId: this.currentTab.id
            });
            console.log('🎬 Content script injection response:', injectResponse);
          }
          // Give it a moment to initialize and find video
          await new Promise(resolve => setTimeout(resolve, 1500));
          
          // Check again for video after injection
          try {
            const statusResponse = await tabs.sendMessage(this.currentTab.id, { type: 'GET_STATUS' });
            hasVideo = statusResponse?.hasVideo || false;
          } catch (e) {
            console.log('🎬 Could not check video status');
          }
        } catch (injectError) {
          console.error('🎬 Content script injection failed:', injectError.message);
          alert('Could not inject content script. Please refresh the page and try again.');
          return;
        }
      }
      
      // Check if video is present - retry a few times if not found initially
      if (!hasVideo) {
        console.log('🎬 Video not found on first check, retrying...');
        // Wait and retry up to 3 times
        for (let i = 0; i < 3; i++) {
          await new Promise(resolve => setTimeout(resolve, 1000));
          try {
            const retryStatus = await tabs.sendMessage(this.currentTab.id, { type: 'GET_STATUS' });
            hasVideo = retryStatus?.hasVideo || false;
            console.log(`🎬 Video check retry ${i + 1}:`, hasVideo);
            if (hasVideo) break;
          } catch (e) {
            console.log('🎬 Retry failed');
          }
        }
      }
      
      // Final check - if still no video, show error
      if (!hasVideo) {
        alert('No video found on this page. Please start playing a video first, then create a party.');
        return;
      }
      
      const response = await tabs.sendMessage(this.currentTab.id, {
        type: 'CREATE_ROOM',
        roomName: 'Watch Party',
        hostOnlyControl,
        disableReactions,
        // Pass user info to content script
        user: this.user ? {
          name: this.user.name,
          email: this.user.email,
          image: this.user.image || this.user.avatar
        } : null
      });
      
      console.log('🎬 Create room response:', response);
      
      if (response && response.success) {
        this.roomCode = response.roomCode;
        this.roomName = response.roomName || 'Watch Party';
        await this.saveState({ roomCode: this.roomCode, roomName: this.roomName });
        
        // Show party active view
        this.showView('partyActive');
        this.updatePartyActiveView();
        
        // Close popup after a short delay
        setTimeout(() => window.close(), 500);
      } else {
        console.error('🎬 Failed to create room:', response);
        alert('Could not create party. Please refresh the page and try again.');
      }
    } catch (error) {
      console.error('🎬 Error creating room:', error);
      alert('Could not connect to the page. Please refresh the video page and try again.');
    }
  }
  
  async joinParty(roomCode) {
    if (!this.currentTab) {
      alert('Please open a video streaming site first.');
      return;
    }
    
    console.log('🎬 Joining party:', roomCode);
    
    try {
      const response = await tabs.sendMessage(this.currentTab.id, {
        type: 'JOIN_ROOM',
        roomCode: roomCode,
        // Pass user info to content script
        user: this.user ? {
          name: this.user.name,
          email: this.user.email,
          image: this.user.image || this.user.avatar
        } : null
      });
      
      console.log('🎬 Join room response:', response);
      
      if (response && response.success) {
        this.roomCode = roomCode;
        this.roomName = response.roomName || 'Watch Party';
        await this.saveState({ roomCode: this.roomCode, roomName: this.roomName });
        
        this.showView('partyActive');
        this.updatePartyActiveView();
        
        setTimeout(() => window.close(), 500);
      } else {
        const errorMsg = response?.error || 'Could not join party. Please check the room code and try again.';
        alert(errorMsg);
      }
    } catch (error) {
      console.error('Failed to join room:', error);
      alert('Could not connect to the page. Please refresh and try again.');
    }
  }
  
  async copyInviteLink() {
    const link = `${COWATCH_URL}/join/${this.roomCode}`;
    const btn = document.getElementById('copyLinkBtn');
    
    try {
      await navigator.clipboard.writeText(link);
      btn.classList.add('copied');
      setTimeout(() => btn.classList.remove('copied'), 2000);
    } catch (error) {
      console.error('Failed to copy:', error);
      // Fallback
      const textarea = document.createElement('textarea');
      textarea.value = link;
      document.body.appendChild(textarea);
      textarea.select();
      document.execCommand('copy');
      document.body.removeChild(textarea);
      btn.classList.add('copied');
      setTimeout(() => btn.classList.remove('copied'), 2000);
    }
  }
  
  async disconnect() {
    console.log('🎬 Disconnecting from party');
    
    try {
      if (this.currentTab?.id) {
        // Wrap in try-catch to handle case when content script is not loaded
        try {
          await tabs.sendMessage(this.currentTab.id, { type: 'LEAVE_ROOM' });
        } catch (sendError) {
          // Content script may not be loaded, that's okay
          console.log('Content script not available, continuing disconnect');
        }
      }
    } catch (error) {
      console.error('Error leaving room:', error);
    }
    
    this.roomCode = null;
    this.roomName = null;
    await this.saveState({ roomCode: null, roomName: null });
    
    // Return to appropriate view
    if (this.isOnVideoSite) {
      this.showView('partySettings');
    } else {
      this.showView('platforms');
    }
  }
  
  // ============================================
  // POPUP TABS (Friends/Schedule)
  // ============================================
  
  setupPopupTabs() {
    const tabs = document.querySelectorAll('.popup-tab');
    
    tabs.forEach(tab => {
      tab.addEventListener('click', () => {
        const tabName = tab.dataset.tab;
        this.switchPopupTab(tabName);
      });
    });
    
    // Notification button
    const notificationBtn = document.getElementById('notificationBtn');
    notificationBtn?.addEventListener('click', (e) => {
      e.stopPropagation();
      this.toggleNotificationDropdown();
    });
    
    // Close notification dropdown on outside click
    document.addEventListener('click', (e) => {
      const notificationDropdown = document.getElementById('notificationDropdown');
      const notificationBtn = document.getElementById('notificationBtn');
      if (notificationDropdown && !notificationBtn?.contains(e.target) && !notificationDropdown.contains(e.target)) {
        notificationDropdown.classList.add('hidden');
      }
    });
    
    // Clear notifications button
    document.getElementById('clearNotificationsBtn')?.addEventListener('click', () => {
      this.clearNotifications();
    });
    
    // Add friend button
    document.getElementById('addFriendBtn')?.addEventListener('click', () => {
      this.addFriend();
    });
    
    // Enter key on friend search
    document.getElementById('friendSearchInput')?.addEventListener('keypress', (e) => {
      if (e.key === 'Enter') {
        this.addFriend();
      }
    });
    
    // Create schedule button
    document.getElementById('createScheduleBtn')?.addEventListener('click', () => {
      this.openScheduleModal();
    });
    
    // Schedule modal events
    document.getElementById('closeScheduleModal')?.addEventListener('click', () => {
      this.closeScheduleModal();
    });
    document.getElementById('cancelScheduleBtn')?.addEventListener('click', () => {
      this.closeScheduleModal();
    });
    document.getElementById('saveScheduleBtn')?.addEventListener('click', () => {
      this.createScheduledParty();
    });
    
    // Modal overlay click
    document.getElementById('scheduleModal')?.addEventListener('click', (e) => {
      if (e.target.id === 'scheduleModal') {
        this.closeScheduleModal();
      }
    });
  }
  
  switchPopupTab(tabName) {
    this.activePopupTab = tabName;
    
    // Update tab buttons
    document.querySelectorAll('.popup-tab').forEach(tab => {
      tab.classList.toggle('active', tab.dataset.tab === tabName);
    });
    
    // Show/hide content based on tab
    const viewPlatforms = document.getElementById('viewPlatforms');
    const friendsContent = document.getElementById('friendsContent');
    const scheduleContent = document.getElementById('scheduleContent');
    const lastRoomsContent = document.getElementById('lastRoomsContent');
    
    // Hide all tab content
    if (friendsContent) friendsContent.classList.remove('active');
    if (scheduleContent) scheduleContent.classList.remove('active');
    if (lastRoomsContent) lastRoomsContent.classList.remove('active');
    
    if (tabName === 'platforms') {
      if (viewPlatforms) viewPlatforms.classList.add('active');
    } else if (tabName === 'lastRooms') {
      if (viewPlatforms) viewPlatforms.classList.remove('active');
      if (lastRoomsContent) lastRoomsContent.classList.add('active');
      this.renderLastRooms(); // Load rooms when tab is clicked
    } else if (tabName === 'friends') {
      if (viewPlatforms) viewPlatforms.classList.remove('active');
      if (friendsContent) friendsContent.classList.add('active');
    } else if (tabName === 'schedule') {
      if (viewPlatforms) viewPlatforms.classList.remove('active');
      if (scheduleContent) scheduleContent.classList.add('active');
    }
  }
  
  toggleNotificationDropdown() {
    const dropdown = document.getElementById('notificationDropdown');
    if (dropdown) {
      dropdown.classList.toggle('hidden');
    }
  }
  
  // ============================================
  // NOTIFICATIONS
  // ============================================
  
  async loadNotifications() {
    if (!this.user) return;
    
    try {
      // Build notifications from pending friend requests
      this.notifications = [];
      
      // Fetch friend requests
      const response = await fetch(`${COWATCH_URL}/api/friends`, {
        credentials: 'include'
      });
      
      if (response.ok) {
        const data = await response.json();
        this.pendingRequests = data.pendingRequests || [];
        
        // Convert pending requests to notifications
        this.pendingRequests.forEach(req => {
          this.notifications.push({
            id: req.id,
            type: 'friend_request',
            from: req.from,
            message: req.message,
            createdAt: req.createdAt,
            read: false
          });
        });
      }
      
      // Fetch scheduled parties to check for upcoming
      const scheduleRes = await fetch(`${COWATCH_URL}/api/schedule`, {
        credentials: 'include'
      });
      
      if (scheduleRes.ok) {
        const scheduleData = await scheduleRes.json();
        this.scheduledParties = scheduleData.parties || [];
        
        // Add notifications for parties starting soon (within 1 hour)
        const now = new Date();
        const oneHour = 60 * 60 * 1000;
        
        this.scheduledParties.forEach(party => {
          const partyTime = new Date(party.scheduledAt);
          const diff = partyTime - now;
          
          if (diff > 0 && diff <= oneHour) {
            this.notifications.push({
              id: 'party_' + party.id,
              type: 'party_soon',
              party: party,
              createdAt: new Date().toISOString(),
              read: false
            });
          }
        });
      }
      
      this.updateNotificationBadge();
      this.renderNotifications();
    } catch (error) {
      console.error('Failed to load notifications:', error);
    }
  }
  
  updateNotificationBadge() {
    const badge = document.getElementById('notificationBadge');
    const friendsBadge = document.getElementById('friendsTabBadge');
    const unreadCount = this.notifications.filter(n => !n.read).length;
    
    if (badge) {
      if (unreadCount > 0) {
        badge.textContent = unreadCount > 9 ? '9+' : unreadCount;
        badge.classList.remove('hidden');
      } else {
        badge.classList.add('hidden');
      }
    }
    
    // Update friends tab badge with pending requests count
    if (friendsBadge) {
      const requestCount = this.pendingRequests.length;
      if (requestCount > 0) {
        friendsBadge.textContent = requestCount;
        friendsBadge.classList.remove('hidden');
      } else {
        friendsBadge.classList.add('hidden');
      }
    }
  }
  
  renderNotifications() {
    const list = document.getElementById('notificationList');
    const empty = document.getElementById('notificationEmpty');
    
    if (!list) return;
    
    if (this.notifications.length === 0) {
      if (empty) empty.style.display = 'flex';
      return;
    }
    
    if (empty) empty.style.display = 'none';
    
    // Clear existing items except empty state
    list.querySelectorAll('.notification-item').forEach(el => el.remove());
    
    this.notifications.forEach(notif => {
      const item = document.createElement('div');
      item.className = `notification-item ${notif.read ? '' : 'unread'}`;
      
      if (notif.type === 'friend_request') {
        const initial = (notif.from.username || 'U').charAt(0).toUpperCase();
        item.innerHTML = `
          <div class="notification-avatar">
            ${notif.from.avatar ? `<img src="${notif.from.avatar}" alt="">` : initial}
          </div>
          <div class="notification-content">
            <div class="notification-text">
              <strong>${this.escapeHtml(notif.from.username)}</strong> ${this.getTranslation('sentFriendRequest', 'sent you a friend request')}
            </div>
            <div class="notification-time">${this.formatTimeAgo(notif.createdAt)}</div>
            <div class="notification-actions">
              <button class="notification-action-btn accept" data-id="${notif.id}" data-action="accept">
                ${this.getTranslation('accept', 'Accept')}
              </button>
              <button class="notification-action-btn decline" data-id="${notif.id}" data-action="decline">
                ${this.getTranslation('decline', 'Decline')}
              </button>
            </div>
          </div>
        `;
        
        // Add event listeners
        item.querySelector('.accept')?.addEventListener('click', () => this.handleFriendRequest(notif.id, 'accept'));
        item.querySelector('.decline')?.addEventListener('click', () => this.handleFriendRequest(notif.id, 'decline'));
      } else if (notif.type === 'party_soon') {
        item.innerHTML = `
          <div class="notification-avatar">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="width:20px;height:20px;color:#ff3b30">
              <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
              <line x1="16" y1="2" x2="16" y2="6"/>
              <line x1="8" y1="2" x2="8" y2="6"/>
              <line x1="3" y1="10" x2="21" y2="10"/>
            </svg>
          </div>
          <div class="notification-content">
            <div class="notification-text">
              <strong>${this.escapeHtml(notif.party.title)}</strong> ${this.getTranslation('startingSoon', 'is starting soon!')}
            </div>
            <div class="notification-time">${this.formatPartyTime(notif.party.scheduledAt)}</div>
          </div>
        `;
      }
      
      list.appendChild(item);
    });
  }
  
  async handleFriendRequest(requestId, action) {
    try {
      const url = `${COWATCH_URL}/api/friends/${requestId}/accept`;
      const method = action === 'accept' ? 'POST' : 'DELETE';
      
      const response = await fetch(url, {
        method,
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json'
        }
      });
      
      if (response.ok) {
        this.showToast(
          action === 'accept' 
            ? this.getTranslation('friendRequestAccepted', 'Friend request accepted!')
            : this.getTranslation('friendRequestDeclined', 'Friend request declined'),
          'success'
        );
        
        // Refresh notifications and friends
        await this.loadNotifications();
        await this.loadFriends();
      } else {
        const data = await response.json();
        this.showToast(data.error || 'Failed to process request', 'error');
      }
    } catch (error) {
      console.error('Error handling friend request:', error);
      this.showToast('Network error', 'error');
    }
  }
  
  clearNotifications() {
    this.notifications = this.notifications.map(n => ({ ...n, read: true }));
    this.updateNotificationBadge();
    this.renderNotifications();
  }
  
  // ============================================
  // FRIENDS
  // ============================================
  
  async loadFriends() {
    if (!this.user) return;
    
    try {
      const response = await fetch(`${COWATCH_URL}/api/friends`, {
        credentials: 'include'
      });
      
      if (response.ok) {
        const data = await response.json();
        this.friends = data.friends || [];
        this.pendingRequests = data.pendingRequests || [];
        this.renderFriends();
      }
    } catch (error) {
      console.error('Failed to load friends:', error);
    }
  }
  
  renderFriends() {
    const list = document.getElementById('friendsList');
    const empty = document.getElementById('friendsEmpty');
    
    if (!list) return;
    
    // Clear existing items
    list.querySelectorAll('.friend-item, .friend-request-item, .pending-requests-section').forEach(el => el.remove());
    
    const hasFriends = this.friends.length > 0;
    const hasPending = this.pendingRequests && this.pendingRequests.length > 0;
    
    if (!hasFriends && !hasPending) {
      if (empty) empty.style.display = 'flex';
      return;
    }
    
    if (empty) empty.style.display = 'none';
    
    // Render pending friend requests first
    if (hasPending) {
      const pendingSection = document.createElement('div');
      pendingSection.className = 'pending-requests-section';
      pendingSection.innerHTML = `
        <div class="pending-requests-header">
          <span class="pending-requests-title">${this.getTranslation('pendingRequests', 'Pending Requests')}</span>
          <span class="pending-requests-count">${this.pendingRequests.length}</span>
        </div>
      `;
      
      this.pendingRequests.forEach(request => {
        const item = document.createElement('div');
        item.className = 'friend-request-item';
        
        const initial = (request.from?.username || 'U').charAt(0).toUpperCase();
        
        item.innerHTML = `
          <div class="friend-avatar request-avatar">
            ${request.from?.avatar ? `<img src="${request.from.avatar}" alt="">` : initial}
          </div>
          <div class="friend-info">
            <div class="friend-name">${this.escapeHtml(request.from?.username || 'Unknown')}</div>
            <div class="friend-status request-status">${this.getTranslation('wantsToBeYourFriend', 'Wants to be your friend')}</div>
          </div>
          <div class="request-actions">
            <button class="request-accept-btn" data-id="${request.id}" title="${this.getTranslation('accept', 'Accept')}">
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                <polyline points="20 6 9 17 4 12"></polyline>
              </svg>
            </button>
            <button class="request-reject-btn" data-id="${request.id}" title="${this.getTranslation('reject', 'Reject')}">
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5">
                <line x1="18" y1="6" x2="6" y2="18"></line>
                <line x1="6" y1="6" x2="18" y2="18"></line>
              </svg>
            </button>
          </div>
        `;
        
        // Add event listeners
        item.querySelector('.request-accept-btn')?.addEventListener('click', () => {
          this.acceptFriendRequest(request.id);
        });
        
        item.querySelector('.request-reject-btn')?.addEventListener('click', () => {
          this.rejectFriendRequest(request.id);
        });
        
        pendingSection.appendChild(item);
      });
      
      list.appendChild(pendingSection);
    }
    
    // Render existing friends
    this.friends.forEach(friend => {
      const item = document.createElement('div');
      item.className = 'friend-item';
      
      const initial = (friend.username || friend.nickname || 'U').charAt(0).toUpperCase();
      
      item.innerHTML = `
        <div class="friend-avatar">
          ${friend.avatar ? `<img src="${friend.avatar}" alt="">` : initial}
        </div>
        <div class="friend-info">
          <div class="friend-name">${this.escapeHtml(friend.nickname || friend.username)}</div>
          <div class="friend-status">${this.getTranslation('offline', 'Offline')}</div>
        </div>
        <button class="friend-invite-btn" data-id="${friend.id}">
          ${this.getTranslation('invite', 'Invite')}
        </button>
      `;
      
      // Add invite event listener
      item.querySelector('.friend-invite-btn')?.addEventListener('click', () => {
        this.inviteFriend(friend);
      });
      
      list.appendChild(item);
    });
  }
  
  async rejectFriendRequest(requestId) {
    try {
      const response = await fetch(`${COWATCH_URL}/api/friends/${requestId}`, {
        method: 'DELETE',
        credentials: 'include'
      });
      
      if (response.ok) {
        this.showToast(this.getTranslation('requestRejected', 'Request rejected'), 'info');
        await this.loadFriends();
        await this.loadNotifications();
      } else {
        const data = await response.json();
        this.showToast(data.error || 'Failed to reject request', 'error');
      }
    } catch (error) {
      console.error('Error rejecting friend request:', error);
      this.showToast('Network error', 'error');
    }
  }
  
  async addFriend() {
    const input = document.getElementById('friendSearchInput');
    const value = input?.value.trim();
    
    if (!value) {
      this.showToast(this.getTranslation('enterUsernameOrEmail', 'Enter username or email'), 'error');
      return;
    }
    
    try {
      const isEmail = value.includes('@');
      const body = isEmail ? { email: value } : { username: value };
      
      const response = await fetch(`${COWATCH_URL}/api/friends`, {
        method: 'POST',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(body)
      });
      
      const data = await response.json();
      
      if (response.ok) {
        if (data.accepted) {
          this.showToast(this.getTranslation('friendAdded', 'Friend added!'), 'success');
          await this.loadFriends();
        } else {
          this.showToast(this.getTranslation('friendRequestSent', 'Friend request sent!'), 'success');
        }
        if (input) input.value = '';
      } else {
        this.showToast(data.error || 'Failed to send request', 'error');
      }
    } catch (error) {
      console.error('Error adding friend:', error);
      this.showToast('Network error', 'error');
    }
  }
  
  inviteFriend(friend) {
    if (!this.roomCode) {
      this.showToast(this.getTranslation('startPartyFirst', 'Start a party first to invite friends'), 'info');
      return;
    }
    
    // Copy invite link (could also send notification)
    const link = `${COWATCH_URL}/join/${this.roomCode}`;
    navigator.clipboard.writeText(link).then(() => {
      this.showToast(this.getTranslation('inviteLinkCopied', 'Invite link copied!'), 'success');
    });
  }
  
  // ============================================
  // SCHEDULED PARTIES
  // ============================================
  
  async loadScheduledParties() {
    if (!this.user) return;
    
    try {
      const response = await fetch(`${COWATCH_URL}/api/schedule`, {
        credentials: 'include'
      });
      
      if (response.ok) {
        const data = await response.json();
        this.scheduledParties = data.parties || [];
        this.renderScheduledParties();
      }
    } catch (error) {
      console.error('Failed to load scheduled parties:', error);
    }
  }
  
  renderScheduledParties() {
    const list = document.getElementById('scheduleList');
    const empty = document.getElementById('scheduleEmpty');
    
    if (!list) return;
    
    if (this.scheduledParties.length === 0) {
      if (empty) empty.style.display = 'flex';
      return;
    }
    
    if (empty) empty.style.display = 'none';
    
    // Clear existing items except empty state
    list.querySelectorAll('.schedule-item').forEach(el => el.remove());
    
    this.scheduledParties.forEach(party => {
      const item = document.createElement('div');
      item.className = 'schedule-item';
      
      const partyTime = new Date(party.scheduledAt);
      const now = new Date();
      const diff = partyTime - now;
      const isLive = party.status === 'started';
      const isSoon = diff > 0 && diff <= 60 * 60 * 1000;
      
      let timeBadgeClass = '';
      let timeBadgeText = '';
      
      if (isLive) {
        timeBadgeClass = 'live';
        timeBadgeText = 'LIVE';
      } else if (isSoon) {
        timeBadgeClass = 'soon';
        timeBadgeText = this.formatTimeUntil(party.scheduledAt);
      } else {
        timeBadgeText = this.formatPartyTime(party.scheduledAt);
      }
      
      item.innerHTML = `
        <div class="schedule-icon">
          <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polygon points="5 3 19 12 5 21 5 3"/>
          </svg>
        </div>
        <div class="schedule-info">
          <div class="schedule-title">${this.escapeHtml(party.title)}</div>
          <div class="schedule-meta">
            <span>
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <rect x="2" y="3" width="20" height="14" rx="2" ry="2"/>
              </svg>
              ${party.platform}
            </span>
            <span>
              <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                <circle cx="9" cy="7" r="4"/>
              </svg>
              ${party.invitedCount} ${this.getTranslation('invited', 'invited')}
            </span>
          </div>
        </div>
        <div class="schedule-actions">
          <div class="schedule-time-badge ${timeBadgeClass}">${timeBadgeText}</div>
          ${isLive && party.roomCode ? `
            <button class="schedule-join-btn" data-code="${party.roomCode}">
              ${this.getTranslation('join', 'Join')}
            </button>
          ` : ''}
        </div>
      `;
      
      // Add join event listener
      const joinBtn = item.querySelector('.schedule-join-btn');
      if (joinBtn) {
        joinBtn.addEventListener('click', () => {
          this.joinScheduledParty(party);
        });
      }
      
      list.appendChild(item);
    });
  }
  
  openScheduleModal() {
    const modal = document.getElementById('scheduleModal');
    const invitesSelect = document.getElementById('scheduleInvites');
    
    // Set default date/time
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    tomorrow.setHours(20, 0, 0, 0);
    
    document.getElementById('scheduleDate').value = tomorrow.toISOString().split('T')[0];
    document.getElementById('scheduleTime').value = '20:00';
    
    // Populate friends list
    if (invitesSelect) {
      invitesSelect.innerHTML = this.friends.map(f => 
        `<option value="${f.id}">${this.escapeHtml(f.username)}</option>`
      ).join('');
    }
    
    if (modal) modal.classList.remove('hidden');
  }
  
  closeScheduleModal() {
    const modal = document.getElementById('scheduleModal');
    if (modal) modal.classList.add('hidden');
    
    // Clear form
    document.getElementById('scheduleTitle').value = '';
    document.getElementById('scheduleContentUrl').value = '';
  }
  
  async createScheduledParty() {
    const title = document.getElementById('scheduleTitle')?.value.trim();
    const platform = document.getElementById('schedulePlatform')?.value;
    const contentUrl = document.getElementById('scheduleContentUrl')?.value.trim();
    const date = document.getElementById('scheduleDate')?.value;
    const time = document.getElementById('scheduleTime')?.value;
    const invitesSelect = document.getElementById('scheduleInvites');
    
    if (!title) {
      this.showToast(this.getTranslation('enterPartyTitle', 'Enter a party title'), 'error');
      return;
    }
    
    if (!date || !time) {
      this.showToast(this.getTranslation('selectDateTime', 'Select date and time'), 'error');
      return;
    }
    
    const scheduledAt = new Date(`${date}T${time}`);
    if (scheduledAt <= new Date()) {
      this.showToast(this.getTranslation('selectFutureTime', 'Select a future time'), 'error');
      return;
    }
    
    // Get selected friend IDs
    const invitedUserIds = invitesSelect ? Array.from(invitesSelect.selectedOptions).map(opt => opt.value) : [];
    
    try {
      const response = await fetch(`${COWATCH_URL}/api/schedule`, {
        method: 'POST',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          title,
          platform,
          contentUrl,
          scheduledAt: scheduledAt.toISOString(),
          invitedUserIds,
          timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
        })
      });
      
      const data = await response.json();
      
      if (response.ok) {
        this.showToast(this.getTranslation('partyScheduled', 'Party scheduled!'), 'success');
        this.closeScheduleModal();
        await this.loadScheduledParties();
      } else {
        this.showToast(data.error || 'Failed to create party', 'error');
      }
    } catch (error) {
      console.error('Error creating scheduled party:', error);
      this.showToast('Network error', 'error');
    }
  }
  
  joinScheduledParty(party) {
    if (party.roomCode) {
      // Open the join link
      chrome.tabs.create({ url: `${COWATCH_URL}/join/${party.roomCode}` });
      window.close();
    }
  }
  
  // ============================================
  // UTILITY FUNCTIONS
  // ============================================
  
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
  
  formatTimeAgo(dateStr) {
    const date = new Date(dateStr);
    const now = new Date();
    const diff = Math.floor((now - date) / 1000);
    
    if (diff < 60) return this.getTranslation('justNow', 'just now');
    if (diff < 3600) return `${Math.floor(diff / 60)}m ${this.getTranslation('ago', 'ago')}`;
    if (diff < 86400) return `${Math.floor(diff / 3600)}h ${this.getTranslation('ago', 'ago')}`;
    return `${Math.floor(diff / 86400)}d ${this.getTranslation('ago', 'ago')}`;
  }
  
  formatPartyTime(dateStr) {
    const date = new Date(dateStr);
    const options = { 
      month: 'short', 
      day: 'numeric', 
      hour: '2-digit', 
      minute: '2-digit' 
    };
    return date.toLocaleDateString(undefined, options);
  }
  
  formatTimeUntil(dateStr) {
    const date = new Date(dateStr);
    const now = new Date();
    const diff = Math.floor((date - now) / 1000 / 60);
    
    if (diff <= 0) return 'Now';
    if (diff < 60) return `${diff}m`;
    return `${Math.floor(diff / 60)}h ${diff % 60}m`;
  }
  
  getTranslation(key, fallback) {
    // Try to get translation from i18n system
    if (typeof translations !== 'undefined' && typeof currentLanguage !== 'undefined') {
      const langData = translations[currentLanguage] || translations['en'];
      if (langData && langData[key]) return langData[key];
    }
    return fallback;
  }
  
  // ============================================
  // LAST ROOMS FEATURE (Room History)
  // ============================================
  
  async loadLastRooms() {
    try {
      const data = await storage.get(['lastRooms']);
      this.lastRooms = data.lastRooms || [];
      console.log('🎬 Loaded last rooms:', this.lastRooms.length);
      return this.lastRooms;
    } catch (error) {
      console.error('🎬 Error loading last rooms:', error);
      this.lastRooms = [];
      return [];
    }
  }
  
  async saveLastRoom(roomData) {
    try {
      await this.loadLastRooms();
      
      // Room data structure
      const room = {
        code: roomData.code,
        name: roomData.name || `Room ${roomData.code}`,
        platform: roomData.platform,
        videoTitle: roomData.videoTitle || '',
        videoUrl: roomData.videoUrl || '',
        thumbnail: roomData.thumbnail || '',
        joinedAt: Date.now(),
        lastActive: Date.now(),
        isHost: roomData.isHost || false,
        participantCount: roomData.participantCount || 1
      };
      
      // Remove if already exists (we'll add it to top)
      this.lastRooms = this.lastRooms.filter(r => r.code !== room.code);
      
      // Add to top
      this.lastRooms.unshift(room);
      
      // Keep only last 10 rooms
      this.lastRooms = this.lastRooms.slice(0, 10);
      
      await storage.set({ lastRooms: this.lastRooms });
      console.log('🎬 Saved room to history:', room.code);
    } catch (error) {
      console.error('🎬 Error saving last room:', error);
    }
  }
  
  async updateLastRoomActivity(roomCode, updates = {}) {
    try {
      await this.loadLastRooms();
      
      const room = this.lastRooms.find(r => r.code === roomCode);
      if (room) {
        room.lastActive = Date.now();
        Object.assign(room, updates);
        await storage.set({ lastRooms: this.lastRooms });
      }
    } catch (error) {
      console.error('🎬 Error updating room activity:', error);
    }
  }
  
  async checkRoomStatus(roomCode) {
    try {
      // Check if room is still active via server
      const response = await fetch(`${COWATCH_URL}/api/rooms/check?code=${roomCode}`);
      const data = await response.json();
      return {
        isActive: data.isActive,
        participantCount: data.participantCount || 0
      };
    } catch (error) {
      console.log('🎬 Could not check room status:', roomCode);
      return { isActive: false, participantCount: 0 };
    }
  }
  
  async removeLastRoom(roomCode) {
    try {
      await this.loadLastRooms();
      this.lastRooms = this.lastRooms.filter(r => r.code !== roomCode);
      await storage.set({ lastRooms: this.lastRooms });
      this.renderLastRooms();
    } catch (error) {
      console.error('🎬 Error removing room from history:', error);
    }
  }
  
  async clearLastRooms() {
    try {
      this.lastRooms = [];
      await storage.set({ lastRooms: [] });
      this.renderLastRooms();
      this.showToast(this.getTranslation('historyCleared', 'History cleared'), 'success');
    } catch (error) {
      console.error('🎬 Error clearing room history:', error);
    }
  }
  
  async renderLastRooms() {
    const container = document.getElementById('lastRoomsContent');
    if (!container) return;
    
    await this.loadLastRooms();
    
    if (this.lastRooms.length === 0) {
      container.innerHTML = `
        <div class="empty-state">
          <div class="empty-icon">🎬</div>
          <p>${this.getTranslation('noRecentRooms', 'No recent watch parties')}</p>
          <span class="empty-subtitle">${this.getTranslation('roomsWillAppearHere', 'Your watch parties will appear here')}</span>
        </div>
      `;
      return;
    }
    
    // Check status for each room (in parallel)
    const statusChecks = this.lastRooms.map(async room => {
      const status = await this.checkRoomStatus(room.code);
      return { ...room, ...status };
    });
    
    const roomsWithStatus = await Promise.all(statusChecks);
    
    container.innerHTML = `
      <div class="last-rooms-header">
        <h4>${this.getTranslation('recentRooms', 'Recent Rooms')}</h4>
        <button class="clear-history-btn" id="clearHistoryBtn">
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="3 6 5 6 21 6"/>
            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
          </svg>
          ${this.getTranslation('clear', 'Clear')}
        </button>
      </div>
      <div class="last-rooms-list">
        ${roomsWithStatus.map(room => this.renderLastRoomItem(room)).join('')}
      </div>
    `;
    
    // Bind events
    document.getElementById('clearHistoryBtn')?.addEventListener('click', () => this.clearLastRooms());
    
    container.querySelectorAll('.last-room-item').forEach(item => {
      const roomCode = item.dataset.roomCode;
      
      item.querySelector('.join-room-btn')?.addEventListener('click', (e) => {
        e.stopPropagation();
        this.joinLastRoom(roomCode);
      });
      
      item.querySelector('.remove-room-btn')?.addEventListener('click', (e) => {
        e.stopPropagation();
        this.removeLastRoom(roomCode);
      });
    });
  }
  
  renderLastRoomItem(room) {
    const platformInfo = this.getPlatformInfo(room.platform);
    const timeAgo = this.formatTimeAgo(room.lastActive || room.joinedAt);
    
    return `
      <div class="last-room-item ${room.isActive ? 'active' : 'inactive'}" data-room-code="${room.code}">
        <div class="room-thumbnail">
          ${room.thumbnail 
            ? `<img src="${room.thumbnail}" alt="${this.escapeHtml(room.videoTitle || room.name)}" onerror="this.style.display='none'">`
            : `<div class="room-thumbnail-placeholder" style="background: ${platformInfo?.color || '#333'}">
                <span>${platformInfo?.name?.charAt(0) || '🎬'}</span>
              </div>`
          }
          ${room.isActive ? `<span class="live-badge">LIVE</span>` : ''}
        </div>
        <div class="room-info">
          <div class="room-title">${this.escapeHtml(room.videoTitle || room.name)}</div>
          <div class="room-meta">
            <span class="room-platform">${platformInfo?.name || 'Video'}</span>
            <span class="room-separator">•</span>
            <span class="room-code">${room.code}</span>
            <span class="room-separator">•</span>
            <span class="room-time">${timeAgo}</span>
          </div>
          ${room.isActive 
            ? `<div class="room-participants">
                <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                  <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                  <circle cx="9" cy="7" r="4"/>
                  <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                  <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                </svg>
                ${room.participantCount} ${this.getTranslation('watching', 'watching')}
              </div>` 
            : ''
          }
        </div>
        <div class="room-actions">
          ${room.isActive 
            ? `<button class="join-room-btn">
                ${this.getTranslation('join', 'Join')}
              </button>`
            : `<span class="room-ended">${this.getTranslation('ended', 'Ended')}</span>`
          }
          <button class="remove-room-btn" title="${this.getTranslation('remove', 'Remove')}">
            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
              <line x1="18" y1="6" x2="6" y2="18"/>
              <line x1="6" y1="6" x2="18" y2="18"/>
            </svg>
          </button>
        </div>
      </div>
    `;
  }
  
  getPlatformInfo(platform) {
    // Try to find platform info from PLATFORM_INFO
    for (const [domain, info] of Object.entries(PLATFORM_INFO)) {
      if (domain.includes(platform) || info.name.toLowerCase() === platform?.toLowerCase()) {
        return info;
      }
    }
    return { name: platform || 'Video', color: '#333' };
  }
  
  async joinLastRoom(roomCode) {
    try {
      // Check if room is active
      const status = await this.checkRoomStatus(roomCode);
      
      if (!status.isActive) {
        this.showToast(this.getTranslation('roomEnded', 'This room has ended'), 'error');
        return;
      }
      
      // Find the room to get its video URL
      const room = this.lastRooms.find(r => r.code === roomCode);
      
      if (room?.videoUrl) {
        // Open the video URL with room code
        chrome.tabs.create({ url: `${room.videoUrl}#cowatch=${roomCode}` });
      } else {
        // Open via cowatch.app join page
        chrome.tabs.create({ url: `${COWATCH_URL}/join/${roomCode}` });
      }
      
      window.close();
    } catch (error) {
      console.error('🎬 Error joining room:', error);
      this.showToast(this.getTranslation('joinError', 'Could not join room'), 'error');
    }
  }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  new CoWatchPopup();
});
